/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <GL/gl.h>
#include "object.h"

Object         *
object_create(char *name)
{
	Object         *object;	/* New object. */
	FILE           *file;	/* Input file. */
	int             i;	/* Loop index. */

	/* Create new object. */
	object = (Object *) malloc(sizeof(Object));
	assert(object != NULL);

	/* Open input file. */
	file = fopen(name, "r");
	assert(file != NULL);

	/* Read boundary box. */
	assert(fscanf(file, "%f%f%f%f%f%f", &object->lo[X], &object->lo[Y], &object->lo[Z], &object->hi[X], &object->hi[Y], &object->hi[Z]) == 6);

	/* Read vertex count. */
	assert(fscanf(file, "%d", &object->vertex_count) == 1);

	/* Allocate memory for and read object vertices. */
	object->vertices = (Vertex *) malloc(object->vertex_count * sizeof(Vertex));
	assert(object->vertices != NULL);
	for (i = 0; i < object->vertex_count; i++)
		assert(fscanf(file, "%f%f%f%f%f%f", &object->vertices[i].point[X], &object->vertices[i].point[Y], &object->vertices[i].point[Z], &object->vertices[i].color[R], &object->vertices[i].color[G], &object->vertices[i].color[B]) == 6);

	/* Read normal count. */
	assert(fscanf(file, "%d", &object->normal_count) == 1);

	/* Allocate memory for and read object normals. */
	object->normals = (Vector *) malloc(object->normal_count * sizeof(Vector));
	assert(object->normals != NULL);
	for (i = 0; i < object->normal_count; i++)
		assert(fscanf(file, "%f%f%f", &object->normals[i][X], &object->normals[i][Y], &object->normals[i][Z]) == 3);

	/* Read material count. */
	assert(fscanf(file, "%d", &object->material_count) == 1);

	/* Allocate memory for and read object materials. */
	object->materials = (Material *) malloc(object->material_count * sizeof(Material));
	assert(object->materials != NULL);
	for (i = 0; i < object->material_count; i++)
		assert(fscanf(file, "%f%f%f%f%f%f%f", &object->materials[i].ambient[R], &object->materials[i].ambient[G], &object->materials[i].ambient[B], &object->materials[i].specular[R], &object->materials[i].specular[G], &object->materials[i].specular[B], &object->materials[i].shininess) == 7);

	/* Read triangle count. */
	assert(fscanf(file, "%d", &object->triangle_count) == 1);

	/* Allocate memory for and read object triangles. */
	object->triangles = (Triangle *) malloc(object->triangle_count * sizeof(Triangle));
	assert(object->triangles != NULL);
	for (i = 0; i < object->triangle_count; i++)
		assert(fscanf(file, "%d%d%d%d%d%d%d", &object->triangles[i].v0, &object->triangles[i].v1, &object->triangles[i].v2, &object->triangles[i].n0, &object->triangles[i].n1, &object->triangles[i].n2, &object->triangles[i].m) == 7);

	/* Close input file. */
	fclose(file);

	return object;
}

Object         *
object_copy(Object * other)
{
	Object         *object;	/* New object. */

	/* Create new object. */
	object = (Object *) malloc(sizeof(Object));
	assert(object != NULL);

	/* Copy lower and upper corner of bounding box. */
	vector_copy(object->lo, other->lo);
	vector_copy(object->hi, other->hi);

	/* Copy vertices. */
	object->vertex_count = other->vertex_count;
	object->vertices = (Vertex *) malloc(object->vertex_count * sizeof(Vertex));
	assert(object->vertices != NULL);
	memcpy(object->vertices, other->vertices, object->vertex_count * sizeof(Vertex));

	/* Copy normals. */
	object->normal_count = other->normal_count;
	object->normals = (Vector *) malloc(object->normal_count * sizeof(Vector));
	assert(object->normals != NULL);
	memcpy(object->normals, other->normals, object->normal_count * sizeof(Vector));

	/* Copy materials. */
	object->material_count = other->material_count;
	object->materials = (Material *) malloc(object->material_count * sizeof(Material));
	assert(object->materials != NULL);
	memcpy(object->materials, other->materials, object->material_count * sizeof(Material));

	/* Copy triangles. */
	object->triangle_count = other->triangle_count;
	object->triangles = (Triangle *) malloc(object->triangle_count * sizeof(Triangle));
	assert(object->triangles != NULL);
	memcpy(object->triangles, other->triangles, object->triangle_count * sizeof(Triangle));

	return object;
}

void
object_destroy(Object * object)
{
	/*
	 * Free memory used for object vertices, normals, materials and
	 * triangles, as well as for object itself.
	 */
	free(object->vertices);
	free(object->normals);
	free(object->materials);
	free(object->triangles);
	free(object);
}

void
object_render(Object * object)
{
	int             i;	/* Loop index. */

	/*
	 * Render object triangles using OpenGL commands. Sequence of
	 * commands below is intentionaly left as is, without any OpenGL
	 * optimizations, for simplicity.
	 */
	glBegin(GL_TRIANGLES);
	for (i = 0; i < object->triangle_count; i++) {
		/* Apply appropriate material parameters. */
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, object->materials[object->triangles[i].m].ambient);
		glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, object->materials[object->triangles[i].m].specular);
		glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, object->materials[object->triangles[i].m].shininess);

		/* Draw first vertex. */
		glColor3fv(object->vertices[object->triangles[i].v0].color);
		glNormal3fv(object->normals[object->triangles[i].n0]);
		glVertex3fv(object->vertices[object->triangles[i].v0].point);

		/* Draw second vertex. */
		glColor3fv(object->vertices[object->triangles[i].v1].color);
		glNormal3fv(object->normals[object->triangles[i].n1]);
		glVertex3fv(object->vertices[object->triangles[i].v1].point);

		/* Draw third vertex. */
		glColor3fv(object->vertices[object->triangles[i].v2].color);
		glNormal3fv(object->normals[object->triangles[i].n2]);
		glVertex3fv(object->vertices[object->triangles[i].v2].point);
	}
	glEnd();
}
