# Kubernetes and SSH Integration Guide

Teleport v.3.0+ has the
ability to act as a compliance gateway for managing privileged access to Kubernetes clusters. This enables the following capabilities:

* A Teleport Proxy can act as a single authentication endpoint for both SSH and
  Kubernetes. Users can authenticate against a Teleport proxy using Teleport's `tsh login` command and retrieve credentials for both SSH and Kubernetes API.
* Users RBAC roles are always synchronized between SSH and Kubernetes, making
  it easier to implement policies like _developers must not access production
  data_.
* Teleport's session recording and audit log extend to Kubernetes, as well.
  Regular `kubectl exec` commands are logged into the audit log and the
  interactive commands are recorded as regular sessions that can be stored and replayed in the
  future.

![ssh-kubernetes-integration](img/teleport-kube.png)

This guide will walk you through the steps required to configure Teleport to
work as a unified gateway for both SSH and Kubernetes. We will cover both the open
source and enterprise editions of Teleport.

For this guide, we'll be using an instance of Kubernetes running on [Google's GKE](https://cloud.google.com/kubernetes-engine/)
but this guide should apply with any upstream Kubernetes instance. 

## Teleport Proxy Service

By default, the Kubernetes integration is turned off in Teleport. The configuration setting to enable the integration is the `proxy_service/kubernetes/enabled` setting which can be found in the proxy service section in the `/etc/teleport.yaml` file, as shown below:

```bash
# snippet from /etc/teleport.yaml on the Teleport proxy service:
proxy_service:
    # create the 'kubernetes' section and set 'enabled' to 'yes':
    kubernetes:
        enabled: yes
        public_addr: [teleport.example.com:3026]
        listen_addr: 0.0.0.0:3026
```

Let's take a closer look at the available Kubernetes settings:

* `public_addr` defines the publicly accessible address which Kubernetes API
  clients like `kubectl` will connect to. This address will be placed inside of
  `kubeconfig` on a client's machine when a client executes `tsh login` command
  to retrieve its certificate. If you intend to run multiple Teleport proxies behind
  a load balancer, this must be the load balancer's public address.

* `listen_addr` defines which network interface and port the Teleport proxy server
  should bind to. It defaults to port 3026 on all NICs.

Notice that the proxy configuration does not contain any Kubernetes credentials.
This is because a Teleport proxy is not supposed to keep any secrets, as it uses
the Teleport Auth service to retrieve Kubernetes credentials.

## Teleport Auth Service

The next step is to configure Teleport Auth to be able to request Kubernetes TLS
certificates using the [Kubernetes CSR API](https://kubernetes.io/docs/tasks/tls/managing-tls-in-a-cluster/).

There are two ways this can be done:

1. Deploy Teleport Auth service as a Kubernetes pod inside the Kubernetes
   cluster you want the proxy to have access to. No configuration changes are
   required in this case.
2. Deploy the Teleport auth service outside of Kubernetes and update the Teleport Auth configuration with Kubernetes credentials. In this case, we need to update `/etc/teleport.yaml` of the auth service as
   shown below:

```bash
# snippet from /etc/teleport.yaml on the auth service deployed outside k8s:
auth_service:
    kubeconfig_file: /path/to/kubeconfig
```

To retrieve the Kubernetes credentials for the Teleport auth service, you have
to authenticate against your Kubernetes cluster directly then copy the file to
`/path/to/kubeconfig` on the Teleport auth server.

Unfortunately for GKE users, GKE requires its own client-side extensions to
authenticate, so we've created a [simple script](https://github.com/gravitational/ops/blob/master/get-kubeconfig.sh) 
you can run to generate `kubeconfig` for the Teleport auth service.

## Kubernetes RBAC

Once you perform the steps above, your Teleport instance should become a fully functional Kubernetes API
proxy. The next step is to configure Teleport to assign the correct Kubernetes
groups to Teleport users. 

Mapping Kubernetes groups to Teleport users depends on how Teleport is
configured. In this guide we'll look at two common configurations:

* **Open source, Teleport Community edition** configured to authenticate users via [Github](admin-guide.md#github-oauth-20).
  In this case, we'll need to map Github teams to Kubernetes groups.

* **Commercial, Teleport Enterprise edition** configured to authenticate users via [Okta SSO](ssh_okta.md). 
  In this case, we'll need to map users' groups that come from Okta to Kubernetes
  groups.

### Github Auth

When configuring Teleport to authenticate against Github, you have to create a
Teleport connector for Github, like the one shown below. Notice the `kubernetes_groups`
setting which assigns Kubernetes groups to a given Github team:

```
kind: github
version: v3
metadata:
  # connector name that will be used with `tsh --auth=github login`
  name: github
spec:
  # client ID of Github OAuth app
  client_id: <client-id>
  # client secret of Github OAuth app
  client_secret: <client-secret>
  # connector display name that will be shown on web UI login screen
  display: Github
  # callback URL that will be called after successful authentication
  redirect_url: https://teleport.example.com:3080/v1/webapi/github/callback
  # mapping of org/team memberships onto allowed logins and roles
  teams_to_logins:
    - organization: octocats # Github organization name
      team: admins           # Github team name within that organization
      # allowed UNIX logins for team octocats/admins:
      logins:
        - root
      # list of Kubernetes groups this Github team is allowed to connect to
      kubernetes_groups: ["system:masters"]
```
To obtain client ID and client secret from Github, please follow [Github documentation](https://developer.github.com/apps/building-oauth-apps/creating-an-oauth-app/) on how to create and register an OAuth app. Be sure to set the "Authorization callback URL" to the same value as redirect_url in the resource spec.

Finally, create the Github connector with the command: `tctl create -f github.yaml`. Now, when Teleport users execute the Teleport's `tsh login` command, they will be prompted to login through the Github SSO and upon successful authentication, they have access to Kubernetes.

```bash
# Login via Github SSO and retrieve SSH+Kubernetes certificates:
$ tsh login --proxy=teleport.example.com --auth=github login

# Use Kubernetes API!
$ kubectl exec -ti <pod-name>
```

The `kubectl exec` request will be routed through the Teleport proxy and
Teleport will log the audit record and record the session.

!!! note:
    For more information on integrating Teleport with Github SSO, please see the [Github section in the Admin Manual](admin-guide/#github-oauth-20).

### Okta Auth

With Okta (or any other SAML/OIDC/Active Directory provider), you must update
Teleport's roles to include the mapping to Kubernetes groups. 

Let's assume you have the Teleport role called "admin". Add `kubernetes_groups`
setting to it as shown below:

```bash
# NOTE: the role definition is edited to remove the unnecessary fields
kind: role
version: v3
metadata:
  name: admin
spec:
  allow:
    # if kubernetes integration is enabled, this setting configures which 
    # kubernetes groups the users of this role will be assigned to.
    # note that you can refer to a SAML/OIDC trait via the "external" property bag,
    # this allows you to specify Kubernetes group membership in an identity manager:
    kubernetes_groups: ["system:masters", "{{external.trait_name}}"]]
```

To add `kubernetes_groups` setting to an existing Teleport role, you can either
use the Web UI or `tctl`:

```bash
# Dump the "admin" role into a file:
$ tctl get roles/admin > admin.yaml
# Edit the file, add kubernetes_groups setting
# and then execute:
$ tctl create -f admin.yaml
```

!!! tip "Advanced Usage":
    `{{ external.trait_name }}` example is shown to demonstrate how to fetch
    the Kubernetes groups dynamically from Okta during login. In this case, you
    need to define Kubernetes group membership in Okta (as a trait) and use
    that trait name in the Teleport role.

Once this is complete, when users execute `tsh login` and go through the usual Okta login
sequence, their `kubeconfig` will be updated with their Kubernetes credentials.

!!! note:
    For more information on integrating Teleport with Okta, please see the [Okta integration guide](/ssh_okta/).
