
/*

    File: tcpproxy.c

    Copyright (C) 1999,2000,2004,2005,2007  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <ctype.h>
#include <signal.h>
#include <wait.h>
#include <errno.h>
#include <time.h>

#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <sys/time.h>

#include <linux/netfilter_ipv4.h>



#include "lib.h"
#include "ip-lib.h"
#include "procinfo.h"
#include "tcpproxy.h"


char	*program =		"";
char	progname[80] =		"";

int	debug =			0;
int	listports =		0;
int	onlynumbers =		0;
char	configfile[300] =	"";
/* char	varname[40] =		"PROXY_"; */


int	use_errorlog =		0;
char	errorlog[200] =		ERROR_LOG;



int printerror(int rc, char *type, char *format, ...)
{
	char	tag[30], error[400];
	va_list	ap;

	va_start(ap, format);
	vsnprintf (error, sizeof(error) - 2, format, ap);
	va_end(ap);

	*tag = 0;
	if (*type != 0)
		snprintf (tag, sizeof(tag) - 2, "%s: ", type);

	if (debug != 0)
		fprintf (stderr, "%s[%u]: %s%s\n", program, getpid(), tag, error);
	else if ((rc & ERR_STDERR) != 0  &&  isatty(0) != 0)
		fprintf (stderr, "%s: %s\n", program, error);
	else
		syslog(LOG_NOTICE, "%s%s", tag, error);

	if (pi.sessionfp != NULL) {
		fprintf (pi.sessionfp, "%lu %s[%u]: %s%s\n", time(NULL), program, getpid(), tag, error);
		fflush(pi.sessionfp);
		}


	if ((rc & ERR_INFO) != 0)
		pi.havemessages = 1;

	if ((rc & ERR_EXITCODEMASK) != 0  &&  (rc & ERR_STDERR) == 0) {
		run_errorhandler(rc & (~ERR_EXITCODEMASK));
		exit (rc & ERR_EXITCODEMASK);
		}

	return (0);
}

int writestatfile(proxy_t *x, char *status)
{
	if (getstatfp() == NULL)
		return (0);

	if (pi.statfp != NULL) {
		rewind(pi.statfp);
		fprintf (pi.statfp, "%s %s %u %lu %s:%u %s %s %s:%u %s:%u %s\n",
				PROXYNAME, *x->logname == 0? program: x->logname, getpid(),
				x->started,
				x->sock.ipnum, x->sock.port,
				x->client.ipnum, x->client.name,
				x->s.ipnum, x->s.port,
				x->origdst.ipnum, x->origdst.port,
				status);
		fflush(pi.statfp);
		}

	return (0);
}

void signalhandler(int sig)
{
	if (sig == SIGPIPE) {

		/* Ignore here and make write() return -1. */
		 
		signal(SIGPIPE, signalhandler);
		return;
		}

	printerror(0, "+INFO", "caught signal #%d", sig);
	exit(0);
}

char *gettime(void)
{
	char	time[20];
	struct tm tm;
	struct timeval tv;
	struct timezone tz;
	static char string[80];

	gettimeofday(&tv, &tz);
	tm = *localtime(&tv.tv_sec);
	strftime(time, sizeof(time) - 2, "%H:%M:%S", &tm);
	snprintf (string, sizeof(string) - 2, "%s.%06d", time, (int) tv.tv_usec);

	return (string);
}


int multiaccept_loop(config_t *config)
{
	int	i, max;
	fd_set	bound, available;

	FD_ZERO(&bound);
	max = 0;
	for (i=0; i < config->binds; i++) {
		if (config->bindport[i].sock < 0) {
			syslog(LOG_NOTICE, "invalid socket number found: %d", config->bindport[i].sock);
			exit (1);
			}
			
		FD_SET(config->bindport[i].sock, &bound);
		if (config->bindport[i].sock > max)
			max = config->bindport[i].sock;
		}


	while (1) {
		memmove(&available, &bound, sizeof(fd_set));
		if (select(max + 1, &available, NULL, NULL, NULL) < 0) {
			if (errno == EINTR) {
				syslog(LOG_NOTICE, "interrupted select call");
				continue;
				}

			syslog(LOG_NOTICE, "select error: %m");
			exit (1);
			}

		for (i=0; i < config->binds; i++) {
			if (FD_ISSET(config->bindport[i].sock, &available)) {
				unsigned int len;
				int	connect, pid;
				struct sockaddr_in client;

				/*
				 * hier kommt ein accept an
				 */

				len = sizeof(client);
				if ((connect = accept(config->bindport[i].sock, (struct sockaddr *) &client, &len)) < 0) {
					if (errno == EINTR  ||  errno == ECONNABORTED)
						continue;

					syslog(LOG_NOTICE, "accept error: %m");
					exit (1);
					}
		
				if ((pid = fork()) < 0) {
					syslog(LOG_NOTICE, "can't fork process: %m");
					exit (1);
					}
				else if (pid == 0) {
					dup2(connect, 0);
					dup2(connect, 1);
/*					dup2(connect, 2); */

					close (connect);
					close (config->bindport[i].sock);

					/*
					 * We have to close all listening ports
					 * before returning -- 24MAR00wzk
					 */

					for (i=0; i < config->binds; i++) {
						if (config->bindport[i].sock >= 0)
							close (config->bindport[i].sock);
						}

					return (0);
					}

				close (connect);
				}
			}
		}

	printerror(1, "-ERR", "server broke while loop -- terminating");
	return (0);
}


int proxy_request(proxy_t *x, int cfd)
{
	int	sfd, rc, bytes, errcode;
	unsigned long started, now;
	char	buffer[4096];
	struct timeval tov;
	fd_set	connection, available;
	FILE	*fp;


	fp = NULL;
	if (*x->writefile != 0) {
		char	filename[200];

		snprintf (filename, sizeof(filename) - 2, "%s.%d.log", x->writefile, getpid());
		if ((fp = fopen(filename, "a")) == NULL)
			printerror(0 | ERR_INFO, "-INFO", "can't open writefile: %s, error= %s", filename, strerror(errno));
		}

	time((time_t *) &started); 
	cfd = cfd;

	setvar(varname, "NAME", x->name);
	setvar(varname, "SERVERARG", x->server);

	x->srcip.port = 0;
	if (x->srcip.count == 0)
		sfd = openip(x->server, x->port, NULL, 0, &x->s);
	else if (x->srcip.count == 1)
		sfd = openip(x->server, x->port, x->srcip.srcip[0], 0, &x->s);
	else {
		int	i;
		unsigned int srcport;

		srcport = 8000 + x->port;
		sfd = -1;	/* to make the compiler happy */
		for (i=0; i<x->srcip.count; i++) {
			sfd = openip(x->server, x->port, x->srcip.srcip[i], srcport, &x->s);
			if (sfd >= 0) {
				x->srcip.port = srcport;
				copy_string(x->srcip.interface, x->srcip.srcip[i], sizeof(x->srcip.interface));
				break;
				}
			}
		}

	if (sfd < 0) {
		printerror(1 | ERR_CONNECT, "-ERR", "connection error: client= %s, server= %s:%u, error= %s",
			x->client.name, x->s.name, x->s.port, strerror(errno));
		}
	

	setvar(varname, "SERVER", x->s.ipnum);
	setvar(varname, "SERVERNAME", x->s.name);
	setnumvar(varname, "SERVERPORT", x->s.port);

	if (x->extendedinfo != 0) {
		peer_t	incoming, outgoing;

		incoming.port = get_interface_info(0, &incoming);
		outgoing.port = get_interface_info(sfd, &outgoing);
		printerror(0, "+INFO", "connect: client= %s:%u, interface= %s:%u, outgoing= %s:%u, server= %s:%u, timeout= %d, uid/gid= %u/%u",
			x->client.ipnum, x->client.port,
			incoming.ipnum, incoming.port,
			outgoing.ipnum, outgoing.port,
			x->s.name, x->s.port, x->timeout, getuid(), getgid());
		}
	else if (x->srcip.port != 0) {
		printerror(0, "+INFO", "connect: client= %s, interface= %s:%u, server= %s:%u, timeout= %d, uid/gid= %u/%u",
			x->client.name,
			x->srcip.interface, x->srcip.port,
			x->s.name, x->s.port, x->timeout, getuid(), getgid());
		}
	else {
		printerror(0, "+INFO", "connect: client= %s, server= %s:%u, timeout= %d, uid/gid= %u/%u",
			x->client.name, x->s.name, x->s.port, x->timeout, getuid(), getgid());
		}

	writestatfile(x, "connected");
	FD_ZERO(&connection);
	FD_SET(cfd, &connection);
	FD_SET(sfd, &connection);

	errcode = 0;
	while (1) {
		memmove(&available, &connection, sizeof(fd_set));
		tov.tv_sec  = x->timeout;
		tov.tv_usec = 0;

		rc = select(sfd + 1, &available, (fd_set *) NULL, (fd_set *) NULL, &tov);
		if (rc < 0) {
			printerror(0, "-ERR", "select() error, error= %s", strerror(errno));
			break;
			}
		else if (rc == 0) {
			errcode = ERR_TIMEOUT;
			printerror(0, "-ERR", "connection timed out: client= %s, server= %s:%u",
				x->client.name, x->s.name, x->s.port);
			break;
			}

		if (FD_ISSET(cfd, &available)) {
			if ((bytes = read(cfd, buffer, sizeof(buffer))) <= 0) {
				if (bytes < 0) {
					if (errno != ECONNRESET) {
						errcode = ERR_CLIENT;
						printerror(0 | ERR_INFO, "-INFO", "client close() detected, error= %s", strerror(errno));
						}
					}
				else if (debug != 0)	/* bytes == 0 */
					printerror(0, "+DEBUG", "client close() detected");

/*				shutdown(sfd, 1); */
				shutdown(sfd, SHUT_WR);

				FD_ZERO(&connection);
				FD_SET(sfd, &connection);
				}
			else if (write(sfd, buffer, bytes) != bytes) {
				errcode = ERR_SERVER;
				printerror(0, "-ERR", "server write error, error= %s", strerror(errno));
				break;
				}

			if (fp != NULL) {
				int	c, i;

				fprintf (fp, "--- CLIENT [%d] %s (%d):\n", getpid(), gettime(), bytes);
				for (i=0; i<bytes; i++) {
					c = buffer[i];
					if (c < 32  ||  c > 127  ||  c == '#')
						fprintf (fp, "#%02X", c);
					else
						fprintf (fp, "%c", c);
					}

				fprintf (fp, "\n");
				}

			if (bytes > 0)
				x->bytes_send = x->bytes_send + bytes;
			}

		if (FD_ISSET(sfd, &available)) {
			bytes = read(sfd, buffer, sizeof(buffer));
			if (bytes <= 0) {
				if (bytes < 0) {
					if (errno != ECONNRESET) {
						errcode = ERR_SERVER;
						printerror(0 | ERR_INFO, "-INFO", "server close() detected, error= %s", strerror(errno));
						}
					}
				else if (debug != 0) {
					printerror(0, "+INFO", "server close() detected");
					}

				break;
				}
			else {
				int	count;

				count = write(1, buffer, bytes);
				if (count != bytes) {
					errcode = ERR_CLIENT;
					printerror(0, "-ERR", "client write error, error= %s", strerror(errno));
					break;
					}
				}

			if (fp != NULL) {
				int	c, i;

				fprintf (fp, "--- SERVER [%d] %s (%d):\n", getpid(), gettime(), bytes);
				for (i=0; i<bytes; i++) {
					c = buffer[i];
					if (c < 32  ||  c > 127  ||  c == '#')
						fprintf (fp, "#%02X", c);
					else
						fprintf (fp, "%c", c);
					}

				fprintf (fp, "\n");
				}

			if (bytes > 0)
				x->bytes_received = x->bytes_received + bytes;
			}
		}

	time((time_t *) &now); 
	setnumvar(varname, "BYTES_CTS", x->bytes_send);
	setnumvar(varname, "BYTES_STC", x->bytes_received);
	setnumvar(varname, "DURATION", now - started);

	writestatfile(x, "disconnected");
	printerror(0, "+OK", "disconnect: client= %s, server= %s:%u, received= %ld, send= %ld, duration= %ld",
		x->client.name, x->s.name, x->s.port,
		x->bytes_received, x->bytes_send,
		now - started);

	if (errcode != 0)
		run_errorhandler(errcode);

	return (0);
}


int open_errorlog(char *filename)
{
	int	fno;

	if (use_errorlog == 0)
		return (0);

	if ((fno = open(filename, O_WRONLY | O_CREAT | O_APPEND, 0664)) >= 0)
		dup2(fno, 2);
	else 
		printerror(ERR_STDERR, "-INFO", "can't open %s, using stderr", filename);

	return (0);
}

void missing_arg(int c, char *string)
{
	printerror(ERR_STDERR, "-ERR", "missing arg: -%c, %s", c, string);
	exit (1);
}

int main(int argc, char *argv[], char *envp[])
{
	int	c, i, k, bound;
	char	*p, option[80];
	proxy_t *x, *precon;
	config_t *config;


	if ((p = strrchr(argv[0], '/')) == NULL)
		program = argv[0];
	else {
		copy_string(progname, &p[1], sizeof(progname));
		program = progname;
		}

	config = allocate(sizeof(config_t));

	x = config->def = allocate(sizeof(proxy_t));
	x->timeout   = 60;
	x->redirmode = REDIR_NONE;
	set_username(x, "nobody", "default-config", 0);

	init_procinfo("PROXY_");
	precon = allocate(sizeof(proxy_t));

	config->bindport = allocate(sizeof(port_t) * MAX_PORT);
	openlog(program, LOG_PID, LOG_MAIL);

	bound = 0;
	k = 1;
	while (k < argc  &&  argv[k][0] == '-'  &&  argv[k][1] != 0) {
		copy_string(option, argv[k++], sizeof(option));
		for (i=1; (c = option[i]) != 0; i++) {
			if (c == 'd')
				debug = 1;
			else if (c == 'a') {
				if (k >= argc)
					missing_arg(c, "access control program");

				copy_string(x->acp, argv[k++], sizeof(x->acp));
				}
			else if (c == 'b') {
				char	addr[300];
				
				if (k >= argc)
					missing_arg(c, "bind address and port");

				copy_string(addr, argv[k++], sizeof(addr));
				if ((p = strchr(addr, ':')) == NULL)
					x->port = getportnum(addr);
				else {
					*p++ = 0;
					x->port = getportnum(p);
					copy_string(x->interface, addr, sizeof(x->interface));
					}

				bound = 1;
				config->standalone = 1;
				}
			else if (c == 'e') {
				if (use_errorlog != 0) {
					if (k >= argc)
						missing_arg(c, "logfile name");

					copy_string(errorlog, argv[k++], sizeof(errorlog));
					}
					
				use_errorlog = 1;
				}
			else if (c == 'f'  ||  c == 'c') {
				if (k >= argc)
					missing_arg(c, "configuration file");

				copy_string(configfile, argv[k++], sizeof(configfile));
				}
			else if (c == 'i') {
				if (k >= argc)
					missing_arg(c, "rotation interfaces");

				addsrc(x, argv[k++], "command-line", 0);
				}
			else if (c == 'l') {
				if (k >= argc)
					missing_arg(c, "logname");

				copy_string(config->logname, argv[k++], sizeof(config->logname));
				}
			else if (c == 'n')
				onlynumbers = 1;
			else if (c == 'o') {
				if (k >= argc)
					missing_arg(c, "statdir");

				copy_string(statdir, argv[k++], sizeof(statdir));
				}
			else if (c == 'p') {
				if (*pi.pidfile == 0)
					setpidfile(PIDFILE);
				else {
					if (k >= argc)
						missing_arg(c, "pidfile");

					setpidfile(argv[k++]);
					}
				}
			else if (c == 'q') {
				if (k >= argc)
					missing_arg(c, "source ipnum");

				addsrcip(x , argv[k++]);
				}
			else if (c == 'r') {
				if (k >= argc)
					missing_arg(c, "redirect mode");

				set_redirmode(x, argv[k++], "command-line", 0);
				}
			else if (c == 's')
				config->standalone = 1;
			else if (c == 't') {
				if (k >= argc)
					missing_arg(c, "timeout");

				x->timeout = atoi(argv[k++]);
				if (x->timeout < 1)
					x->timeout = 60;
				}
			else if (c == 'u') {
				if (k >= argc)
					missing_arg(c, "username");

				set_username(x, argv[k++], "command-line", 0);
				}
			else if (c == 'v') {
				if (k >= argc)
					missing_arg(c, "varname prefix");

				copy_string(varname, argv[k++], sizeof(varname));
				}
			else if (c == 'w') {
				if (k >= argc)
					missing_arg(c, "write file");

				copy_string(x->writefile, argv[k++], sizeof(x->writefile));
				}
			else if (c == 'x') {
				if (k >= argc)
					missing_arg(c, "ctp");

				copy_string(x->ctp, argv[k++], sizeof(x->ctp));
				}
			else if (c == 'y')
				x->cleanenv = 1;
			else if (c == 'z') {
				if (listports == 0)
					listports = 1;
				else
					listports = 2;
				}
			else if (c == 'A') {
				if (k >= argc)
					missing_arg(c, "acl parameter");

				append(&x->acl, x->acl.len > 0? " ": "", argv[k++]);
				}
			else if (c == 'E') {
				char	mode[40], par[200];

				if (k >= argc)
					missing_arg(c, "session mode parameter");

				p = copy_string(par, argv[k++], sizeof(par));
				p = get_quoted(&p, ',', mode, sizeof(mode));
				set_sessionmode(mode, "command-line", 0);
				get_quoted(&p, ',', sessiondir, sizeof(sessiondir));

				get_quoted(&p, ',', par, sizeof(par));
				set_exithandler(par);

				set_exitcodes(ERR_ANY);
				}
			else if (c == 'I')
				x->extendedinfo = 1;
			else if (c == 'O') {
				if (k >= argc)
					missing_arg(c, "sessiondir");

				copy_string(sessiondir, argv[k++], sizeof(sessiondir));
				}
			else if (c == 'V') {
				printf ("%s %s\n", program, VERSION);
				exit (0);
				}
			else {
				printerror(ERR_STDERR, "-ERR", "unknown option: -%c", c);
				exit (1);
				}
			}
		}



	if (*configfile != 0)
		read_configuration(config, configfile);
 	else if (bound == 0  &&  k >= argc) {
		copy_string(configfile, "/etc/tcpproxy.conf", sizeof(configfile));

		/* This default is the most common cause for tcpproxy installations
		 * to fail initially.
		 *
		 * config->standalone = 1;
		 *
		 * Only if stdin is not a socket then tcpproxy will bind and go
		 * into standalone. -- 2007-08-30/wzk
		 */

		if (issock(0) != 0)
			config->standalone = 1;

		read_configuration(config, configfile);
		}
	else if (listports == 1)
		/* ist ok, ein Serverprogramm wird nicht benoetigt */ ;
	else if (k >= argc) {
		printerror(ERR_STDERR, "-ERR", "missing server");
		exit (1);
		}
	else {
		if (*argv[k] == '/'  ||  *argv[k] == '.') {
			x->proxytype = PROXY_PROGRAM;
			copy_string(x->server, argv[k++], sizeof(x->server));

/*			x->proxytype = PROXY_PROGRAM;
 *
 *			i = 0;
 *			while (k < argc)
 *				x->u.program.argv[i++] = argv[k++];
 *
 *			x->u.program.argv[i] = NULL;
 */
			}
		else {
			x->proxytype = PROXY_SERVER;

			copy_string(x->server, argv[k++], sizeof(x->server));
			if (k < argc) {
				printerror(ERR_STDERR, "-ERR", "extra arguments after server");
				exit (1);
				}
			}
		}
	

	if (*sessiondir == 0) {
		if (pi.errormode != ERRMODE_NONE  &&  *statdir != 0)
			copy_string(sessiondir, statdir, sizeof(sessiondir));
		}

	if (*config->logname != 0) {
		copy_string(progname, config->logname, sizeof(progname));
		closelog();
		openlog(config->logname, LOG_PID, LOG_MAIL);
		}

	if (config->first == NULL) {
		x = new_configuration(config, x);
		config->bindport[config->binds].port  = x->port;
		config->bindport[config->binds].count = 1;
		config->binds++;
		}


	if (listports == 1) {
		for (i=0; i < config->binds; i++) {
			if (config->bindport[i].port != 0)
				printf ("%u\n", config->bindport[i].port);
			}
			
		exit (0);
		}
	else if (listports == 2) {
		proxy_t *y;

		y = config->first;
		while (y != NULL) {
			printf ("%s %u", y->interface, y->port);
			if (y->proxytype == PROXY_PROGRAM) {
/*				int	i; */

				printf ("  exec:");
/*				for (i=0; y->u.program.argv[i] != NULL; i++)
 *					printf (" %s", y->u.program.argv[i]);
 */
				}
			else {
				printf ("  proxy:");
				printf (" %s", y->server);
				}

			printf ("\n");
			y = y->next;
			}

		exit (0);
		}


	atexit(exithandler);
	signal(SIGHUP, signalhandler);
	signal(SIGINT, signalhandler);
	signal(SIGQUIT, signalhandler);
	signal(SIGUSR1, signalhandler);
	signal(SIGUSR2, signalhandler);
	signal(SIGTERM, signalhandler);
	signal(SIGPIPE, signalhandler);

	if (config->standalone != 0) {
		int	i;
		char	*ip, *bindip;
		proxy_t	*x;

		/*
		 * Compute to which ports we have to bind.
		 */

		config->binds = 0;
		x = config->first;
		while (x != NULL) {
			for (i=0; i < config->binds; i++) {
				if (config->bindport[i].port == x->port) {
					config->bindport[i].count++;
					*config->bindport[i].ip = 0;

					break;
					}
				}

			if (i >= config->binds) {
				int	k;

				k = config->binds++;
				config->bindport[k].count = 1;
				config->bindport[k].port  = x->port;
				copy_string(config->bindport[k].ip, x->interface, sizeof(config->bindport[k].ip));
				}

			x = x->next;
			}


		pi.mainpid = getpid();
		for (i=0; i < config->binds; i++) {
			if (config->bindport[i].port == 0)
				continue;	/* Mogelt sich irgendwie in die Port-Liste -- 20AUG99wzk */
				
			ip = config->bindport[i].ip;
			if (strcmp(ip, "*") == 0  ||  config->bindport[i].count > 1)
				ip = "0.0.0.0";

			if (*ip == 0  ||  strcmp(ip, "0.0.0.0") == 0  ||  isalpha(*ip) != 0)
				bindip = "0.0.0.0";
			else
				bindip = ip;

/*			config->bindport[i].sock = bind_to_port(strcmp(ip, "0.0.0.0") == 0? "": ip, config->bindport[i].port); */
			config->bindport[i].sock = bind_to_port(bindip, config->bindport[i].port);
			printerror(0, "", "bound to %s:%u",
				bindip, config->bindport[i].port);
			}
			
		signal(SIGCHLD, SIG_IGN);
		config->standalone = 1;
		}


	/*
	 * Go into the background.
	 */

	if (debug == 0) {
		if (config->standalone != 0) {
			int	pid;
		
			if ((pid = fork()) < 0) {
				printerror(ERR_STDERR, "-ERR", "can't fork into background");
				exit (1);
				}
			else if (pid > 0)
				exit (0);

			printerror(0, "", "ready");
			}
		
		open_errorlog(errorlog);
		}

	/*
	 * Write the pidfile if we are in standalone mode.
	 */

	if (config->standalone != 0  &&  *pi.pidfile != 0) {
		FILE	*fp;

		if ((fp = fopen(pi.pidfile, "w")) == NULL)
			printerror(1, "-ERR", "can't write pidfile: %s, error= %s", pi.pidfile, strerror(errno));

		fprintf (fp, "%d\n", getpid());
		pi.mainpid = getpid();

		fclose (fp);
		}


	/*
	 * We are ready to run.
	 */
	 
	if (config->standalone != 0)
		multiaccept_loop(config);

	signal(SIGCHLD, SIG_DFL);
	setvar(varname, "PROXYNAME", PROXYNAME);
	setnumvar(varname, "PID", getpid());


	/*
	 * At this point we have an incoming connection on stdin and
	 * stdout.  If tcpproxy is in standalone mode the server
	 * process is still in multiaccept_loop().
	 *
	 * Let's do some basic setup first.
	 */

	if (*sessiondir != 0) {
		snprintf (pi.sessionfile, sizeof(pi.sessionfile) - 2, "%s/%s-%05d.log",
				sessiondir, program, getpid());
		if ((pi.sessionfp = fopen(pi.sessionfile, "w+")) == NULL) {
			printerror(0, "-ERR", "can't create session file %s, error= %s",
					pi.sessionfile, strerror(errno));
			}
		}

	/*
	 * Fill the "pre connection" proxy data structure to use
	 * it until we have the correct configuration.
	 */

	get_interface_info(0, &precon->sock);
	setvar(varname, "INTERFACE", precon->sock.ipnum);
	setnumvar(varname, "PORT", precon->sock.port);

	precon->port = get_client_info(0, &precon->client, onlynumbers);
	setvar(varname, "CLIENT", precon->client.ipnum);
	setvar(varname, "CLIENTNAME", precon->client.name);

	printerror(0, "+INFO", "connected on %s:%u, client= %s:%u",
					precon->sock.ipnum, precon->sock.port,
					precon->client.ipnum, precon->client.port);

	/*
	 * Search the configuration for the port and interface of the
	 * connection.  It there is none, terminate.
	 */

	x = find_configuration(config, precon->sock.ipnum, precon->sock.port);
	if (x == NULL) {
		printerror(1, "-ERR", "no configuration found for interface %s:%u, client= %s",
				precon->sock.ipnum, precon->sock.port, precon->client.ipnum);
		}

	setvar(varname, "LOGNAME", (*x->logname != 0)? x->logname: program);
	snprintf (option, sizeof(option) - 2, "%s:%u", x->interface, x->port);
	setvar(varname, "CONFIG", option);


	/*
	 * Take socket and client information and create the statfile.
	 */

	x->sock   = precon->sock;
	x->client = precon->client;
	free(precon);

	/*
	 * Initialize the original destination ...
	 */

	copy_string(x->origdst.ipnum, "0.0.0.0", sizeof(x->origdst.ipnum));
	x->origdst.port = 0;

	/*
	 * ... the startup time ...
	 */
	
	x->started = time(NULL);
	setnumvar(varname, "STARTED", x->started);


	/*
	 * ... and write the statfile if configured.
	 */

	writestatfile(x, "request");



	/*
	 * Apply uid/gid and log settings from the configuration.
	 */

	if (*x->logname != 0) {
		closelog();
		openlog(x->logname, LOG_PID, LOG_MAIL);
		}

 	if (getuid() != 0  &&  geteuid() != 0)
		/* nichts */ ;
	else {
		struct passwd *pw;

		if (pi.statfp != NULL)
			fchown(fileno(pi.statfp), x->uid, x->gid);

		if (pi.sessionfp != NULL)
			fchown(fileno(pi.sessionfp), x->uid, x->gid);

		if ((pw = getpwuid(x->uid)) != NULL) {
			if (initgroups(pw->pw_name, pw->pw_gid) != 0) {
				printerror(0, "-INFO", "can't initgroups, error= %s", strerror(errno));
				}
			}

		if (setgid(x->gid) != 0  ||  setuid(x->uid) != 0) {
			printerror(1, "-ERR", "can't change uid/gid to %u/%u, error= %s",
					x->uid, x->gid, strerror(errno));
			}
		}


	if (x->redirmode != 0) {
		int	rc;
		size_t	socksize;
		struct sockaddr_in sock;

		socksize = sizeof(sock);
		rc = getsockopt(0, SOL_IP, SO_ORIGINAL_DST, &sock, &socksize);
		if (rc != 0)
			;
		else if (strcmp((char *) inet_ntoa(sock.sin_addr), x->sock.ipnum) != 0  ||
			 ntohs(sock.sin_port) != x->sock.port) {

			/*
			 * Take the original server information if it's
			 * a redirected request.
			 */

			copy_string(x->origdst.ipnum, (char *) inet_ntoa(sock.sin_addr), sizeof(x->origdst.ipnum));
			x->origdst.port = ntohs(sock.sin_port);
			setvar(varname, "ORIGDST_SERVER", x->origdst.ipnum);
			setnumvar(varname, "ORIGDST_PORT", x->origdst.port);

			printerror(0, "+INFO", "connection redirected, origdst: %s:%u", x->origdst.ipnum, x->origdst.port);
			}
		}


	/*
	 * Check acl if we have one.
	 */

	if (x->acl.len > 0) {
		if (checkacl(x->acl.ptr, x->client.ipnum, ACL_DENY) != ACL_ALLOW) {
			printerror(1 | ERR_ACCESS, "-ERR", "access denied by acl: client= %s, ip= %s",
				x->client.name, x->client.ipnum);
			}
		}


	/*
	 * Access control Programm ausfuehren
	 */

	if (*x->acp != 0) {
		int	rc;

		rc = run_acp(x);
		if (rc != 0)
			exit (0);
		}


	/*
	 * Handle transparent redirection if required.
	 */

	if (x->origdst.port != 0) {
		if (x->redirmode == REDIR_FORWARD  ||  x->redirmode == REDIR_FORWARD_ONLY) {

			/*
			 * Take original requested destination as server.
			 */

			snprintf (x->server, sizeof(x->server) - 2, "%s:%u",
					x->origdst.ipnum, x->origdst.port);
			}
		}
	else if (x->redirmode == REDIR_FORWARD_ONLY)
		printerror(1, "-ERR", "connection not redirected, client= %s", x->client.name);


	/*
	 * Start server program or proxy connection.
	 */

	if (*x->server == '/') {
		int	argc;
		char	*argv[32];

		printerror(0, "", "starting `%s', client= %s, uid/gid= %u/%u",
				x->server, x->client.name, getuid(), getgid());
		
		exithandler();		/* cleanup */
		if (x->cleanenv != 0)
			cleanenv(envp);

		argc = split(x->server, argv, ' ', 30);
		argv[argc] = NULL;
		execvp(argv[0], argv);
		
		printerror(1, "-ERR", "can't execute %s, error= %s", argv[0], strerror(errno));
		exit (1);
		}
	else {
		proxy_request(x, 0);
		close(0);

		if (pi.errorcode == 0) {
			if (pi.havemessages != 0)
				run_errorhandler(ERR_INFO);
			else
				run_errorhandler(ERR_OK);
			}
		}

	exit (0);
}


