//
//  RulesTests+General.swift
//  SwiftFormatTests
//
//  Created by Nick Lockwood on 02/10/2021.
//  Copyright © 2021 Nick Lockwood. All rights reserved.
//

import XCTest
@testable import SwiftFormat

private enum TestDateFormat: String {
    case basic = "yyyy-MM-dd"
    case time = "HH:mmZZZZZ"
    case timestamp = "yyyy-MM-dd'T'HH:mm:ss.SSSZZZZZ"
}

private func createTestDate(
    _ input: String,
    _ format: TestDateFormat = .basic
) -> Date {
    let formatter = DateFormatter()
    formatter.dateFormat = format.rawValue
    formatter.timeZone = .current

    return formatter.date(from: input)!
}

class GeneralTests: RulesTests {
    // MARK: - initCoderUnavailable

    func testInitCoderUnavailableEmptyFunction() {
        let input = """
        struct A: UIView {
            required init?(coder aDecoder: NSCoder) {}
        }
        """
        let output = """
        struct A: UIView {
            @available(*, unavailable)
            required init?(coder aDecoder: NSCoder) {}
        }
        """
        testFormatting(for: input, output, rule: FormatRules.initCoderUnavailable,
                       exclude: ["unusedArguments"])
    }

    func testInitCoderUnavailableFatalErrorNilDisabled() {
        let input = """
        extension Module {
            final class A: UIView {
                required init?(coder _: NSCoder) {
                    fatalError("init(coder:) has not been implemented")
                }
            }
        }
        """
        let output = """
        extension Module {
            final class A: UIView {
                @available(*, unavailable)
                required init?(coder _: NSCoder) {
                    fatalError("init(coder:) has not been implemented")
                }
            }
        }
        """
        let options = FormatOptions(initCoderNil: false)
        testFormatting(for: input, output, rule: FormatRules.initCoderUnavailable, options: options)
    }

    func testInitCoderUnavailableFatalErrorNilEnabled() {
        let input = """
        extension Module {
            final class A: UIView {
                required init?(coder _: NSCoder) {
                    fatalError("init(coder:) has not been implemented")
                }
            }
        }
        """
        let output = """
        extension Module {
            final class A: UIView {
                @available(*, unavailable)
                required init?(coder _: NSCoder) {
                    nil
                }
            }
        }
        """
        let options = FormatOptions(initCoderNil: true)
        testFormatting(for: input, output, rule: FormatRules.initCoderUnavailable, options: options)
    }

    func testInitCoderUnavailableAlreadyPresent() {
        let input = """
        extension Module {
            final class A: UIView {
                @available(*, unavailable)
                required init?(coder _: NSCoder) {
                    fatalError()
                }
            }
        }
        """
        testFormatting(for: input, rule: FormatRules.initCoderUnavailable)
    }

    func testInitCoderUnavailableImplemented() {
        let input = """
        extension Module {
            final class A: UIView {
                required init?(coder aCoder: NSCoder) {
                    aCoder.doSomething()
                }
            }
        }
        """
        testFormatting(for: input, rule: FormatRules.initCoderUnavailable)
    }

    func testPublicInitCoderUnavailable() {
        let input = """
        class Foo: UIView {
            public required init?(coder _: NSCoder) {
                fatalError("init(coder:) has not been implemented")
            }
        }
        """
        let output = """
        class Foo: UIView {
            @available(*, unavailable)
            public required init?(coder _: NSCoder) {
                fatalError("init(coder:) has not been implemented")
            }
        }
        """
        testFormatting(for: input, output, rule: FormatRules.initCoderUnavailable)
    }

    func testPublicInitCoderUnavailable2() {
        let input = """
        class Foo: UIView {
            required public init?(coder _: NSCoder) {
                fatalError("init(coder:) has not been implemented")
            }
        }
        """
        let output = """
        class Foo: UIView {
            @available(*, unavailable)
            required public init?(coder _: NSCoder) {
                nil
            }
        }
        """
        let options = FormatOptions(initCoderNil: true)
        testFormatting(for: input, output, rule: FormatRules.initCoderUnavailable,
                       options: options, exclude: ["modifierOrder"])
    }

    // MARK: - trailingCommas

    func testCommaAddedToSingleItem() {
        let input = "[\n    foo\n]"
        let output = "[\n    foo,\n]"
        testFormatting(for: input, output, rule: FormatRules.trailingCommas)
    }

    func testCommaAddedToLastItem() {
        let input = "[\n    foo,\n    bar\n]"
        let output = "[\n    foo,\n    bar,\n]"
        testFormatting(for: input, output, rule: FormatRules.trailingCommas)
    }

    func testCommaAddedToDictionary() {
        let input = "[\n    foo: bar\n]"
        let output = "[\n    foo: bar,\n]"
        testFormatting(for: input, output, rule: FormatRules.trailingCommas)
    }

    func testCommaNotAddedToInlineArray() {
        let input = "[foo, bar]"
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testCommaNotAddedToInlineDictionary() {
        let input = "[foo: bar]"
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testCommaNotAddedToSubscript() {
        let input = "foo[bar]"
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testCommaAddedBeforeComment() {
        let input = "[\n    foo // comment\n]"
        let output = "[\n    foo, // comment\n]"
        testFormatting(for: input, output, rule: FormatRules.trailingCommas)
    }

    func testCommaNotAddedAfterComment() {
        let input = "[\n    foo, // comment\n]"
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testCommaNotAddedInsideEmptyArrayLiteral() {
        let input = "foo = [\n]"
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testCommaNotAddedInsideEmptyDictionaryLiteral() {
        let input = "foo = [:\n]"
        let options = FormatOptions(wrapCollections: .disabled)
        testFormatting(for: input, rule: FormatRules.trailingCommas, options: options)
    }

    func testTrailingCommaRemovedInInlineArray() {
        let input = "[foo,]"
        let output = "[foo]"
        testFormatting(for: input, output, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToSubscript() {
        let input = "foo[\n    bar\n]"
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToSubscript2() {
        let input = "foo?[\n    bar\n]"
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToSubscript3() {
        let input = "foo()[\n    bar\n]"
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToSubscriptInsideArrayLiteral() {
        let input = """
        let array = [
            foo
                .bar[
                    0
                ]
                .baz,
        ]
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaAddedToArrayLiteralInsideTuple() {
        let input = """
        let arrays = ([
            foo
        ], [
            bar
        ])
        """
        let output = """
        let arrays = ([
            foo,
        ], [
            bar,
        ])
        """
        testFormatting(for: input, output, rule: FormatRules.trailingCommas)
    }

    func testNoTrailingCommaAddedToArrayLiteralInsideTuple() {
        let input = """
        let arrays = ([
            Int
        ], [
            Int
        ]).self
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToTypeDeclaration() {
        let input = """
        var foo: [
            Int:
                String
        ]
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToTypeDeclaration2() {
        let input = """
        func foo(bar: [
            Int:
                String
        ])
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToTypeDeclaration3() {
        let input = """
        func foo() -> [
            String: String
        ]
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToTypeDeclaration4() {
        let input = """
        func foo() -> [String: [
            String: Int
        ]]
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToTypeDeclaration5() {
        let input = """
        let foo = [String: [
            String: Int
        ]]()
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToTypeDeclaration6() {
        let input = """
        let foo = [String: [
            (Foo<[
                String
            ]>, [
                Int
            ])
        ]]()
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToTypeDeclaration7() {
        let input = """
        func foo() -> Foo<[String: [
            String: Int
        ]]>
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToTypeDeclaration8() {
        let input = """
        extension Foo {
            var bar: [
                Int
            ] {
                fatalError()
            }
        }
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToTypealias() {
        let input = """
        typealias Foo = [
            Int
        ]
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToCaptureList() {
        let input = """
        let foo = { [
            self
        ] in }
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToCaptureListWithComment() {
        let input = """
        let foo = { [
            self // captures self
        ] in }
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    func testTrailingCommaNotAddedToCaptureListWithMainActor() {
        let input = """
        let closure = { @MainActor [
            foo = state.foo,
            baz = state.baz
        ] _ in }
        """
        testFormatting(for: input, rule: FormatRules.trailingCommas)
    }

    // trailingCommas = false

    func testCommaNotAddedToLastItem() {
        let input = "[\n    foo,\n    bar\n]"
        let options = FormatOptions(trailingCommas: false)
        testFormatting(for: input, rule: FormatRules.trailingCommas, options: options)
    }

    func testCommaRemovedFromLastItem() {
        let input = "[\n    foo,\n    bar,\n]"
        let output = "[\n    foo,\n    bar\n]"
        let options = FormatOptions(trailingCommas: false)
        testFormatting(for: input, output, rule: FormatRules.trailingCommas, options: options)
    }

    // MARK: - fileHeader

    func testStripHeader() {
        let input = "//\n//  test.swift\n//  SwiftFormat\n//\n//  Created by Nick Lockwood on 08/11/2016.\n//  Copyright © 2016 Nick Lockwood. All rights reserved.\n//\n\n/// func\nfunc foo() {}"
        let output = "/// func\nfunc foo() {}"
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testStripHeaderWithWhenHeaderContainsUrl() {
        let input = """
        //
        //  RulesTests+General.swift
        //  SwiftFormatTests
        //
        //  Created by Nick Lockwood on 02/10/2021.
        //  Copyright © 2021 Nick Lockwood. All rights reserved.
        //  https://some.example.com
        //

        /// func
        func foo() {}
        """
        let output = "/// func\nfunc foo() {}"
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testReplaceHeaderWhenFileContainsNoCode() {
        let input = "// foobar"
        let options = FormatOptions(fileHeader: "// foobar")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options,
                       exclude: ["linebreakAtEndOfFile"])
    }

    func testReplaceHeaderWhenFileContainsNoCode2() {
        let input = "// foobar\n"
        let options = FormatOptions(fileHeader: "// foobar")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testMultilineCommentHeader() {
        let input = "/****************************/\n/* Created by Nick Lockwood */\n/****************************/\n\n\n/// func\nfunc foo() {}"
        let output = "/// func\nfunc foo() {}"
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testNoStripHeaderWhenDisabled() {
        let input = "//\n//  test.swift\n//  SwiftFormat\n//\n//  Created by Nick Lockwood on 08/11/2016.\n//  Copyright © 2016 Nick Lockwood. All rights reserved.\n//\n\n/// func\nfunc foo() {}"
        let options = FormatOptions(fileHeader: .ignore)
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testNoStripComment() {
        let input = "\n/// func\nfunc foo() {}"
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testNoStripPackageHeader() {
        let input = "// swift-tools-version:4.2\n\nimport PackageDescription"
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testNoStripFormatDirective() {
        let input = "// swiftformat:options --swiftversion 5.2\n\nimport PackageDescription"
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testNoStripFormatDirectiveAfterHeader() {
        let input = "// header\n// swiftformat:options --swiftversion 5.2\n\nimport PackageDescription"
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testNoReplaceFormatDirective() {
        let input = "// swiftformat:options --swiftversion 5.2\n\nimport PackageDescription"
        let output = "// Hello World\n\n// swiftformat:options --swiftversion 5.2\n\nimport PackageDescription"
        let options = FormatOptions(fileHeader: "// Hello World")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testSetSingleLineHeader() {
        let input = "//\n//  test.swift\n//  SwiftFormat\n//\n//  Created by Nick Lockwood on 08/11/2016.\n//  Copyright © 2016 Nick Lockwood. All rights reserved.\n//\n\n/// func\nfunc foo() {}"
        let output = "// Hello World\n\n/// func\nfunc foo() {}"
        let options = FormatOptions(fileHeader: "// Hello World")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testSetMultilineHeader() {
        let input = "//\n//  test.swift\n//  SwiftFormat\n//\n//  Created by Nick Lockwood on 08/11/2016.\n//  Copyright © 2016 Nick Lockwood. All rights reserved.\n//\n\n/// func\nfunc foo() {}"
        let output = "// Hello\n// World\n\n/// func\nfunc foo() {}"
        let options = FormatOptions(fileHeader: "// Hello\n// World")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testSetMultilineHeaderWithMarkup() {
        let input = "//\n//  test.swift\n//  SwiftFormat\n//\n//  Created by Nick Lockwood on 08/11/2016.\n//  Copyright © 2016 Nick Lockwood. All rights reserved.\n//\n\n/// func\nfunc foo() {}"
        let output = "/*--- Hello ---*/\n/*--- World ---*/\n\n/// func\nfunc foo() {}"
        let options = FormatOptions(fileHeader: "/*--- Hello ---*/\n/*--- World ---*/")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testNoStripHeaderIfRuleDisabled() {
        let input = "// swiftformat:disable fileHeader\n// test\n// swiftformat:enable fileHeader\n\nfunc foo() {}"
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testNoStripHeaderIfNextRuleDisabled() {
        let input = "// swiftformat:disable:next fileHeader\n// test\n\nfunc foo() {}"
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testNoStripHeaderDocWithNewlineBeforeCode() {
        let input = "/// Header doc\n\nclass Foo {}"
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options, exclude: ["docComments"])
    }

    func testNoDuplicateHeaderIfMissingTrailingBlankLine() {
        let input = "// Header comment\nclass Foo {}"
        let output = "// Header comment\n\nclass Foo {}"
        let options = FormatOptions(fileHeader: "Header comment")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testNoDuplicateHeaderContainingPossibleCommentDirective() {
        let input = """
        // Copyright (c) 2010-2023 Foobar
        //
        // SPDX-License-Identifier: EPL-2.0

        class Foo {}
        """
        let output = """
        // Copyright (c) 2010-2024 Foobar
        //
        // SPDX-License-Identifier: EPL-2.0

        class Foo {}
        """
        let options = FormatOptions(fileHeader: "// Copyright (c) 2010-2024 Foobar\n//\n// SPDX-License-Identifier: EPL-2.0")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testNoDuplicateHeaderContainingCommentDirective() {
        let input = """
        // Copyright (c) 2010-2023 Foobar
        //
        // swiftformat:disable all

        class Foo {}
        """
        let output = """
        // Copyright (c) 2010-2024 Foobar
        //
        // swiftformat:disable all

        class Foo {}
        """
        let options = FormatOptions(fileHeader: "// Copyright (c) 2010-2024 Foobar\n//\n// swiftformat:disable all")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderYearReplacement() {
        let input = "let foo = bar"
        let output: String = {
            let formatter = DateFormatter()
            formatter.dateFormat = "yyyy"
            return "// Copyright © \(formatter.string(from: Date()))\n\nlet foo = bar"
        }()
        let options = FormatOptions(fileHeader: "// Copyright © {year}")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderCreationYearReplacement() {
        let input = "let foo = bar"
        let date = Date(timeIntervalSince1970: 0)
        let output: String = {
            let formatter = DateFormatter()
            formatter.dateFormat = "yyyy"
            return "// Copyright © \(formatter.string(from: date))\n\nlet foo = bar"
        }()
        let fileInfo = FileInfo(creationDate: date)
        let options = FormatOptions(fileHeader: "// Copyright © {created.year}", fileInfo: fileInfo)
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderAuthorReplacement() {
        let name = "Test User"
        let email = "test@email.com"
        let input = "let foo = bar"
        let output = "// Created by \(name) \(email)\n\nlet foo = bar"
        let fileInfo = FileInfo(replacements: [.authorName: .constant(name), .authorEmail: .constant(email)])
        let options = FormatOptions(fileHeader: "// Created by {author.name} {author.email}", fileInfo: fileInfo)
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderAuthorReplacement2() {
        let author = "Test User <test@email.com>"
        let input = "let foo = bar"
        let output = "// Created by \(author)\n\nlet foo = bar"
        let fileInfo = FileInfo(replacements: [.author: .constant(author)])
        let options = FormatOptions(fileHeader: "// Created by {author}", fileInfo: fileInfo)
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderMultipleReplacement() {
        let name = "Test User"
        let input = "let foo = bar"
        let output = "// Copyright © \(name)\n// Created by \(name)\n\nlet foo = bar"
        let fileInfo = FileInfo(replacements: [.authorName: .constant(name)])
        let options = FormatOptions(fileHeader: "// Copyright © {author.name}\n// Created by {author.name}", fileInfo: fileInfo)
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderCreationDateReplacement() {
        let input = "let foo = bar"
        let date = Date(timeIntervalSince1970: 0)
        let output: String = {
            let formatter = DateFormatter()
            formatter.dateStyle = .short
            formatter.timeStyle = .none
            return "// Created by Nick Lockwood on \(formatter.string(from: date)).\n\nlet foo = bar"
        }()
        let fileInfo = FileInfo(creationDate: date)
        let options = FormatOptions(fileHeader: "// Created by Nick Lockwood on {created}.", fileInfo: fileInfo)
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderDateFormattingIso() {
        let date = createTestDate("2023-08-09")

        let input = "let foo = bar"
        let output = "// 2023-08-09\n\nlet foo = bar"
        let fileInfo = FileInfo(creationDate: date)
        let options = FormatOptions(fileHeader: "// {created}", dateFormat: .iso, fileInfo: fileInfo)
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderDateFormattingDayMonthYear() {
        let date = createTestDate("2023-08-09")

        let input = "let foo = bar"
        let output = "// 09/08/2023\n\nlet foo = bar"
        let fileInfo = FileInfo(creationDate: date)
        let options = FormatOptions(fileHeader: "// {created}", dateFormat: .dayMonthYear, fileInfo: fileInfo)
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderDateFormattingMonthDayYear() {
        let date = createTestDate("2023-08-09")

        let input = "let foo = bar"
        let output = "// 08/09/2023\n\nlet foo = bar"
        let fileInfo = FileInfo(creationDate: date)
        let options = FormatOptions(fileHeader: "// {created}",
                                    dateFormat: .monthDayYear,
                                    fileInfo: fileInfo)
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderDateFormattingCustom() {
        let date = createTestDate("2023-08-09T12:59:30.345Z", .timestamp)

        let input = "let foo = bar"
        let output = "// 23.08.09-12.59.30.345\n\nlet foo = bar"
        let fileInfo = FileInfo(creationDate: date)
        let options = FormatOptions(fileHeader: "// {created}",
                                    dateFormat: .custom("yy.MM.dd-HH.mm.ss.SSS"),
                                    timeZone: .identifier("UTC"),
                                    fileInfo: fileInfo)
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    private func testTimeZone(
        timeZone: FormatTimeZone,
        tests: [String: String]
    ) {
        for (input, expected) in tests {
            let date = createTestDate(input, .time)
            let input = "let foo = bar"
            let output = "// \(expected)\n\nlet foo = bar"

            let fileInfo = FileInfo(creationDate: date)

            let options = FormatOptions(
                fileHeader: "// {created}",
                dateFormat: .custom("HH:mm"),
                timeZone: timeZone,
                fileInfo: fileInfo
            )

            testFormatting(for: input, output,
                           rule: FormatRules.fileHeader,
                           options: options)
        }
    }

    func testFileHeaderDateTimeZoneSystem() {
        let baseDate = createTestDate("15:00Z", .time)
        let offset = TimeZone.current.secondsFromGMT(for: baseDate)

        let date = baseDate.addingTimeInterval(Double(offset))

        let formatter = DateFormatter()
        formatter.dateFormat = "HH:mm"
        formatter.timeZone = TimeZone(secondsFromGMT: 0)

        let expected = formatter.string(from: date)

        testTimeZone(timeZone: .system, tests: [
            "15:00Z": expected,
            "16:00+1": expected,
            "01:00+10": expected,
            "16:30+0130": expected,
        ])
    }

    func testFileHeaderDateTimeZoneAbbreviations() {
        // GMT+0530
        testTimeZone(timeZone: FormatTimeZone(rawValue: "IST")!, tests: [
            "15:00Z": "20:30",
            "16:00+1": "20:30",
            "01:00+10": "20:30",
            "16:30+0130": "20:30",
        ])
    }

    func testFileHeaderDateTimeZoneIdentifiers() {
        // GMT+0845
        testTimeZone(timeZone: FormatTimeZone(rawValue: "Australia/Eucla")!, tests: [
            "15:00Z": "23:45",
            "16:00+1": "23:45",
            "01:00+10": "23:45",
            "16:30+0130": "23:45",
        ])
    }

    func testGitHelpersReturnsInfo() {
        let info = GitFileInfo(url: URL(fileURLWithPath: #file))
        XCTAssertNotNil(info?.authorName)
        XCTAssertNotNil(info?.authorEmail)
        XCTAssertNotNil(info?.creationDate)
    }

    func testFileHeaderRuleThrowsIfCreationDateUnavailable() {
        let input = "let foo = bar"
        let options = FormatOptions(fileHeader: "// Created by Nick Lockwood on {created}.", fileInfo: FileInfo())
        XCTAssertThrowsError(try format(input, rules: [FormatRules.fileHeader], options: options))
    }

    func testFileHeaderFileReplacement() {
        let input = "let foo = bar"
        let output = "// MyFile.swift\n\nlet foo = bar"
        let fileInfo = FileInfo(filePath: "~/MyFile.swift")
        let options = FormatOptions(fileHeader: "// {file}", fileInfo: fileInfo)
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderRuleThrowsIfFileNameUnavailable() {
        let input = "let foo = bar"
        let options = FormatOptions(fileHeader: "// {file}.", fileInfo: FileInfo())
        XCTAssertThrowsError(try format(input, rules: [FormatRules.fileHeader], options: options))
    }

    func testEdgeCaseHeaderEndIndexPlusNewHeaderTokensCountEqualsFileTokensEndIndex() {
        let input = "// Header comment\n\nclass Foo {}"
        let output = "// Header line1\n// Header line2\n\nclass Foo {}"
        let options = FormatOptions(fileHeader: "// Header line1\n// Header line2")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderBlankLineNotRemovedBeforeFollowingComment() {
        let input = """
        //
        // Header
        //

        // Something else...
        """
        let options = FormatOptions(fileHeader: "//\n// Header\n//")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderBlankLineNotRemovedBeforeFollowingComment2() {
        let input = """
        //
        // Header
        //

        //
        // Something else...
        //
        """
        let options = FormatOptions(fileHeader: "//\n// Header\n//")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderRemovedAfterHashbang() {
        let input = """
        #!/usr/bin/swift

        // Header line1
        // Header line2

        let foo = 5
        """
        let output = """
        #!/usr/bin/swift

        let foo = 5
        """
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testFileHeaderPlacedAfterHashbang() {
        let input = """
        #!/usr/bin/swift

        let foo = 5
        """
        let output = """
        #!/usr/bin/swift

        // Header line1
        // Header line2

        let foo = 5
        """
        let options = FormatOptions(fileHeader: "// Header line1\n// Header line2")
        testFormatting(for: input, output, rule: FormatRules.fileHeader, options: options)
    }

    func testBlankLineAfterHashbangNotRemovedByFileHeader() {
        let input = """
        #!/usr/bin/swift

        let foo = 5
        """
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testLineAfterHashbangNotAffectedByFileHeaderRemoval() {
        let input = """
        #!/usr/bin/swift
        let foo = 5
        """
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testDisableFileHeaderCommentRespectedAfterHashbang() {
        let input = """
        #!/usr/bin/swift
        // swiftformat:disable fileHeader

        // Header line1
        // Header line2

        let foo = 5
        """
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    func testDisableFileHeaderCommentRespectedAfterHashbang2() {
        let input = """
        #!/usr/bin/swift

        // swiftformat:disable fileHeader
        // Header line1
        // Header line2

        let foo = 5
        """
        let options = FormatOptions(fileHeader: "")
        testFormatting(for: input, rule: FormatRules.fileHeader, options: options)
    }

    // MARK: - headerFileName

    func testHeaderFileNameReplaced() {
        let input = """
        // MyFile.swift

        let foo = bar
        """
        let output = """
        // YourFile.swift

        let foo = bar
        """
        let options = FormatOptions(fileInfo: FileInfo(filePath: "~/YourFile.swift"))
        testFormatting(for: input, output, rule: FormatRules.headerFileName, options: options)
    }

    // MARK: - strongOutlets

    func testRemoveWeakFromOutlet() {
        let input = "@IBOutlet weak var label: UILabel!"
        let output = "@IBOutlet var label: UILabel!"
        testFormatting(for: input, output, rule: FormatRules.strongOutlets)
    }

    func testRemoveWeakFromPrivateOutlet() {
        let input = "@IBOutlet private weak var label: UILabel!"
        let output = "@IBOutlet private var label: UILabel!"
        testFormatting(for: input, output, rule: FormatRules.strongOutlets)
    }

    func testRemoveWeakFromOutletOnSplitLine() {
        let input = "@IBOutlet\nweak var label: UILabel!"
        let output = "@IBOutlet\nvar label: UILabel!"
        testFormatting(for: input, output, rule: FormatRules.strongOutlets)
    }

    func testNoRemoveWeakFromNonOutlet() {
        let input = "weak var label: UILabel!"
        testFormatting(for: input, rule: FormatRules.strongOutlets)
    }

    func testNoRemoveWeakFromNonOutletAfterOutlet() {
        let input = "@IBOutlet weak var label1: UILabel!\nweak var label2: UILabel!"
        let output = "@IBOutlet var label1: UILabel!\nweak var label2: UILabel!"
        testFormatting(for: input, output, rule: FormatRules.strongOutlets)
    }

    func testNoRemoveWeakFromDelegateOutlet() {
        let input = "@IBOutlet weak var delegate: UITableViewDelegate?"
        testFormatting(for: input, rule: FormatRules.strongOutlets)
    }

    func testNoRemoveWeakFromDataSourceOutlet() {
        let input = "@IBOutlet weak var dataSource: UITableViewDataSource?"
        testFormatting(for: input, rule: FormatRules.strongOutlets)
    }

    func testRemoveWeakFromOutletAfterDelegateOutlet() {
        let input = "@IBOutlet weak var delegate: UITableViewDelegate?\n@IBOutlet weak var label1: UILabel!"
        let output = "@IBOutlet weak var delegate: UITableViewDelegate?\n@IBOutlet var label1: UILabel!"
        testFormatting(for: input, output, rule: FormatRules.strongOutlets)
    }

    func testRemoveWeakFromOutletAfterDataSourceOutlet() {
        let input = "@IBOutlet weak var dataSource: UITableViewDataSource?\n@IBOutlet weak var label1: UILabel!"
        let output = "@IBOutlet weak var dataSource: UITableViewDataSource?\n@IBOutlet var label1: UILabel!"
        testFormatting(for: input, output, rule: FormatRules.strongOutlets)
    }

    // MARK: - strongifiedSelf

    func testBacktickedSelfConvertedToSelfInGuard() {
        let input = """
        { [weak self] in
            guard let `self` = self else { return }
        }
        """
        let output = """
        { [weak self] in
            guard let self = self else { return }
        }
        """
        let options = FormatOptions(swiftVersion: "4.2")
        testFormatting(for: input, output, rule: FormatRules.strongifiedSelf, options: options,
                       exclude: ["wrapConditionalBodies"])
    }

    func testBacktickedSelfConvertedToSelfInIf() {
        let input = """
        { [weak self] in
            if let `self` = self else { print(self) }
        }
        """
        let output = """
        { [weak self] in
            if let self = self else { print(self) }
        }
        """
        let options = FormatOptions(swiftVersion: "4.2")
        testFormatting(for: input, output, rule: FormatRules.strongifiedSelf, options: options,
                       exclude: ["wrapConditionalBodies"])
    }

    func testBacktickedSelfNotConvertedIfVersionLessThan4_2() {
        let input = """
        { [weak self] in
            guard let `self` = self else { return }
        }
        """
        let options = FormatOptions(swiftVersion: "4.1.5")
        testFormatting(for: input, rule: FormatRules.strongifiedSelf, options: options,
                       exclude: ["wrapConditionalBodies"])
    }

    func testBacktickedSelfNotConvertedIfVersionUnspecified() {
        let input = """
        { [weak self] in
            guard let `self` = self else { return }
        }
        """
        testFormatting(for: input, rule: FormatRules.strongifiedSelf,
                       exclude: ["wrapConditionalBodies"])
    }

    func testBacktickedSelfNotConvertedIfNotConditional() {
        let input = "nonisolated(unsafe) let `self` = self"
        let options = FormatOptions(swiftVersion: "4.2")
        testFormatting(for: input, rule: FormatRules.strongifiedSelf, options: options)
    }

    // MARK: - yodaConditions

    func testNumericLiteralEqualYodaCondition() {
        let input = "5 == foo"
        let output = "foo == 5"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testNumericLiteralGreaterYodaCondition() {
        let input = "5.1 > foo"
        let output = "foo < 5.1"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testStringLiteralNotEqualYodaCondition() {
        let input = "\"foo\" != foo"
        let output = "foo != \"foo\""
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testNilNotEqualYodaCondition() {
        let input = "nil != foo"
        let output = "foo != nil"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testTrueNotEqualYodaCondition() {
        let input = "true != foo"
        let output = "foo != true"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testEnumCaseNotEqualYodaCondition() {
        let input = ".foo != foo"
        let output = "foo != .foo"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testArrayLiteralNotEqualYodaCondition() {
        let input = "[5, 6] != foo"
        let output = "foo != [5, 6]"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testNestedArrayLiteralNotEqualYodaCondition() {
        let input = "[5, [6, 7]] != foo"
        let output = "foo != [5, [6, 7]]"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testDictionaryLiteralNotEqualYodaCondition() {
        let input = "[foo: 5, bar: 6] != foo"
        let output = "foo != [foo: 5, bar: 6]"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testSubscriptNotTreatedAsYodaCondition() {
        let input = "foo[5] != bar"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testSubscriptOfParenthesizedExpressionNotTreatedAsYodaCondition() {
        let input = "(foo + bar)[5] != baz"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testSubscriptOfUnwrappedValueNotTreatedAsYodaCondition() {
        let input = "foo![5] != bar"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testSubscriptOfExpressionWithInlineCommentNotTreatedAsYodaCondition() {
        let input = "foo /* foo */ [5] != bar"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testSubscriptOfCollectionNotTreatedAsYodaCondition() {
        let input = "[foo][5] != bar"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testSubscriptOfTrailingClosureNotTreatedAsYodaCondition() {
        let input = "foo { [5] }[0] != bar"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testSubscriptOfRhsNotMangledInYodaCondition() {
        let input = "[1] == foo[0]"
        let output = "foo[0] == [1]"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testTupleYodaCondition() {
        let input = "(5, 6) != bar"
        let output = "bar != (5, 6)"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testLabeledTupleYodaCondition() {
        let input = "(foo: 5, bar: 6) != baz"
        let output = "baz != (foo: 5, bar: 6)"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testNestedTupleYodaCondition() {
        let input = "(5, (6, 7)) != baz"
        let output = "baz != (5, (6, 7))"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testFunctionCallNotTreatedAsYodaCondition() {
        let input = "foo(5) != bar"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testCallOfParenthesizedExpressionNotTreatedAsYodaCondition() {
        let input = "(foo + bar)(5) != baz"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testCallOfUnwrappedValueNotTreatedAsYodaCondition() {
        let input = "foo!(5) != bar"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testCallOfExpressionWithInlineCommentNotTreatedAsYodaCondition() {
        let input = "foo /* foo */ (5) != bar"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testCallOfRhsNotMangledInYodaCondition() {
        let input = "(1, 2) == foo(0)"
        let output = "foo(0) == (1, 2)"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testTrailingClosureOnRhsNotMangledInYodaCondition() {
        let input = "(1, 2) == foo { $0 }"
        let output = "foo { $0 } == (1, 2)"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testYodaConditionInIfStatement() {
        let input = "if 5 != foo {}"
        let output = "if foo != 5 {}"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testSubscriptYodaConditionInIfStatementWithBraceOnNextLine() {
        let input = "if [0] == foo.bar[0]\n{ baz() }"
        let output = "if foo.bar[0] == [0]\n{ baz() }"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions,
                       exclude: ["wrapConditionalBodies"])
    }

    func testYodaConditionInSecondClauseOfIfStatement() {
        let input = "if foo, 5 != bar {}"
        let output = "if foo, bar != 5 {}"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testYodaConditionInExpression() {
        let input = "let foo = 5 < bar\nbaz()"
        let output = "let foo = bar > 5\nbaz()"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testYodaConditionInExpressionWithTrailingClosure() {
        let input = "let foo = 5 < bar { baz() }"
        let output = "let foo = bar { baz() } > 5"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testYodaConditionInFunctionCall() {
        let input = "foo(5 < bar)"
        let output = "foo(bar > 5)"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testYodaConditionFollowedByExpression() {
        let input = "5 == foo + 6"
        let output = "foo + 6 == 5"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testPrefixExpressionYodaCondition() {
        let input = "!false == foo"
        let output = "foo == !false"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testPrefixExpressionYodaCondition2() {
        let input = "true == !foo"
        let output = "!foo == true"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testPostfixExpressionYodaCondition() {
        let input = "5<*> == foo"
        let output = "foo == 5<*>"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testDoublePostfixExpressionYodaCondition() {
        let input = "5!! == foo"
        let output = "foo == 5!!"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testPostfixExpressionNonYodaCondition() {
        let input = "5 == 5<*>"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testPostfixExpressionNonYodaCondition2() {
        let input = "5<*> == 5"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testStringEqualsStringNonYodaCondition() {
        let input = "\"foo\" == \"bar\""
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testConstantAfterNullCoalescingNonYodaCondition() {
        let input = "foo.last ?? -1 < bar"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testNoMangleYodaConditionFollowedByAndOperator() {
        let input = "5 <= foo && foo <= 7"
        let output = "foo >= 5 && foo <= 7"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testNoMangleYodaConditionFollowedByOrOperator() {
        let input = "5 <= foo || foo <= 7"
        let output = "foo >= 5 || foo <= 7"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testNoMangleYodaConditionFollowedByParentheses() {
        let input = "0 <= (foo + bar)"
        let output = "(foo + bar) >= 0"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testNoMangleYodaConditionInTernary() {
        let input = "let z = 0 < y ? 3 : 4"
        let output = "let z = y > 0 ? 3 : 4"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testNoMangleYodaConditionInTernary2() {
        let input = "let z = y > 0 ? 0 < x : 4"
        let output = "let z = y > 0 ? x > 0 : 4"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testNoMangleYodaConditionInTernary3() {
        let input = "let z = y > 0 ? 3 : 0 < x"
        let output = "let z = y > 0 ? 3 : x > 0"
        testFormatting(for: input, output, rule: FormatRules.yodaConditions)
    }

    func testKeyPathNotMangledAndNotTreatedAsYodaCondition() {
        let input = "\\.foo == bar"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    func testEnumCaseLessThanEnumCase() {
        let input = "XCTAssertFalse(.never < .never)"
        testFormatting(for: input, rule: FormatRules.yodaConditions)
    }

    // yodaSwap = literalsOnly

    func testNoSwapYodaDotMember() {
        let input = "foo(where: .bar == baz)"
        let options = FormatOptions(yodaSwap: .literalsOnly)
        testFormatting(for: input, rule: FormatRules.yodaConditions, options: options)
    }

    // MARK: - leadingDelimiters

    func testLeadingCommaMovedToPreviousLine() {
        let input = """
        let foo = 5
            , bar = 6
        """
        let output = """
        let foo = 5,
            bar = 6
        """
        testFormatting(for: input, output, rule: FormatRules.leadingDelimiters)
    }

    func testLeadingColonFollowedByCommentMovedToPreviousLine() {
        let input = """
        let foo
            : /* string */ String
        """
        let output = """
        let foo:
            /* string */ String
        """
        testFormatting(for: input, output, rule: FormatRules.leadingDelimiters)
    }

    func testCommaMovedBeforeCommentIfLineEndsInComment() {
        let input = """
        let foo = 5 // first
            , bar = 6
        """
        let output = """
        let foo = 5, // first
            bar = 6
        """
        testFormatting(for: input, output, rule: FormatRules.leadingDelimiters)
    }
}
