// $Id: utility.cc 4207 2022-08-26 04:36:28Z peter $

/*
	Copyright (C) 2005, 2006 Jari Häkkinen, Markus Ringnér
	Copyright (C) 2007, 2008 Jari Häkkinen, Peter Johansson
	Copyright (C) 2010, 2012, 2013, 2014, 2017, 2018, 2019, 2020, 2021 Peter Johansson

	This file is part of the yat library, http://dev.thep.lu.se/yat

	The yat library is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License as
	published by the Free Software Foundation; either version 3 of the
	License, or (at your option) any later version.

	The yat library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with yat. If not, see <http://www.gnu.org/licenses/>.
*/

#include <config.h>

#include "utility.h"

#include "Exception.h"
#include "FileUtil.h"
#include "stl_utility.h"

#include <cassert>
#include <cerrno>
#include <fnmatch.h>
#include <fstream>
#include <sstream>
#include <string>
#include <sys/stat.h>

namespace theplu {
namespace yat {
namespace utility {

	std::string basename(const std::string& path)
	{
		if (path.empty())
			return ".";
		if (path.size()==1)
			return path;
		size_t pos = path.find_last_of('/');
		if (pos==std::string::npos)
			return path;
		if (pos==path.size()-1)
			return basename(path.substr(0, path.size()-1));
		return path.substr(pos+1);
	}


	void chdir(const std::string& dir)
	{
		if (::chdir(dir.c_str()) )
			throw errno_error("chdir: '" + dir + "': ");
	}


	void chmod(const std::string& filename, mode_t mode)
	{
		if (::chmod(filename.c_str(), mode))
			throw errno_error("chmod: '" + filename + "': ");
	}


	void copy_file(const std::string& source, const std::string& target)
	{
		std::ifstream is;
		std::ofstream os;

		is.open(source.c_str(), std::ios::in | std::ios::binary);
		os.open(target.c_str(), std::ios::out | std::ios::binary);
		os << is.rdbuf();
		if (!os.good()) {
			std::ostringstream ss;
			ss << "copy_file failed writing to '" << target << "'\n";
			throw runtime_error(ss.str());
		}
		if (is.bad() || is.fail()) {
			std::ostringstream ss;
			ss << "copy_file failed reading from '" << source << "'\n";
			throw runtime_error(ss.str());
		}
		is.close();
		os.close();

		// copy permissions
		struct stat nodestat;
		stat(source.c_str(), &nodestat);
		chmod(target.c_str(), nodestat.st_mode);
	}


	std::string dirname(const std::string& path)
	{
		if (path.size()<=1) {
			if (path=="/")
				return path;
			else
				return ".";
		}
		assert(path.size()>=2);
		size_t pos = path.find_last_of('/', path.size()-2);
		if (pos==std::string::npos)
			return ".";
		if (pos==0)
			return "/";
		return path.substr(0,pos);
	}


	bool fnmatch(const std::string& pattern, const std::string& str, int flags)
	{
		int res = ::fnmatch(pattern.c_str(), str.c_str(), flags);
		if (res==0)
			return true;
		if (res!=FNM_NOMATCH) {
			std::stringstream ss;
			ss << "fnmatch(" << pattern << ", " << str << ")";
			throw runtime_error(ss.str());
		}
		return false;
	}


	std::string getcwd(void)
	{
		char buffer[256];
		if (::getcwd(buffer, 256))
			return std::string(buffer);
		if (errno == ERANGE) {
			char buffer2[4096];
			if (::getcwd(buffer2, 4096))
				return std::string(buffer2);
		}
		throw errno_error("getcwd failed: ");
		return "";
	}


	bool is_equal(std::string s, std::string other)
	{
		std::stringstream ss(s);
		std::string s2;
		ss >> s2; // to trim surrounding whitespaces
		to_lower(s2);
		if (s2!=other)
			return false;
		// Check that nothing is left on stream
		std::string s3;
		ss >> s3;
		return s3.empty();
	}


	bool is_nan(const std::string& s)
	{
		return is_equal(s, "nan");
	}


	void mkdir(const std::string& dir, mode_t mode)
	{
		int code = ::mkdir(dir.c_str(), mode);
		if (code){
			std::stringstream ss;
			ss << "mkdir: '" << dir << "': ";
			throw errno_error(ss.str());
		}
	}


	void mkdir_p(const std::string& dir, mode_t mode)
	{
		if (FileUtil(dir).exists())
			return;
		mkdir_p(dirname(dir), mode);
		mkdir(dir, mode);
	}


	void print_what(const std::exception& error, std::ostream& out)
	{
		// implemented as example at
		// http://www.cplusplus.com/reference/exception/rethrow_if_nested/
		out << error.what() << "\n";
		try {
			std::rethrow_if_nested(error);
		}
		catch (const std::exception& nested) {
			print_what(nested, out);
		}
	}


	void remove(const std::string& fn)
	{
		if (::remove(fn.c_str())) {
			std::string msg("remove: '");
			msg += fn;
			msg += "': ";
			throw errno_error(msg);
		}
	}


	void rename(const std::string& from, const std::string& to)
	{
		if (::rename(from.c_str(), to.c_str())) {
			std::stringstream ss;
			ss << "rename '" << from << "' to '" << to << "': ";
			throw errno_error(ss.str());
		}
	}


	void symlink(const std::string& path1, const std::string& path2)
	{
		if (::symlink(path1.c_str(), path2.c_str())) {
			std::string msg("symlink: '");
			msg += path1;
			msg += "' -> '";
			msg += path2;
			msg += "': ";
			throw errno_error(msg);
		}
	}


}}} // end of namespace utility, yat and thep
