#include <stan/math/prim.hpp>
#include <gtest/gtest.h>
#include <limits>

TEST(MathFunctions, inv_Phi) {
  using stan::math::inv_Phi;
  using stan::math::Phi;
  EXPECT_FLOAT_EQ(0.0, inv_Phi(0.5));
  double p = 0.123456789;
  EXPECT_FLOAT_EQ(p, Phi(inv_Phi(p)));
  p = 8e-311;
  EXPECT_FLOAT_EQ(p, Phi(inv_Phi(p)));
  p = 0.99;
  EXPECT_FLOAT_EQ(p, Phi(inv_Phi(p)));

  // breakpoints
  p = 0.02425;
  EXPECT_FLOAT_EQ(p, Phi(inv_Phi(p)));
  p = 0.97575;
  EXPECT_FLOAT_EQ(p, Phi(inv_Phi(p)));
}

TEST(MathFunctions, Equal) {
  using stan::math::inv_Phi;
  // test output generated with WolframAlpha

  double p[] = {0.000000001, 0.0000001, 0.00001,   0.001,      0.05, 0.15, 0.25,
                0.35,        0.45,      0.55,      0.65,       0.75, 0.85, 0.95,
                0.999,       0.99999,   0.9999999, 0.999999999};

  double exact[]
      = {-5.9978070150076868715623102049115374195951202210145432633059905935,
         -5.199337582192816931587347266962336866509737160238716454318950531,
         -4.264890793922824628498524698906344629356053222695490726201050806,
         -3.090232306167813541540399830107379205491008491865808855697171108,
         -1.644853626951472714863848907991632136083195744275322071769672094,
         -1.036433389493789579713244074673503366134740595985917627904454866,
         -0.674489750196081743202227014541307185386904415049861895662093788,
         -0.385320466407567623810762390270473572721091620003945341485947546,
         -0.125661346855074034210184388300799303397350646690021834224555394,
         0.1256613468550740342101843883007993033973506466900218342245553948,
         0.3853204664075676238107623902704735727210916200039453414859475461,
         0.6744897501960817432022270145413071853869044150498618956620937885,
         1.0364333894937895797132440746735033661347405959859176279044548663,
         1.6448536269514727148638489079916321360831957442753220717696720944,
         3.0902323061678135415403998301073792054910084918658088556971711085,
         4.2648907939228246284985246989063446293560532226954907262010508062,
         5.1993375821928169315873472669623368665097371602387164543189505310,
         5.9978070150076868715623102049115374195951202210145432633059905935};

  int numValues = sizeof(p) / sizeof(double);

  for (int i = 0; i < numValues; ++i) {
    EXPECT_NEAR(exact[i], inv_Phi(p[i]), 1.5e-15);
  }
}

TEST(MathFunctions, inv_Phi_inf) {
  using stan::math::inv_Phi;
  double p = 7e-311;
  const double inf = std::numeric_limits<double>::infinity();
  EXPECT_EQ(inv_Phi(p), -inf);
  p = 1.0;
  EXPECT_EQ(inv_Phi(p), inf);
}
TEST(MathFunctions, inv_Phi_nan) {
  using stan::math::inv_Phi;
  double nan = std::numeric_limits<double>::quiet_NaN();
  EXPECT_THROW(inv_Phi(nan), std::domain_error);
  EXPECT_THROW(inv_Phi(-2.0), std::domain_error);
  EXPECT_THROW(inv_Phi(2.0), std::domain_error);
}
