// Noise shaper coefficients for SSRC, written by Naoki Shibata ( shibatch@users.sourceforge.net )

const int freqs[] = {
  8000, 11025, 16000,
  22050, 32000, 37800, 44056,
  44100, 47250, 48000, 50000,
  50400, 88200, 96000, 176400,
  192000, 352800, 2822400, 5644800,
  11289600, 22579200, -1
};

#define LIBDITHER_TYPE_SIMPLE 0
#define LIBDITHER_TYPE_OLD 1
#define LIBDITHER_TYPE_ATH_A 2
#define LIBDITHER_TYPE_ATH_B 3

#define SHAPERLENMAX 64

typedef struct DitherCoef {
  int fs, id;

  int type;
  const char *friendlyName;

  int len;
  const double coefs[SHAPERLENMAX];
} DitherCoef;

const DitherCoef ditherCoef[] = {
  {
    44100, 0,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 0",
    12,
    {
      -0.59543782472610473633,
      0.0025078731123358011246,
      0.18518058955669403076,
      0.0010374293196946382523,
      0.1036634296178817749,
      0.053248628973960876465,
      8.4030049038119614124e-05,
      3.8569933025200953125e-08,
      0.026413010433316230774,
      0.00068438396556302905083,
      -3.1580505037709372118e-06,
      -0.031739629805088043213,
    },
  },

  {
    44100, 1,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 1",
    12,
    {
      -0.9982020258903503418,
      0.59951537847518920898,
      -0.081278324127197265625,
      -9.2977395979687571526e-05,
      0.20252060890197753906,
      0.024194804951548576355,
      -0.00090227433247491717339,
      0.045577544718980789185,
      0.044477824121713638306,
      0.0030681777279824018478,
      0.00016936450265347957611,
      6.8561035959646687843e-07,
    },
  },

  {
    44100, 2,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 2",
    24,
    {
      -1.356863856315612793,
      1.2252935171127319336,
      -0.62355506420135498047,
      0.22562094032764434814,
      0.23557975888252258301,
      -0.13536362349987030029,
      0.091538146138191223145,
      0.056445639580488204956,
      -3.9614424167666584253e-05,
      0.023561919108033180237,
      0.010756319388747215271,
      0.00031949131516739726067,
      -0.0014337620232254266739,
      0.0084551237523555755615,
      0.00021318180370144546032,
      -7.6175922004040330648e-05,
      -0.0010102330707013607025,
      -4.5030275941826403141e-05,
      -0.0013433821732178330421,
      -0.0013937242329120635986,
      -0.00043306700536049902439,
      -0.00046949787065386772156,
      -0.00014775841555092483759,
      4.1060175135498866439e-05,
    },
  },

  {
    44100, 3,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 3",
    16,
    {
      -1.7714835405349731445,
      2.160381317138671875,
      -1.8512212038040161133,
      1.3459416627883911133,
      -0.52356457710266113281,
      0.1598011404275894165,
      0.079563401639461517334,
      -0.017584055662155151367,
      0.039745520800352096558,
      0.021822927519679069519,
      0.0072338962927460670471,
      0.00083879311569035053253,
      0.0094792330637574195862,
      0.0068564494140446186066,
      -0.00039525484316982328892,
      0.0040870164521038532257,
    },
  },

  {
    44100, 4,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 4",
    20,
    {
      -2.1551730632781982422,
      3.1482026576995849609,
      -3.4208805561065673828,
      3.1343655586242675781,
      -2.1552324295043945312,
      1.2698539495468139648,
      -0.50336593389511108398,
      0.16464470326900482178,
      0.013838037848472595215,
      0.0062505742534995079041,
      -0.0041691507212817668915,
      0.013679158873856067657,
      0.0024516219273209571838,
      -0.00024407464661635458469,
      0.0052452399395406246185,
      0.00042019927059300243855,
      -0.0004135201452299952507,
      -0.00016322935698553919792,
      0.00047321102465502917767,
      -0.00093277986161410808563,
    },
  },

  {
    44100, 5,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 5",
    16,
    {
      -2.5096075534820556641,
      4.2519822120666503906,
      -5.4792313575744628906,
      5.97249603271484375,
      -5.294708251953125,
      4.0664181709289550781,
      -2.5247132778167724609,
      1.3039399385452270508,
      -0.44613662362098693848,
      0.097044020891189575195,
      0.016150593757629394531,
      0.0060916151851415634155,
      -0.01326625235378742218,
      0.017414959147572517395,
      -0.00079903466394171118736,
      -7.114160212040587794e-07,
    },
  },

  {
    44100, 6,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 6",
    20,
    {
      -2.8263266086578369141,
      5.3534359931945800781,
      -7.8042058944702148438,
      9.6793689727783203125,
      -10.157135009765625,
      9.4399957656860351562,
      -7.614612579345703125,
      5.4245176315307617188,
      -3.2478282451629638672,
      1.630185246467590332,
      -0.58538019657135009766,
      0.11710002273321151733,
      0.033543668687343597412,
      -0.0088841468095779418945,
      -0.017314357683062553406,
      0.033262729644775390625,
      -0.018168220296502113342,
      0.0068015027791261672974,
      0.00096911948639899492264,
      -0.00096489343559369444847,
    },
  },

  //

  {
    44100, 10,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 0",
    12,
    {
      -0.56017816066741943359,
      0.00032013398595154285431,
      0.28353634476661682129,
      2.0154238882241770625e-05,
      2.8024095399814541452e-07,
      0.0013137641362845897675,
      0.13863940536975860596,
      -4.6821247678963118233e-07,
      4.3743430069298483431e-06,
      0.0040733972564339637756,
      0.00017294117424171417952,
      -0.011740575544536113739,
    },
  },

  {
    44100, 11,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 1",
    11,
    {
      -1.0638480186462402344,
      0.66863000392913818359,
      2.2443387024395633489e-06,
      -0.011665554717183113098,
      -1.3796089660900179297e-05,
      0.15546366572380065918,
      0.034445952624082565308,
      0.0051354211755096912384,
      0.04103728756308555603,
      0.019456973299384117126,
      0.00030115628032945096493,
      0,
    },
  },

  {
    44100, 12,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 2",
    20,
    {
      -1.5528930425643920898,
      1.4595929384231567383,
      -0.55787962675094604492,
      -0.0042479122057557106018,
      0.24744692444801330566,
      0.010727410204708576202,
      -0.024524994194507598877,
      0.084559261798858642578,
      0.021680492907762527466,
      0.0078514227643609046936,
      0.010477361269295215607,
      0.0061291921883821487427,
      0.0056792846880853176117,
      0.0011211306555196642876,
      0.0014330057892948389053,
      0.00097884633578360080719,
      0.00021456897957250475883,
      -0.00010452028800500556827,
      -0.0010217631934210658073,
      -0.0014363396912813186646,
    },
  },

  {
    44100, 13,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 3",
    20,
    {
      -2.0674252510070800781,
      2.6145982742309570312,
      -1.9563752412796020508,
      0.92442721128463745117,
      0.0049272147007286548615,
      -0.18960046768188476562,
      0.16765585541725158691,
      0.020780716091394424438,
      0.00021837331587448716164,
      0.013521989807486534119,
      0.019153339788317680359,
      0.0059661092236638069153,
      0.0014546891907230019569,
      0.0061949742957949638367,
      0.0024373924825340509415,
      0.0032744421623647212982,
      0.00016406788199674338102,
      0.00043720021494664251804,
      0.00068944186205044388771,
      -0.00069793453440070152283,
    },
  },

  {
    44100, 14,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 4",
    23,
    {
      -2.5422441959381103516,
      3.9492623805999755859,
      -4.082462310791015625,
      3.1288163661956787109,
      -1.5902516841888427734,
      0.52063500881195068359,
      0.034919787198305130005,
      -0.054061867296695709229,
      0.038279779255390167236,
      0.0070470790378749370575,
      0.022740921005606651306,
      -0.0088992407545447349548,
      0.0074287117458879947662,
      0.0079405372962355613708,
      0.00075413304148241877556,
      0.00014326006930787116289,
      0.0031513145659118890762,
      0.00010745167674031108618,
      -9.9756995041389018297e-05,
      -0.0002400560770183801651,
      -0.00022666374570690095425,
      6.6967804741580039263e-06,
      -0.00028052827110514044762,
      0,
    },
  },

  {
    44100, 15,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 5",
    20,
    {
      -3.0226423740386962891,
      5.5270595550537109375,
      -7.0363101959228515625,
      6.8852219581604003906,
      -5.1172428131103515625,
      2.9304540157318115234,
      -1.1258685588836669922,
      0.25490742921829223633,
      0.019723705947399139404,
      0.015306896530091762543,
      -0.023763004690408706665,
      0.015180897898972034454,
      0.013075054623186588287,
      -0.0064899893477559089661,
      0.0024698353372514247894,
      0.0042672571726143360138,
      -1.1534318673511734232e-06,
      -7.4574086283973883837e-06,
      5.058841452409978956e-07,
      8.1296411735820583999e-06,
    },
  },

  {
    44100, 16,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 6",
    24,
    {
      -3.5288703441619873047,
      7.3978977203369140625,
      -11.062552452087402344,
      12.915676116943359375,
      -12.005429267883300781,
      8.9765348434448242188,
      -5.1754517555236816406,
      2.1034460067749023438,
      -0.34860187768936157227,
      -0.21167574822902679443,
      0.17974011600017547607,
      -0.030822610482573509216,
      -0.024646494537591934204,
      0.010657157748937606812,
      0.0079134432598948478699,
      -0.0043759951367974281311,
      -0.0048155616968870162964,
      0.0055646821856498718262,
      -0.0013907034881412982941,
      -0.0022238297387957572937,
      0.00068998814094811677933,
      0.00026916823117062449455,
      -0.00075513543561100959778,
      0,
    },
  },

  //

  {
    44100, 90,
    LIBDITHER_TYPE_OLD, "ATH Curve A, Old Low",
    15,
    {-2.0833916664123535156,   3.0418450832366943359,  -3.2047898769378662109,   2.7571926116943359375,
     -1.4978630542755126953,   0.3427594602108001709,   0.71733748912811279297, -1.0737057924270629883,
     1.0225815773010253906,  -0.56649994850158691406,  0.20968692004680633545,  0.065378531813621520996,
     -0.10322438180446624756,  0.067442022264003753662, 0.00495197344571352005, 0,
    },
  },

  {
    44100, 91,
    LIBDITHER_TYPE_OLD, "ATH Curve A, Old Mid",
    20,
    {-2.6773197650909423828,   4.8308925628662109375,  -6.570110321044921875,    7.4572014808654785156,
     -6.7263274192810058594,   4.8481650352478027344,  -2.0412089824676513672,  -0.7006359100341796875,
     2.9537565708160400391,  -4.0800385475158691406,   4.1845216751098632812,  -3.3311812877655029297,
     2.1179926395416259766,  -0.879302978515625,       0.031759146600961685181, 0.42382788658142089844,
     -0.47882103919982910156,  0.35490813851356506348, -0.17496839165687561035,  0.060908168554306030273,
    },
  },

  {
    44100, 92,
    LIBDITHER_TYPE_OLD, "ATH Curve A, Old High",
    20,
    { -3.0259189605712890625,  6.0268716812133789062,  -9.195003509521484375,   11.824929237365722656,
      -12.767142295837402344,  11.917946815490722656,   -9.1739168167114257812,   5.3712320327758789062,
      -1.1393624544143676758, -2.4484779834747314453,   4.9719839096069335938,  -6.0392003059387207031,
      5.9359521865844726562, -4.903278350830078125,    3.5527443885803222656,  -2.1909697055816650391,
      1.1672389507293701172, -0.4903914332389831543,   0.16519790887832641602, -0.023217858746647834778,
    },
  },

  {
    44100, 98,
    LIBDITHER_TYPE_SIMPLE, "Simple triangular",
    1,
    {
      -1,
    },
  },

  {
    44100, 99,
    LIBDITHER_TYPE_SIMPLE, "No shaper",
    0,
    {
    },
  },

  //

  {
    48000, 0,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 0",
    16,
    {
      -0.64815437793731689453,
      0.00013292329094838351011,
      0.15284439921379089355,
      0.024795081466436386108,
      0.028879294171929359436,
      0.097741305828094482422,
      -3.7233345210552215576e-05,
      -3.0361816243384964764e-06,
      2.6851517759496346116e-05,
      0.015118855983018875122,
      0.00011908156011486425996,
      -4.0203917706094216555e-06,
      -0.03214230760931968689,
      -1.2108691862522391602e-06,
      0,
      -2.4130819564760486173e-09,
    },
  },

  {
    48000, 1,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 1",
    16,
    {
      -1.0375014543533325195,
      0.55585253238677978516,
      6.2009254179429262877e-05,
      -0.054276779294013977051,
      0.14064073562622070312,
      0.1073406711220741272,
      -6.7414471232041250914e-06,
      -0.00090507784625515341759,
      0.071966759860515594482,
      0.018717750906944274902,
      0.0038517464417964220047,
      0.0057432847097516059875,
      0.0011602794984355568886,
      0.00023562647402286529541,
      6.1770442698616534472e-05,
      0.0016767864581197500229,
    },
  },

  {
    48000, 2,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 2",
    16,
    {
      -1.4919577836990356445,
      1.3089178800582885742,
      -0.54051631689071655273,
      0.00036113749956712126732,
      0.36303195357322692871,
      -0.10911127924919128418,
      -0.0073106382042169570923,
      0.11545914411544799805,
      -0.00377228553406894207,
      0.012545258738100528717,
      0.02927248738706111908,
      0.0050022001378238201141,
      0.00020218851568643003702,
      0.004905734676867723465,
      0.0051279761828482151031,
      0.0025056710001081228256,
    },
  },

  {
    48000, 3,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 3",
    19,
    {
      -1.9601591825485229492,
      2.4060547351837158203,
      -1.9488849639892578125,
      1.1626639366149902344,
      -0.25297921895980834961,
      -0.031299483031034469604,
      0.11234971880912780762,
      0.028672901913523674011,
      -0.0084085874259471893311,
      0.040343299508094787598,
      0.014730192720890045166,
      0.0081526516005396842957,
      0.00078110158210620284081,
      0.010703166946768760681,
      0.007504582870751619339,
      -6.7899374698754400015e-05,
      0.0045952727086842060089,
      0.0015685434918850660324,
      8.0333913501817733049e-05,
      0,
    },
  },

  {
    48000, 4,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 4",
    28,
    {
      -2.4219727516174316406,
      3.6378045082092285156,
      -3.8756568431854248047,
      3.2019908428192138672,
      -1.8469271659851074219,
      0.76111799478530883789,
      -0.083762630820274353027,
      -0.064117647707462310791,
      0.066511705517768859863,
      0.011620322242379188538,
      0.00089677015785127878189,
      -0.0038908864371478557587,
      0.011067331768572330475,
      0.0016393634723499417305,
      -0.0021009922493249177933,
      0.0039737410843372344971,
      0.00064989842940121889114,
      -0.00064297980861738324165,
      -0.0010019573383033275604,
      0.00024940294679254293442,
      -0.00020461704116314649582,
      -0.0014896913198754191399,
      3.6964305763831362128e-05,
      -5.5592459830222651362e-05,
      -0.00022196055215317755938,
      -0.00011919181270059198141,
      0.00021784781711176037788,
      5.855178824276663363e-05,
    },
  },

  {
    48000, 5,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 5",
    20,
    {
      -2.8460333347320556641,
      5.0355429649353027344,
      -6.4927110671997070312,
      6.6689691543579101562,
      -5.3422427177429199219,
      3.4331059455871582031,
      -1.5913537740707397461,
      0.48210138082504272461,
      0.0084637729451060295105,
      -0.035323332995176315308,
      0.0055271289311349391937,
      0.021560266613960266113,
      0.0061011519283056259155,
      -0.0090660518035292625427,
      0.010759642347693443298,
      0.0046441229060292243958,
      -0.0028851281385868787766,
      0.0027118360158056020737,
      0.0008332781144417822361,
      -6.3722327467985451221e-05,
    },
  },

  {
    48000, 6,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 6",
    28,
    {
      -3.2601516246795654297,
      6.5575695037841796875,
      -9.74866485595703125,
      11.7130889892578125,
      -11.504628181457519531,
      9.4859628677368164062,
      -6.4042730331420898438,
      3.4772820472717285156,
      -1.3327382802963256836,
      0.26464575529098510742,
      0.081823304295539855957,
      -0.044643409550189971924,
      -0.021642472594976425171,
      0.04283212125301361084,
      -0.0033832620829343795776,
      -0.016050558537244796753,
      0.019443769007921218872,
      -0.0020140456035733222961,
      -0.0051018465310335159302,
      0.0049441442824900150299,
      0.001399693894200026989,
      -0.0035810119006782770157,
      0.0022099197376519441605,
      0.00010120005026692524552,
      -0.00077120866626501083374,
      4.7727549826959148049e-05,
      0.00047057875781320035458,
      -0.00053522014059126377106,
    },
  },

  //

  {
    48000, 10,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 0",
    12,
    {
      -0.6524372100830078125,
      6.076441422919742763e-06,
      0.23769626021385192871,
      0.000635539996437728405,
      1.2602968126884661615e-06,
      9.5054092525970190763e-06,
      0.052108168601989746094,
      0.051929716020822525024,
      1.5829242556719691493e-06,
      -7.0671262619725894183e-08,
      -8.8266028797079343349e-07,
      -0.0051737497560679912567,
    },
  },

  {
    48000, 11,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 1",
    11,
    {
      -1.133047938346862793,
      0.65767991542816162109,
      0.00032861577346920967102,
      -5.3428146884471061639e-07,
      0.00015824155707377940416,
      0.045498214662075042725,
      0.11496848613023757935,
      1.4669727534055709839e-05,
      2.1825994735991116613e-06,
      0.04351314157247543335,
      0.0047881226055324077606,
      0,
    },
  },

  {
    48000, 12,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 2",
    16,
    {
      -1.6628091335296630859,
      1.4971796274185180664,
      -0.53692084550857543945,
      -5.8460878790356218815e-05,
      0.11764197796583175659,
      0.087953954935073852539,
      0.00031151156872510910034,
      -0.0071666454896330833435,
      0.0624333210289478302,
      0.014062460511922836304,
      0.00068763544550165534019,
      0.0032515849452465772629,
      0.0086512034758925437927,
      0.00013604816922452300787,
      9.8600306955631822348e-05,
      0.00063699163729324936867,
    },
  },

  {
    48000, 13,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 3",
    16,
    {
      -2.2278966903686523438,
      2.7684540748596191406,
      -1.950432896614074707,
      0.76130187511444091797,
      0.059634834527969360352,
      -0.09099829941987991333,
      0.0021133834961801767349,
      0.089261904358863830566,
      0.0056450613774359226227,
      -0.0063776262104511260986,
      0.018127651885151863098,
      0.0075408210977911949158,
      0.0060678133741021156311,
      -0.00023856009647715836763,
      0.0022084014490246772766,
      0.0037000982556492090225,
    },
  },

  {
    48000, 14,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 4",
    20,
    {
      -2.7479245662689208984,
      4.2712616920471191406,
      -4.2730498313903808594,
      2.9785141944885253906,
      -1.2957186698913574219,
      0.31362664699554443359,
      0.029963159933686256409,
      0.023559728637337684631,
      -0.020922580733895301819,
      0.023302176967263221741,
      0.0055510243400931358337,
      0.0067555652931332588196,
      0.0011096433736383914948,
      0.00098128849640488624573,
      0.0034134739544242620468,
      0.0025581021327525377274,
      -0.00023128408065531402826,
      0.00079201703192666172981,
      0.00075990450568497180939,
      4.8069246076920535415e-06,
    },
  },

  {
    48000, 15,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 5",
    20,
    {
      -3.3172307014465332031,
      6.1978197097778320312,
      -7.7909684181213378906,
      7.1505522727966308594,
      -4.7267832756042480469,
      2.1657605171203613281,
      -0.50765365362167358398,
      -0.028446925804018974304,
      0.061983801424503326416,
      0.0052601275965571403503,
      0.0093981074169278144836,
      -0.015898862853646278381,
      0.010791713371872901917,
      0.0031401265878230333328,
      0.00092148158000782132149,
      -0.0030198376625776290894,
      0.0031266580335795879364,
      0.00065209472086280584335,
      -0.00054945144802331924438,
      -4.1805269574979320168e-05,
    },
  },

  {
    48000, 16,
    LIBDITHER_TYPE_ATH_B, "ATH Curve B, Intensity 6",
    16,
    {
      -3.8836567401885986328,
      8.4325723648071289062,
      -12.618600845336914062,
      14.162892341613769531,
      -12.145025253295898438,
      7.9046030044555664062,
      -3.6286561489105224609,
      0.95813679695129394531,
      0.019669502973556518555,
      -0.045369073748588562012,
      -0.078012384474277496338,
      0.083914883434772491455,
      -0.019228875637054443359,
      -0.011115396395325660706,
      0.0085456911474466323853,
      0,
    },
  },

  //

  {
    48000, 90,
    LIBDITHER_TYPE_OLD, "ATH Curve A, Old Low",
    16,
    {-2.3925774097442626953,   3.4350297451019287109,  -3.1853709220886230469,   1.8117271661758422852,
     0.20124770700931549072, -1.4759907722473144531,   1.7210904359817504883,  -0.97746700048446655273,
     0.13790138065814971924,  0.38185903429985046387, -0.27421241998672485352, -0.066584214568138122559,
     0.35223302245140075684, -0.37672343850135803223,  0.23964276909828186035, -0.068674825131893157959,
    }, /* 48k, N=16, amp=10 */
  },

  {
    48000, 91,
    LIBDITHER_TYPE_OLD, "ATH Curve A, Old Mid",
    16,
    {-2.8720729351043701172,   5.0413231849670410156,  -6.2442994117736816406,   5.8483986854553222656,
     -3.7067542076110839844,   1.0495119094848632812,   1.1830236911773681641,  -2.1126792430877685547,
     1.9094531536102294922,  -0.99913084506988525391,  0.17090806365013122559,  0.32615602016448974609,
     -0.39127644896507263184,  0.26876461505889892578, -0.097676105797290802002, 0.023473845794796943665,
    }, /* 48k, N=16, amp=18 */
  },

  {
    48000, 98,
    LIBDITHER_TYPE_SIMPLE, "Simple triangular",
    1,
    {
      -1,
    },
  },

  {
    48000, 99,
    LIBDITHER_TYPE_SIMPLE, "No shaper",
    0,
    {
    },
  },

  //

  {
    88200, 0,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 0",
    24,
    {
      -0.81275081634521484375,
      -1.3415416333373286761e-07,
      1.4003169781062752008e-05,
      0.027366658672690391541,
      0.06308479607105255127,
      0.00041124963900074362755,
      0.0014667811337858438492,
      0.0034636424388736486435,
      0.01444795168936252594,
      0.050686400383710861206,
      0.00031657953513786196709,
      7.6081778388470411301e-07,
      -1.3391935453910264187e-06,
      -1.1084978268627310172e-06,
      -2.3458991904590220656e-07,
      -7.1970474024851682771e-09,
      0.00024097530695144087076,
      0.00081339181633666157722,
      0.0027072627563029527664,
      1.2289029655221384019e-05,
      2.4080820821836823598e-06,
      -2.6516547677601920441e-06,
      -0.022208366543054580688,
      -1.8090953801674913848e-07,
    },
  },

  {
    88200, 1,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 1",
    32,
    {
      -1.1759521961212158203,
      0.004028123803436756134,
      0.47074413299560546875,
      0.00051633489783853292465,
      -0.034613370895385742188,
      -0.090879365801811218262,
      2.4943574317148886621e-05,
      0.040280379354953765869,
      0.084476247429847717285,
      0.020952062681317329407,
      -6.4247273257933557034e-05,
      -0.015425831079483032227,
      -0.00034846802009269595146,
      0.00021460390416905283928,
      0.038064550608396530151,
      0.0075228977948427200317,
      0.00010572007158771157265,
      0.00088893284555524587631,
      0.0051207961514592170715,
      0.0047091664746403694153,
      0.0013088451232761144638,
      0.0010616352083161473274,
      5.3146239224588498473e-05,
      2.692748421395663172e-05,
      -7.112140337994787842e-06,
      -3.7880614399909973145e-05,
      0.00015048061322886496782,
      0.0014544483274221420288,
      0.00033794986666180193424,
      0.00062911584973335266113,
      1.7671519358941623068e-08,
      -1.289341753363260068e-07,
    },
  },

  {
    88200, 2,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 2",
    20,
    {
      -2.0752036571502685547,
      1.4316110610961914062,
      4.1018622141564264894e-05,
      -0.30747786164283752441,
      -0.01503494754433631897,
      0.0020690073724836111069,
      0.095445446670055389404,
      0.017573365941643714905,
      -0.0015146844089031219482,
      -0.0097157200798392295837,
      -0.0032300157472491264343,
      0.0011662221513688564301,
      0.012702429667115211487,
      0.013680535368621349335,
      0.00032695711706764996052,
      0.00033481238642707467079,
      -0.0019418919691815972328,
      0.0065598445944488048553,
      0.0031848684884607791901,
      0.0011857076315209269524,
    },
  },

  {
    88200, 98, 
    LIBDITHER_TYPE_SIMPLE, "Simple triangular",
    1,
    {
      -1,
    },
  },

  {
    88200, 99,
    LIBDITHER_TYPE_SIMPLE, "No shaper",
    0,
    {
    },
  },

  //

  {
    96000, 0,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 0",
    32,
    {
      -0.83362782001495361328,
      -4.7663510827078425791e-07,
      5.5927204812178388238e-05,
      0.00091767607955262064934,
      0.085019297897815704346,
      0.00030864097061567008495,
      2.7474849048303440213e-05,
      3.4470554965082556009e-05,
      0.0068166172131896018982,
      0.0051032402552664279938,
      0.048310291022062301636,
      3.4194424642919329926e-06,
      3.9387387573697196785e-08,
      -5.2296832109277602285e-06,
      -2.1815125364810228348e-05,
      -5.8060527408088091761e-06,
      -8.8975330072571523488e-06,
      2.8793074307031929493e-06,
      1.0142302926396951079e-05,
      0.00088343484094366431236,
      6.6521701228339225054e-05,
      4.3032446228608023375e-07,
      -1.5573209566355217248e-06,
      -0.0032469024881720542908,
      -0.013371952809393405914,
      -0.0016697095707058906555,
      -0.00033745748805813491344,
      -3.8218466215766966343e-05,
      -8.0883961345534771681e-05,
      -1.7631093214731663465e-05,
      -4.7317589633166790009e-06,
      -3.8150733416841831058e-07,
    },
  },

  {
    96000, 1,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 1",
    24,
    {
      -1.2260272502899169922,
      0.0014651787932962179184,
      0.48510685563087463379,
      0.00016445807705167680979,
      -3.7136185710551217198e-05,
      -0.11480120569467544556,
      0.00045887456508353352547,
      0.0017981674754992127419,
      0.077026084065437316895,
      0.040429998189210891724,
      -4.6414548705797642469e-05,
      -0.00040040328167378902435,
      -0.00013407132064457982779,
      -0.003484065178781747818,
      0.00048559889546595513821,
      0.01950314640998840332,
      0.017391569912433624268,
      0.0019728131592273712158,
      8.8219474037032341585e-07,
      0.00064927618950605392456,
      0.0049143577925860881805,
      0.0023035951890051364899,
      0.00063752382993698120117,
      0.00076101027661934494972,
    },
  },

  {
    96000, 2,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 2",
    31,
    {
      -2.1041114330291748047,
      1.4101417064666748047,
      0.0035147387534379959106,
      -0.18617971241474151611,
      -0.11117676645517349243,
      0.0013629450695589184761,
      0.055446717888116836548,
      0.056859914213418960571,
      0.0039573232643306255341,
      -0.0025663347914814949036,
      -0.014090753160417079926,
      -0.0062257084064185619354,
      0.0065397350117564201355,
      0.019066527485847473145,
      0.0035695792175829410553,
      0.001226439489983022213,
      -0.00011440102389315143228,
      0.00019808727665804326534,
      0.0032306648790836334229,
      0.0046777799725532531738,
      0.0010407331865280866623,
      0.00097329093841835856438,
      0.00078034552279859781265,
      0.00038853226578794419765,
      -4.1947296267608180642e-05,
      -0.0001729554060148075223,
      0.00059315189719200134277,
      0.00069724785862490534782,
      0.00050402310444042086601,
      0.00037623706157319247723,
      0.00017440004739910364151,
      0,
    },
  },

  {
    96000, 98, 
    LIBDITHER_TYPE_SIMPLE, "Simple triangular",
    1,
    {
      -1,
    },
  },

  {
    96000, 99,
    LIBDITHER_TYPE_SIMPLE, "No shaper",
    0,
    {
    },
  },

  //

  {
    192000, 0,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 0",
    20,
    {
      -0.92986786365509033203,
      -2.3757004328217590228e-06,
      -1.3239204008641536348e-06,
      -4.5336445708699102397e-08,
      1.0855699201783863828e-06,
      7.5193946713625336997e-07,
      0.010574714280664920807,
      0.015397379174828529358,
      0.0071734646335244178772,
      0.0040416326373815536499,
      0.00031543619115836918354,
      6.079084869270445779e-06,
      2.5614752303226850927e-05,
      6.444113296311115846e-06,
      0.00014342019858304411173,
      9.9886632298762378923e-09,
      0.00011001565144397318363,
      0.00026444403920322656631,
      0.018070342019200325012,
      0.013997578062117099762,
    },
  },

  {
    192000, 1,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 1",
    44,
    {
      -1.3312897682189941406,
      -0.00043348263716325163841,
      0.0040898225270211696625,
      0.45905736088752746582,
      0.00030199048342183232307,
      -2.5365816327393986285e-05,
      0.00053758098511025309563,
      -0.00010294133971910923719,
      -0.12356997281312942505,
      3.46512497344519943e-05,
      -0.0011937879025936126709,
      0.000649263791274279356,
      0.0082960268482565879822,
      0.031590048223733901978,
      0.0019864565692842006683,
      0.0055887317284941673279,
      0.0046014585532248020172,
      0.0045781680382788181305,
      5.6712280638748779893e-05,
      -0.00011149531201226636767,
      -0.00028292578645050525665,
      -0.00034148831036873161793,
      -0.0033573799300938844681,
      -0.0023941406980156898499,
      0.00041498657083138823509,
      6.4982996264006942511e-05,
      0.00090611475752666592598,
      0.0047907871194183826447,
      0.0037444231566041707993,
      0.00016722116561140865088,
      0.0011758945183828473091,
      0.0008123705047182738781,
      1.1241873835388105363e-05,
      0.00011657441064016893506,
      0.00014767059474252164364,
      1.7531696357764303684e-05,
      0.0002746190002653747797,
      0.00073906377656385302544,
      0.00038272203528322279453,
      0.00068217515945434570312,
      0.0010574184125289320946,
      1.5615147503922344185e-06,
      2.1584497517324052751e-05,
      0,
    },
  },

  {
    192000, 2,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 2",
    54,
    {
      -2.1174826622009277344,
      0.79300129413604736328,
      0.5887165069580078125,
      0.0045170621015131473541,
      2.2400596208171918988e-05,
      -0.34981065988540649414,
      -0.0014674699632450938225,
      0.035286050289869308472,
      0.03057491593062877655,
      0.0080999247729778289795,
      0.024920884519815444946,
      0.010276389308273792267,
      0.002827338874340057373,
      -0.011965871788561344147,
      0.0011787357507273554802,
      -0.0015875700628384947777,
      -0.0012219551717862486839,
      -0.0041509792208671569824,
      -0.00023660375154577195644,
      0.00023469136795029044151,
      -0.00024544104235246777534,
      0.002350530354306101799,
      0.0010635280050337314606,
      0.0021934446413069963455,
      -0.00018601950432639569044,
      0.00053444207878783345222,
      0.0005648268270306289196,
      6.5553147578611969948e-05,
      -0.00050351343816146254539,
      -0.00069776934105902910233,
      -0.0002154307876480743289,
      -0.00055884284665808081627,
      0.00095591234276071190834,
      0.00018323963740840554237,
      0.001184734981507062912,
      -5.5957076256163418293e-05,
      0.00021092590759508311749,
      -9.2614163804682902992e-06,
      1.6893125575734302402e-05,
      0.00010291898797731846571,
      8.7052303570089861751e-06,
      2.1893838493269868195e-05,
      -2.0483348635025322437e-05,
      9.3148359155748039484e-05,
      5.4571984946960583329e-05,
      -1.0393147931608837098e-05,
      4.1864630475174635649e-05,
      -3.3142681786557659507e-05,
      -4.6412500864789763e-07,
      3.1690757168689742684e-05,
      -2.9199603886809200048e-05,
      4.1371429688297212124e-05,
      -3.0970045372669119388e-06,
      0.00013081972429063171148,
      0,
      0,
    },
  },

  {
    192000, 98, 
    LIBDITHER_TYPE_SIMPLE, "Simple triangular",
    1,
    {
      -1,
    },
  },

  {
    192000, 99,
    LIBDITHER_TYPE_SIMPLE, "No shaper",
    0,
    {
    },
  },

  //

  {
    8000, 0,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 0",
    8,
    {
      0.76159644126892089844,
      0.19409510493278503418,
      -0.035044945776462554932,
      -2.6458980073584825732e-07,
      0.0038298231083899736404,
      -1.07096025203645695e-06,
      -0.0043701459653675556183,
      0.0010622221743687987328,
    }
  },

  {
    8000, 1,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 1",
    7,
    {
      1.0272867679595947266,
      0.57039743661880493164,
      0.22854259610176086426,
      0.11283638328313827515,
      0.04545628279447555542,
      0.015480343252420425415,
      -1.8298457860055350466e-07,
      0,
    }
  },

  {
    8000, 9,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Saturated",
    16,
    {
      1.1738812923431396484,
      0.87636357545852661133,
      0.61603987216949462891,
      0.52298575639724731445,
      0.44386821985244750977,
      0.38358041644096374512,
      0.32324904203414916992,
      0.27636924386024475098,
      0.22878263890743255615,
      0.1899569779634475708,
      0.15033221244812011719,
      0.11752252280712127686,
      0.084870077669620513916,
      0.057567544281482696533,
      0.032759506255388259888,
      0.013133523054420948029,
    }
  },

  {
    8000, 98,
    LIBDITHER_TYPE_SIMPLE, "Simple triangular",
    1,
    {
      -1,
    },
  },

  {
    8000, 99,
    LIBDITHER_TYPE_SIMPLE, "No shaper",
    0,
    {
    },
  },

  //

  {
    11025, 0,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 0",
    8,
    {
      0.53594851493835449219,
      0.43384709954261779785,
      8.2579242643987527117e-07,
      -0.0023528621532022953033,
      -0.020154647529125213623,
      0.0076960241422057151794,
      0,
      2.1015614137809279782e-08,
    }
  },

  {
    11025, 1,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 1",
    6,
    {
      0.77642798423767089844,
      0.73626911640167236328,
      0.28617018461227416992,
      0.14926058053970336914,
      0.035679142922163009644,
      0.016192987561225891113,
      0,
      0,
    }
  },

  {
    11025, 9,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Saturated",
    16,
    {
      0.90201318264007568359,
      0.96327704191207885742,
      0.59832245111465454102,
      0.48709768056869506836,
      0.36537483334541320801,
      0.31974372267723083496,
      0.26105600595474243164,
      0.22021849453449249268,
      0.17551329731941223145,
      0.1435108184814453125,
      0.10948005318641662598,
      0.085198312997817993164,
      0.059693295508623123169,
      0.041339982300996780396,
      0.021607741713523864746,
      0.0095272064208984375,
    }
  },

  {
    11025, 98,
    LIBDITHER_TYPE_SIMPLE, "Simple triangular",
    1,
    {
      -1,
    },
  },

  {
    11025, 99,
    LIBDITHER_TYPE_SIMPLE, "No shaper",
    0,
    {
    },
  },

  //

  {
    22050, 0,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 0",
    7,
    {
      -0.24690496921539306641,
      0.4056072533130645752,
      0.178049430251121521,
      0.12218115478754043579,
      0.044338006526231765747,
      -7.4252207014069426805e-06,
      -0.0029117804951965808868,
      0,
    }
  },

  {
    22050, 1,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Intensity 1",
    12,
    {
      -0.091535776853561401367,
      0.53762668371200561523,
      0.36664107441902160645,
      0.29549720883369445801,
      0.25240615010261535645,
      0.14534264802932739258,
      0.12099406123161315918,
      0.089816331863403320312,
      0.049367569386959075928,
      0.030478563159704208374,
      0.01209077145904302597,
      0.0082031264901161193848,
    }
  },

  {
    22050, 9,
    LIBDITHER_TYPE_ATH_A, "ATH Curve A, Saturated",
    16,
    {
      -0.06466051936149597168,
      0.55687421560287475586,
      0.39284247159957885742,
      0.3238837122917175293,
      0.28003260493278503418,
      0.17455317080020904541,
      0.15124039351940155029,
      0.12386889010667800903,
      0.087459541857242584229,
      0.071950778365135192871,
      0.052037805318832397461,
      0.044413808733224868774,
      0.029589230194687843323,
      0.020830472931265830994,
      0.010722415521740913391,
      0.0067992187105119228363,
    }
  },

  {
    22050, 98,
    LIBDITHER_TYPE_SIMPLE, "Simple triangular",
    1,
    {
      -1,
    },
  },

  {
    22050, 99,
    LIBDITHER_TYPE_SIMPLE, "No shaper",
    0,
    {
    },
  },

  {
    -1, 0,
    0, "End of the table",
    0,
    {
    },
  },
};

