// -*- coding: utf-8 -*-
// Copyright (C) 2009, 2012, 2013, 2014 Laboratoire de Recherche et
// Développement de l'Epita (LRDE).
//
// This file is part of Spot, a model checking library.
//
// Spot is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// Spot is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
// License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// \file ltlast/formula_tree.hh
/// \brief Trees representing formulae where atomic propositions are unknown
#ifndef SPOT_LTLAST_FORMULA_TREE_HH
# define SPOT_LTLAST_FORMULA_TREE_HH

# include "formula.hh"
# include <vector>
# include <boost/shared_ptr.hpp>
# include "multop.hh"
# include "binop.hh"
# include "unop.hh"
# include "nfa.hh"

namespace spot
{
  namespace ltl
  {
    /// Trees representing formulae where atomic propositions are unknown.
    namespace formula_tree
    {
      // These struct should not need to be made public using
      // SPOT_API, unfortunately dynamic_cast<> with g++-4.0.1 fails
      // on Darwin if we do not.
      struct SPOT_API node
      {
	virtual ~node() {};
      };
      /// We use boost::shared_ptr to easily handle deletion.
      typedef boost::shared_ptr<node> node_ptr;

      struct SPOT_API node_unop : node
      {
	unop::type op;
	node_ptr child;
      };
      struct SPOT_API node_binop : node
      {
	binop::type op;
	node_ptr lhs;
	node_ptr rhs;
      };
      struct SPOT_API node_multop : node
      {
	multop::type op;
	node_ptr lhs;
	node_ptr rhs;
      };
      struct SPOT_API node_nfa : node
      {
	std::vector<node_ptr> children;
	spot::ltl::nfa::ptr nfa;
      };
      /// Integer values for True and False used in node_atomic.
      enum { True = -1, False = -2 };
      struct SPOT_API node_atomic : node
      {
	int i;
      };

      /// Instanciate the formula corresponding to the node with
      /// atomic propositions taken from \a v.
      const formula* instanciate(const node_ptr np,
				 const std::vector<const formula*>& v);

      /// Get the arity.
      size_t arity(const node_ptr np);
    }
  }
}

#endif // SPOT_LTLAST_FORMULA_TREE_HH_
