package Spidy::Database;

use Spidy::Database::Structure;
use Carp;
use strict;

# ignore subroutine warnings until end of
# this file being parsed;
my $WARNSUB;
BEGIN {
  $WARNSUB = $SIG{__WARN__};
  $SIG{__WARN__} = sub {
    return if $_[0] =~ /ubroutine \w+ redefined/;
    warn @_;
  };
}

sub create {
  my $self = shift;
  my $db = new Spidy::Database::Structure;
  for my $table ( $db->get_tables() ) {
    my (@pre_sql, @col_sql, @post_sql);
    for my $col ( $table->get_cols() ) {
      my ($pre_sql, $col_sql, $post_sql) = $self->format_create_col($col);
      push @pre_sql, $pre_sql if $pre_sql;
      push @col_sql, $col_sql if $col_sql;
      push @post_sql, $post_sql if $post_sql;
    }
    
    for my $key ( $table->get_keys() ) {
      my $key_sql = $self->format_create_key( $key );
      push @col_sql, $key_sql if $key_sql;
    }
    my $sql = "CREATE TABLE " . $table->{'name'} . " (\n    ";
       $sql .= join ",\n    ", @col_sql;
       $sql .= "\n)";
    $self->query($_) for @pre_sql;
    $self->query($sql);
    $self->query($_) for @post_sql;
#     warn $_, "\n" for @pre_sql;
#     warn $sql,"\n";
#     warn $_, "\n" for @post_sql;
  }
}

sub format_create_col {
  my $self = shift;
  my $col = shift;
  my( $pre, $cur, $post ) = $self->format_column_value($col);

  my $sql = sprintf(
    "%-20s %s%s", 
    $col->{'name'},
    $self->format_column_type($col),
    $cur ? " $cur" : ""
  );
  return ( $pre, $sql, $post );
}

sub format_column_type {
  my $self = shift;
  my $col = shift;
  return "INT"                     if $col->{'type'} eq 'int';
  return "BOOL"                    if $col->{'type'} eq 'bool';
  return "TEXT"                    if $col->{'type'} eq 'text';
  return "VARCHAR($col->{'size'})" if $col->{'type'} eq 'varchar';
  return "CHAR($col->{'size'})"    if $col->{'type'} eq 'char';
  croak "Database Column Type \"".$col->{'type'}."\" not found for Database Type";
}

sub format_column_value {
  my $self = shift;
  my $col = shift;
  return (
    undef,
    $self->format_null_value($col),
  )
}

sub format_null_value {
  my $self = shift;
  my $col = shift;
  return $col->{'null'} ? "NULL" : "NOT NULL";
}

sub format_create_key {
  my $self = shift;
  my $key = shift;
  my $cols = $key->get_cols();
  return "PRIMARY KEY ($cols)" if $key->get_type() eq 'primary';
  return "FOREIGN KEY ($cols) REFERENCES ".$key->get_ref_table()."(".$key->get_ref_column().")"
    if $key->get_type() eq 'foreign';
  return "UNIQUE ($cols)" if $key->get_type() eq 'unique';  
}

sub tables {
  my $self = shift;
  return $self->tables();
}

sub drop {
  my $self = shift;
  my $db = new Spidy::Database::Structure;
  my @tables = reverse map { $_->{'name'} } $db->get_tables();
  my $dbh = $self->get_dbh();
#  my @current_tables = $self->tables()
  my @current_tables = $dbh->tables();
  for my $table (@tables) {
    $self->query("DROP TABLE $table") 
      if grep { $table eq $_ } @current_tables;
  }
}

# after file is parsed
# reinstall default warning sub.
$SIG{__WARN__} = $WARNSUB;
1;
