// ----------------------------------------------------------------------------
//
#include <stdarg.h>		// Use ...
#include <stdio.h>		// Use vsprintf()
#include <time.h>		// Use time()

#include "command.h"		// use ignore_keys()
#include "memalloc.h"		// use new()
#include "session.h"		// use Session
#include "stringc.h"		// Use Stringy
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "utility.h"		// Use fatal_error()
#include "wait.h"		// use Wait
#include "winsystem.h"

#define DEFAULT_WAIT_DELAY 1.0		// Seconds before dialog pops up.

#define MAX_LINE_LENGTH		4096

class wait_dialog
{
public:
  wait_dialog(Session &);
  void show(const Stringy &msg, bool stoppable, double delay);
  void unshow();
  bool was_stop_requested(const char *message);
private:
  Session &session;
  WinSys &ws;
  Widget dialog;
  bool dialog_shown;
  bool stop_requested;
  time_t start_time, last_report_time;
  double delay;

  static void destroy_cb(Widget, CB_Data wdialog, CB_Data);
  static void abort_cb(Widget, CB_Data wdialog, CB_Data);
};

// ----------------------------------------------------------------------------
//
wait_dialog::wait_dialog(Session &s) : session(s), ws(s.window_system())
{
  dialog = NULL;
  dialog_shown = false;
  stop_requested = false;
}

// ----------------------------------------------------------------------------
//
void wait_dialog::show(const Stringy &msg, bool stoppable, double delay)
{
  if (dialog)
    fatal_error("wait_dialog::show(): Tried to start a second wait dialog.\n");

  if (stoppable)
    dialog = ws.choice_dialog("workingDialog", false, msg.cstring(),
			   "Cancel", abort_cb, (void *) this,
			   NULL);
  else
    dialog = ws.choice_dialog("workingDialog", false, msg.cstring(), NULL);
  ws.add_event_callback(destroy_event, dialog, destroy_cb, this);
  ws.grab_events(dialog);

  //
  // The grab only applies to pointer events.  But see comments on the
  // grab_events routine that say Tk won't switch keyboard focus window
  // during grab, even though window manager switches focus.
  //
  session.command_dispatcher().ignore_keys(true);

  dialog_shown = false;
  stop_requested = false;
  start_time = time(NULL);
  last_report_time = 0;
  this->delay = delay;
}

// ----------------------------------------------------------------------------
//
void wait_dialog::destroy_cb(Widget, CB_Data wdialog, CB_Data)
{
  wait_dialog *wd = (wait_dialog *) wdialog;
  wd->ws.remove_event_callback(destroy_event, wd->dialog, wd->destroy_cb, wd);
  wd->dialog = NULL;
  wd->dialog_shown = false;
  wd->stop_requested = false;
  wd->session.command_dispatcher().ignore_keys(false);
}

// ----------------------------------------------------------------------------
//
void wait_dialog::unshow()
{
  ws.ungrab_events(dialog);
  ws.delete_choice_dialog(dialog);
  dialog = NULL;
  stop_requested = false;
  dialog_shown = false;
}

// ----------------------------------------------------------------------------
//
void wait_dialog::abort_cb(Widget, CB_Data wdialog, CB_Data)
{
  wait_dialog *wd = (wait_dialog *) wdialog;
  wd->stop_requested = true;
}

// ----------------------------------------------------------------------------
//
bool wait_dialog::was_stop_requested(const char *message)
{
  if (dialog)
    {
      if (message)
	if (time(NULL) > last_report_time)
	  {
	    ws.new_choice_question(dialog, message);
	    last_report_time = time(NULL);
	  }

      if (!dialog_shown && (double)(time(NULL) - start_time) >= delay)
	{
	  ws.show_dialog(dialog);
	  ws.raise_widget(dialog);
	  dialog_shown = true;
	}

      if (dialog_shown)
	ws.process_pending_events();
    }

  return stop_requested;
}

// ----------------------------------------------------------------------------
//
class wait_cb : public Wait
{
public:
  wait_cb(Session &);
  virtual ~wait_cb();
  virtual void begin_waiting(const char *msg, bool stoppable);
  virtual void end_waiting();
  virtual bool was_stop_requested();
  virtual void progress_report(const char *format, ...);
private:
  wait_dialog *wd;
};

// ----------------------------------------------------------------------------
//
Wait *wait_callbacks(Session &s)
{
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_data("wait_callbacks") == NULL)
    {
      Wait *wcb = new wait_cb(s);		// never deleted
      dt.set_data("wait_callbacks", wcb);
    }
  return (Wait *) dt.get_data("wait_callbacks");
}

// ----------------------------------------------------------------------------
//
wait_cb::wait_cb(Session &s)
{
  wd = new wait_dialog(s);
}

// ----------------------------------------------------------------------------
//
wait_cb::~wait_cb()
{
  delete wd;
  wd = NULL;
}

// ----------------------------------------------------------------------------
//
void wait_cb::begin_waiting(const char *message, bool stoppable)
  { wd->show(message, stoppable, DEFAULT_WAIT_DELAY); }
void wait_cb::end_waiting()
  { wd->unshow(); }
bool wait_cb::was_stop_requested()
  { return wd->was_stop_requested(NULL); }

// ----------------------------------------------------------------------------
//
void wait_cb::progress_report(const char *format, ...)
{
  char message[MAX_LINE_LENGTH];

  va_list args;
  va_start(args, format);
  vsprintf(message, format, args);
  va_end(args);

  (void) wd->was_stop_requested(message);
}
