// ----------------------------------------------------------------------------
// The View_Window class provides a generic user interface for showing a
// 2-D plot of N-D data.  It handles window layout and events including
// edge panels to show scales, 1-D slices, and other 1-D annotations, and
// zooming, scrolling and selection with the mouse.
//
// The View_Window class knows nothing about spectra or peaks.
//

#ifndef VIEWWIN_HEADER_INCLUDED
#define VIEWWIN_HEADER_INCLUDED

#include "list.h"		// Use List
#include "uidrawing.h"		// use Drawing
#include "spoint.h"		// use SPoint
#include "stringc.h"		// Use Stringy
#include "uiepanel.h"		// Use Edge_Panel
#include "winsystem.h"		// Use Widget, CB_Data

// ----------------------------------------------------------------------------
//
class View_Window : public Drawing
{
 public:
  virtual ~View_Window();

  void set_title(const Stringy &title, const Stringy &icon_title);
  void set_view(const SPoint &center, const SPoint &pixel_size,
		bool sync_other_views = true);
  void set_axis_order(const IPoint &axis_order);
  void set_visible_depth(const SPoint &depth);
  void set_aspect(double aspect);
  void show_view(bool show);

  SPoint visible_depth() const;
  double visible_depth(int axis) const;
  bool shown() const;
  int axis(Axis a) const;
  IPoint axis_order() const;
  SPoint center() const;
  double center(int axis) const;
  double center(Axis axis) const;
  SPoint pixel_size() const;
  double pixel_size(int axis) const;
  double pixel_size(Axis axis) const;
  double aspect() const;
  Widget frame() const;
  bool is_top_level_window() const;
  virtual bool ok_to_destroy();

  SRegion view_region();
  Rectangle flatten(const SRegion &region);
  SRegion flat_region(Rectangle r);
  SRegion thicken(const Rectangle &r);

  void show_edge_panels(const List &inside, const List &outside,
			bool show_scrollbars, bool resize_window);
  bool edge_panel_shown(Edge_Panel ep);
  bool scrollbars_shown();

  void screen_location(int *x, int *y);
  void set_screen_location(int x, int y);

// ----------------------------------------------------------------------------
//
 protected:

  View_Window(WinSys &, Widget parent, int xsize, int ysize,
	      const IPoint &axis_order,
	      const SPoint &center, const SPoint &pixel_size,
	      const SPoint &depth, const SRegion &fullview,
	      const SPoint &font_height, const SPoint &scroll_step,
	      const SRegion &repaint_tile, double repaint_priority,
	      bool contour_graying, Widget popup_menu);

  int dimension() const;
  void set_font_height(const SPoint &font_height);
  void set_axes(Stringy *labels,
		const SPoint &lo_scale, const SPoint &hi_scale);
  void set_repaint_region(const SRegion &fullview);
  void set_repaint_tile(const SRegion &tile);

  virtual void mapped_or_unmapped();
  virtual void synchronize_other_views();

  virtual void move(double x, double y);		// Mouse move.
  virtual void enter_window(double x, double y);	// Mouse entered window
  virtual void exit_window(double x, double y);		// Mouse exited window

// ----------------------------------------------------------------------------
 private:

  IPoint axisorder;
  SPoint lo_scale, hi_scale, centr, pixel_sz, dpth, font_height, scroll_step;
  SRegion drawing_reg, scroll_reg, repaint_tile;
  Stringy axis_labels[DIM];
  Widget framew, hbar, vbar, popup_menu;
  List z_scrollbars;
  List edge_panels;
  List vert_widgets, horz_widgets;

  static void destroy_query_cb(Widget, CB_Data, CB_Data);
  static void destroy_cb(Widget, CB_Data, CB_Data);
  static void map_cb(Widget, CB_Data, CB_Data);
  static void scroll_cb(Widget, CB_Data, CB_Data);
  static void z_value_cb(Widget, CB_Data, CB_Data);
  static void popup_menu_cb(Widget, CB_Data, CB_Data);

  void create_scrollbars();
  void scroll(Widget sbar);
  int find_z_scrollbar_axis(Widget sbar);
  void entered_z_value(Widget text);
  int find_z_text_axis(Widget text);
  void position_view_widgets(bool resize_window);
  void set_window_size();
  void update_scrollbars();
  bool position_edge_panel(Edge_Panel ep, bool show, bool near);
  bool position_scrollbars(bool show);

  void update_view(const IPoint &axis_order, const SPoint &center,
		   const SPoint &pixel_size, const SPoint &depth,
		   bool update_scrollbars, bool sync_other_views);
  virtual void drawing_resized();
  void update_panels();
  void update_z_labels();
  double to_scale_value(double d, int axis);
  double from_scale_value(double d, int axis);
};

bool destroy_view_window(View_Window *vw);

#endif
