// ----------------------------------------------------------------------------
// Dialog to list resonances (group, atom, freq, std dev, # assignments)
//

#include "atom.h"		// use Atom
#include "condition.h"		// use Condition
#include "format.h"		// use load_resonances()
#include "group.h"		// use Group
#include "list.h"		// use List
#include "memalloc.h"		// use new()
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "resonance.h"		// use Resonance
#include "session.h"		// use Session
#include "stringc.h"		// use Stringy
#include "uicomponents.h"	// use Condition_Menu
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uipeak.h"		// use show_peak_list_dialog()

// ----------------------------------------------------------------------------
//
class resonance_list_dialog : public Dialog
{
public:
  resonance_list_dialog(Session &);
  ~resonance_list_dialog();

  static resonance_list_dialog *the(Session &);

  void show(const Condition &c);

private:
  Widget heading, list;
  Condition_Menu *cmenu;

  static void select_cb(Widget, CB_Data, CB_Data);
  static void update_cb(Widget, CB_Data, CB_Data);
  static void save_cb(Widget, CB_Data, CB_Data);
  static void load_cb(Widget, CB_Data, CB_Data);

  void update();
  void set_resonance_list(const List &resonances);
  Stringy heading_text();
  Stringy resonance_text(Resonance *r);
  Resonance *resonance_from_text(const Stringy &text);
};

// ----------------------------------------------------------------------------
//
void show_resonance_list_dialog(Session &s, const Condition &c)
{
  resonance_list_dialog::the(s)->show(c);
}

// ----------------------------------------------------------------------------
//
resonance_list_dialog::resonance_list_dialog(Session &s)
  : Dialog(s, "resonanceListDialog")
{

  ws.set_dialog_title(dialog, "Resonance List");

  cmenu = new Condition_Menu(s, dialog, "condition");

  Widget scroller = ws.create_scrolled_list(dialog, "list", &list);
  ws.extended_selection_list(list);
  ws.add_list_selection_callback(list, select_cb, this);
  heading = ws.scrolled_list_heading(scroller);

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "update", update_cb, this,
			       "save", save_cb, this,
			       "load", load_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(scroller, cmenu->option_menu(), scroller,
		     separator, controls, END_OF_WIDGETS);

}

// ----------------------------------------------------------------------------
//

// ----------------------------------------------------------------------------
//
resonance_list_dialog::~resonance_list_dialog()
{
  session.dialog_table().delete_dialog("resonance_list_dialog", this);

  ws.remove_list_selection_callback(list, select_cb, this);

  delete cmenu;

}

// ----------------------------------------------------------------------------
// The default resonance_list_dialog instance.
//
resonance_list_dialog *resonance_list_dialog::the(Session &s)
{
  Stringy name = "resonance_list_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new resonance_list_dialog(s));
  return (resonance_list_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void resonance_list_dialog::select_cb(Widget, CB_Data rldialog, CB_Data)
{
  resonance_list_dialog *rld = (resonance_list_dialog *) rldialog;

  Stringy text = rld->ws.selected_list_item(rld->list);
  Resonance *r = rld->resonance_from_text(text);
  if (r)
    show_peak_list_dialog(rld->session, r);
}

// ----------------------------------------------------------------------------
//
void resonance_list_dialog::update_cb(Widget, CB_Data rldialog, CB_Data)
{
  resonance_list_dialog *rld = (resonance_list_dialog *) rldialog;
  rld->update();
}

// ----------------------------------------------------------------------------
//
void resonance_list_dialog::save_cb(Widget, CB_Data rldialog, CB_Data)
{
  resonance_list_dialog *rld = (resonance_list_dialog *) rldialog;
  Stringy defpath = rld->session.project().list_path("shifts");
  Stringy path = rld->ws.saveas_file_dialog(rld->dialog, "Save Resonance List",
					    defpath, "resonancelist", true);
  if (! path.is_empty())
    if (!rld->ws.save_list_lines(rld->list, rld->heading_text(), path))
      rld->session.reporter().warning("Couldn't write %s\n", path.cstring());


}

// ----------------------------------------------------------------------------
//
void resonance_list_dialog::load_cb(Widget, CB_Data rldialog, CB_Data)
{
  resonance_list_dialog *rld = (resonance_list_dialog *) rldialog;

  Stringy defpath = rld->session.project().list_path("");
  Stringy path = rld->ws.open_file_dialog(rld->dialog, "Load Resonance List",
					  defpath, "resonancelist", true);
  Condition *c = rld->cmenu->condition_chosen();
  if (! path.is_empty() && c)
    {
      load_resonances(path, c);
      rld->update();
    }
}

// ----------------------------------------------------------------------------
//
void resonance_list_dialog::show(const Condition &c)
{
  cmenu->set_condition_choice(c);
  update();
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void resonance_list_dialog::update()
{
  Condition *c = cmenu->condition_chosen();
  if (c == NULL)
    return;
  
  set_resonance_list(c->resonance_list());
}

// ----------------------------------------------------------------------------
//
void resonance_list_dialog::set_resonance_list(const List &rlist)
{   
  ws.set_label_text(heading, heading_text());

  List sorted_rlist = rlist;
  sorted_rlist.sort(compare_resonance_names);

  List strings;
  for (int ri = 0 ; ri < sorted_rlist.size() ; ++ri)
    {
      Resonance *r = (Resonance *) sorted_rlist[ri];
      Stringy *line = new Stringy(resonance_text(r));
      strings.append(line);
    }

  int top = ws.top_visible_list_position(list);
  ws.set_list_items(list, strings);
  free_string_list_entries(strings);

  if (top < sorted_rlist.size())
    ws.set_top_visible_list_position(list, top);
}

// ----------------------------------------------------------------------------
//
Stringy resonance_list_dialog::heading_text()
{
  return " Group   Atom  Nuc    Shift   SDev  Assignments";
}

// ----------------------------------------------------------------------------
//
Stringy resonance_list_dialog::resonance_text(Resonance *r)
{
  return formatted_string("%6s %6s %4s %8.3f %6.3f %6d",
			  r->group()->name().cstring(),
			  r->atom()->name().cstring(),
			  r->atom()->nucleus().cstring(),
			  r->frequency(),
			  r->frequencySigma(),
			  r->assignment_count());
}

// ----------------------------------------------------------------------------
//
Resonance *resonance_list_dialog::resonance_from_text(const Stringy &text)
{
  Condition *c = cmenu->condition_chosen();
  if (c == NULL)
    return NULL;

  Stringy rest;
  Stringy group = first_token(text, &rest);
  Stringy atom = first_token(rest, &rest);

  return c->find_resonance(group, atom);
}
