// ----------------------------------------------------------------------------
// Program for print UCSF NMR file headers and writing data matrix.
//

#include <iostream>		// use cout
#include <stdio.h>		// Use FILE
#include <stdlib.h>		// Use exit()
#include <string.h>		// Use strcmp()

#include "memalloc.h"		// use new()
#include "memcache.h"		// use Memory_Cache
#include "nmrdata.h"		// Use NMR_Data
#include "spoint.h"		// Use IRegion
#include "stringc.h"		// Use Stringy
#include "ucsffile.h"		// Use print_ucsf_nmr_header()

#define MEMORY_CACHE_SIZE 8000000

// ----------------------------------------------------------------------------
// Command line options.
//
struct Options
{
  bool write_matrix;
  bool long_header;
  Stringy ucsf_input;
  Stringy ucsf_output;
  float negative_threshold;
  float positive_threshold;
  IRegion subregion;
  IPoint squish_cell;
  int project_axis;
  bool reduce_dimension;

  //
  // Options below are for changing UCSF header values
  //
  bool change_axis_name[DIM];
  Stringy axis_name[DIM];
  bool change_origin[DIM];
  float origin[DIM];
  bool change_spectral_width[DIM];
  float spectral_width[DIM];
  bool change_frequency[DIM];
  float frequency[DIM];
};

// ----------------------------------------------------------------------------
//
static bool parse_args(int argc, char **argv, Memory_Cache *,
		       Stringy *err_msg, Options *opt);
static bool parse_subregion(int argc, char **argv, IRegion *region);
static bool parse_cell_size(int argc, char **argv, IPoint *cell_size);
static bool change_ucsf_header(const Options &options);
static NMR_Data *nmr_data_out(NMR_Data *nmr_input, const Options &options);
static void write_data_matrix(NMR_Data *nmr_data, FILE *outfp);

// ----------------------------------------------------------------------------
//
const char *use =
"Usage: ucsfdata [-w1 i j] [-w2 i j] [-w3 i j] [-w4 i j]\n"
"                [-p1|-p2|-p3|-p4]\n"
"                [-s1 r] [-s2 r] [-s3 r] [-s4 r]\n"
"                [-a1 name] [-a2 name] [-a3 name] [-a4 name]\n"
"                [-o1 ppm] [-o2 ppm] [-o3 ppm] [-o4 ppm]\n"
"                [-sw1 Hz] [-sw2 Hz] [-sw3 Hz] [-sw4 Hz]\n"
"                [-f1 MHz] [-f2 MHz] [-f3 MHz] [-f4 MHz]\n"
"                [-t negative-threshold positive-threshold]\n"
"                [-r] [-m] [-l] [-o output-file] input-file\n"
"\n"
"Prints header information for UCSF format NMR data files.\n"
"Produces new UCSF data files for subregions, or with data below\n"
"  specified thresholds zeroed, or with reduced matrix size.\n"
"Produces matrix file with no header information.\n"
"\n"
"  -wN low-index high-index\n"
"	specifies subregion for axis N\n"
"  -pN\n"
"	project along axis N taking data value of largest magnitude.\n"
"  -sN cell-size\n"
"	replace cells by single data value of largest magnitude\n"
"  -aN axis-nucleus-name\n"
"	change the nucleus name (eg 1H, 15N, 13C) for axis N\n"
"  -oN origin-in-ppm\n"
"	change the chemical shift of the downfield edge of spectrum\n"
"  -swN spectral-width-in-hz\n"
"	change the spectral width\n"
"  -fN spectrometer-frequency-in-MHz\n"
"	change the spectrometer frequency (eg 600 Mhz for 1H)\n"
"  -t negative-threshold positive-threshold\n"
"	zeros all data points below threshold values\n"
"  -r\n"
"	reduce dimension by eliminating dimensions with only one plane\n"
"  -m\n"
"	outputs the data matrix (last axis varying fastest) to stdout\n"
"  -l\n"
"	print long format header info for data processed with Striker\n";

// ----------------------------------------------------------------------------
//
int main(int argc, char **argv)
{
  Options options;
  Stringy err_msg;

  Memory_Cache mcache(MEMORY_CACHE_SIZE);
  if (!parse_args(argc, argv, &mcache, &err_msg, &options))
    {
      std::cerr << "ucsfdata: " << err_msg.cstring() << std::endl;
      fputs(use, stderr);
      exit(1);
    }

  if (!change_ucsf_header(options))
    { fputs(use, stderr); exit(1); }

  NMR_Data *nmr_input = ucsf_nmr_data(options.ucsf_input, &mcache);
  if (nmr_input == NULL)
    {
      std::cerr << "ucsfdata: Couldn't open " << options.ucsf_input.cstring()
	<< std::endl;
      exit(1);
    }

  NMR_Data *nmr_data = nmr_data_out(nmr_input, options);

  if (options.write_matrix)
    write_data_matrix(nmr_data, stdout);
  else if (options.ucsf_output.is_empty())
    if (options.long_header)
      print_ucsf_nmr_header(nmr_data->path(), stdout);
    else
      nmr_data->print_header(std::cout);

  if (!options.ucsf_output.is_empty())
    if (!write_ucsf_nmr_data(nmr_data, options.ucsf_output,
			     &err_msg, &mcache))
      {
	std::cerr << "ucsfdata: " << err_msg.cstring() << std::endl;
	exit(1);
      }

  delete nmr_data;

  return 0;
}

// ----------------------------------------------------------------------------
//
static bool parse_args(int argc, char **argv, Memory_Cache *mcache,
		       Stringy *err_msg, Options *options)
{
  bool long_header = false, matrix = false, reduce_dim = false;
  int d, project_axis = -1;
  float neg_thresh = 0, pos_thresh = 0;
  Stringy output_path, input_path, axis_name[DIM];
  bool change_axis_name[DIM], change_origin[DIM], change_spectral_width[DIM];
  bool change_frequency[DIM];
  float origin[DIM], spectral_width[DIM], frequency[DIM];

  for (int a = 0 ; a < DIM ; ++a)
    {
      change_axis_name[a] = false;
      change_origin[a] = false;
      change_spectral_width[a] = false;
      change_frequency[a] = false;
    }

  for (int a = 1 ; a < argc ; ++a)
    if (sscanf(argv[a], "-w%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+2 < argc) { a += 2; }
      else { *err_msg = "Missing -w value"; return false; }
    else if (sscanf(argv[a], "-s%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+1 < argc) { a += 1; }
      else { *err_msg = "Missing -s value"; return false; }
    else if (sscanf(argv[a], "-a%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+1 < argc)
	{ a += 1; axis_name[d-1] = argv[a]; change_axis_name[d-1] = true;}
      else
	{ *err_msg = "Missing -a value"; return false; }
    else if (sscanf(argv[a], "-o%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+1 < argc)
	{ a += 1; origin[d-1] = atof(argv[a]); change_origin[d-1] = true;}
      else
	{ *err_msg = "Missing -o value"; return false; }
    else if (sscanf(argv[a], "-sw%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+1 < argc)
	{
	  a += 1;
	  spectral_width[d-1] = atof(argv[a]);
	  change_spectral_width[d-1] = true;
	}
      else
	{ *err_msg = "Missing -sw value"; return false; }
    else if (sscanf(argv[a], "-f%d", &d) == 1 && d >= 1 && d <= DIM)
      if (a+1 < argc)
	{
	  a += 1;
	  frequency[d-1] = atof(argv[a]);
	  change_frequency[d-1] = true;
	}
      else
	{ *err_msg = "Missing -f value"; return false; }
    else if (sscanf(argv[a], "-p%d", &d) == 1 && d >= 1 && d <= DIM)
      project_axis = d-1;
    else if (strcmp(argv[a], "-r") == 0)
      reduce_dim = true;
    else if (strcmp(argv[a], "-t") == 0)
      if (a+1 < argc && sscanf(argv[a+1], "%f", &neg_thresh) == 1 &&
	  a+2 < argc && sscanf(argv[a+2], "%f", &pos_thresh) == 1 &&
	  neg_thresh <= 0 && pos_thresh >= 0)
	{ a += 2; }
      else
	{ *err_msg = "Missing -t value"; return false; }
    else if (strcmp(argv[a], "-m") == 0)
      matrix = true;
    else if (strcmp(argv[a], "-l") == 0)
      long_header = true;
    else if (strcmp(argv[a], "-o") == 0)
      if (a+1 < argc) { a += 1; output_path = argv[a]; }
      else { *err_msg = "Missing -o value"; return false; }
    else if (a == argc - 1)
      input_path = argv[a];
    else
      {
	*err_msg = formatted_string("Unknown option %s", argv[a]);
	return false;
      }

  NMR_Data *nmr_input = ucsf_nmr_data(input_path, mcache);
  if (nmr_input == NULL)
    {
      *err_msg = formatted_string("Couldn't open %s", input_path.cstring());
      return false;
    }
  int dim = nmr_input->dimension();
  IRegion region = nmr_input->data_region();
  delete nmr_input;

  if (!parse_subregion(argc, argv, &region))
    { *err_msg = "Bad -w option"; return false; }

  IPoint squish_cell(dim);
  if (!parse_cell_size(argc, argv, &squish_cell))
    { *err_msg = "Bad -s option"; return false; }

  options->ucsf_input = input_path;
  options->long_header = long_header;
  options->write_matrix = matrix;
  options->ucsf_output = output_path;
  options->subregion = region;
  options->negative_threshold = neg_thresh;
  options->positive_threshold = pos_thresh;
  options->squish_cell = squish_cell;
  options->project_axis = project_axis;
  options->reduce_dimension = reduce_dim;
  for (int a = 0 ; a < dim ; ++a)
    {
      options->axis_name[a] = axis_name[a];
      options->change_axis_name[a] = change_axis_name[a];
      options->origin[a] = origin[a];
      options->change_origin[a] = change_origin[a];
      options->spectral_width[a] = spectral_width[a];
      options->change_spectral_width[a] = change_spectral_width[a];
      options->frequency[a] = frequency[a];
      options->change_frequency[a] = change_frequency[a];
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static bool parse_subregion(int argc, char **argv, IRegion *region)
{
  int dim = region->dimension();
  int d, lo, hi;
  for (int a = 1 ; a < argc ; ++a)
    if (sscanf(argv[a], "-w%d", &d) == 1 && d >= 1 && d <= dim)
      if (a+1 < argc && sscanf(argv[a+1], "%d", &lo) == 1 &&
	  a+2 < argc && sscanf(argv[a+2], "%d", &hi) == 1 && lo <= hi)
	{
	  a += 2;
	  region->min[d-1] = lo;
	  region->max[d-1] = hi;
	}
      else
	return false;
  return true;
}

// ----------------------------------------------------------------------------
//
static bool parse_cell_size(int argc, char **argv, IPoint *cell_size)
{
  int dim = cell_size->dimension();
  for (int a = 0 ; a < dim ; ++a)
    (*cell_size)[a] = 1;

  int d, size;
  for (int a = 1 ; a < argc ; ++a)
    if (sscanf(argv[a], "-s%d", &d) == 1 && d >= 1 && d <= dim)
      if (a+1 < argc && sscanf(argv[a+1], "%d", &size) == 1 && size >= 1)
	{
	  a += 1;
	  (*cell_size)[d-1] = size;
	}
      else
	return false;
  return true;
}

// ----------------------------------------------------------------------------
//
static bool change_ucsf_header(const Options &options)
{
  int dim = options.subregion.dimension();
  for (int a = 0 ; a < dim ; ++a)
    {
      if (options.change_axis_name[a])
	if (!change_ucsf_axis_label(options.ucsf_input, a,
				    options.axis_name[a]))
	  return false;
      if (options.change_spectral_width[a])
	if (!change_ucsf_spectral_width(options.ucsf_input, a,
					options.spectral_width[a]))
	  return false;
      if (options.change_frequency[a])
	if (!change_ucsf_spectrometer_frequency(options.ucsf_input, a,
						options.frequency[a]))
	  return false;
      if (options.change_origin[a])
	if (!change_ucsf_origin(options.ucsf_input, a, options.origin[a]))
	  return false;
    }
      
  return true;
}

// ----------------------------------------------------------------------------
//
static NMR_Data *nmr_data_out(NMR_Data *nmr_input, const Options &options)
{
  NMR_Data *nmr_output = nmr_input;
  if (options.subregion != nmr_input->data_region())
    nmr_output = nmr_data_region(nmr_output, options.subregion);

  if (options.negative_threshold != 0 || options.positive_threshold != 0)
    nmr_output = thresholded_nmr_data(nmr_output,
				      options.negative_threshold,
				      options.positive_threshold);

  bool squish = false;
  int dim = nmr_input->dimension();
  for (int a = 0 ; a < dim ; ++a)
    if (options.squish_cell[a] != 1)
      squish = true;
  if (squish)
    nmr_output = squished_nmr_data(nmr_output, options.squish_cell);

  if (options.project_axis >= 0 && options.project_axis < dim)
    nmr_output = projected_nmr_data(nmr_output, options.project_axis);

  if (options.reduce_dimension)
    for (int a = nmr_output->dimension() - 1 ; a >= 0 ; --a)
      if (nmr_output->size()[a] == 1)
	nmr_output = extract_nmr_data_plane(nmr_output, a, 0);

  return nmr_output;
}

// ----------------------------------------------------------------------------
//
static void write_data_matrix(NMR_Data *nmr_data, FILE *outfp)
{
  int dim = nmr_data->dimension();
  int row_length = nmr_data->size()[dim-1];
  float *data = new float [row_length];

  IRegion base_region = nmr_data->data_region();
  base_region.max[dim - 1] = 0;

  IPoint row_base = base_region.first_point();
  do
    {
      IRegion row(row_base, row_base);
      row.max[dim - 1] = row_length - 1;
      if (!nmr_data->read_values(row, data))
	{
	  std::cerr << "ucsfdata: Error reading data." << std::endl;
	  exit(1);
	}
      if (fwrite(data, sizeof(float), row_length, outfp)
	  != (unsigned) row_length)
	{
	  std::cerr << "ucsfdata: Error writing data." << std::endl;
	  exit(1);
	}
    }
  while (base_region.next_point(&row_base));

  delete [] data;
}
