// ----------------------------------------------------------------------------
// Maps between keys and values.
//
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "table.h"
#include "utility.h"		// use NULL

//
// I use pointers as keys often with the identity hash function.
// To avoid hash value modulo table size often hitting the same value
// it's best if table size is not a power of 2.
//
#define TABLE_INITIAL_SIZE 931
#define GROW_FACTOR 2
#define RESIZE_FACTOR 3
#define SHRINK_FACTOR 64

struct TableEntry
{
  bool used;
  TableKey key;
  TableData data;
  unsigned long hash;
};

static bool equal_pointers(TableKey k1, TableKey k2);
static unsigned long hash_pointer(TableKey key);

// ----------------------------------------------------------------------------
//
Table::Table()
{
  size = 0;
  used = 0;
  entries = NULL;
  this->equal = equal_pointers;
  this->hash = hash_pointer;
}

// ----------------------------------------------------------------------------
//
Table::Table(bool equal(TableKey k1, TableKey k2),
	     unsigned long hash(TableKey key))
{
  size = 0;
  used = 0;
  entries = NULL;
  this->equal = equal;
  this->hash = hash;
}

// ----------------------------------------------------------------------------
//
Table::~Table()
{
  delete [] entries;
}

// ----------------------------------------------------------------------------
// Replaces existing table entry if this key is already in the table.
//
void Table::insert(TableKey key, TableData data)
{
  if (used >= size / GROW_FACTOR)
    resize_table();

  unsigned long h = hash(key);
  insert(key, data, h);
}

// ----------------------------------------------------------------------------
//
void Table::insert(TableKey key, TableData data, unsigned long hash)
{
  TableEntry *te = table_entry(key, hash);
  if (!te->used)
    used += 1;

  te->used = true;
  te->key = key;
  te->data = data;
  te->hash = hash;
}

// ----------------------------------------------------------------------------
// Removes key from table returning key pointer used in table.  The actual
// table key is returned so it can be deallocated if necessary.  If the key
// is not in the table NULL is returned.
//
TableKey Table::remove(TableKey key)
{
  unsigned long h = hash(key);
  TableEntry *te = table_entry(key, h);
  TableKey tkey = NULL;
  if (te && te->used)
    {
      tkey = te->key;
      te->used = false;

      for (TableEntry *ne = next_entry(te) ; ne->used ; ne = next_entry(ne))
	{
	  TableEntry *pe = entries + (ne->hash % size);

	  // Fill if pe is not in (te, ne] taking wrapping into account.

	  if ((te < ne ? (pe <= te || pe > ne) : (pe <= te && pe > ne)))
	    {
	      *te = *ne;
	      ne->used = false;
	      te = ne;
	    }
	}

      used -= 1;
      if (used < size / SHRINK_FACTOR)
	resize_table();
    }
  return tkey;
}

// ----------------------------------------------------------------------------
//
void Table::erase()
{
  for (int e = 0 ; e < size ; ++e)
    entries[e].used = false;
  used = 0;
}

// ----------------------------------------------------------------------------
//
bool Table::find(TableKey key, TableData *data) const
{
  unsigned long h = hash(key);
  TableEntry *te = table_entry(key, h);
  if (data && te && te->used)
    *data = te->data;

  return te && te->used;
}

// ----------------------------------------------------------------------------
//
int Table::entry_count() const
{
  return used;
}

// ----------------------------------------------------------------------------
//
List Table::keys() const
{
  List klist;

  for (int e = 0 ; e < size ; ++e)
    if (entries[e].used)
      klist.append((void *)entries[e].key);

  return klist;
}

// ----------------------------------------------------------------------------
//
List Table::values() const
{
  List vlist;

  for (int e = 0 ; e < size ; ++e)
    if (entries[e].used)
      vlist.append((void *)entries[e].data);

  return vlist;
}

// ----------------------------------------------------------------------------
//
TableEntry *Table::table_entry(TableKey key, unsigned long h) const
{
  if (size == 0)
    return NULL;

  TableEntry *te = entries + (h % size);
  for (; te->used ; te = next_entry(te))
    if (te->hash == h && equal(te->key, key))
      return te;

  return te;
}

// ----------------------------------------------------------------------------
//
TableEntry *Table::next_entry(TableEntry *te) const
{
  return (te + 1 < entries + size ? te + 1 : entries);
}

// ----------------------------------------------------------------------------
//
void Table::resize_table()
{
  int old_size = size;
  TableEntry *old_entries = entries;

  int new_size = (size == 0 ? TABLE_INITIAL_SIZE : RESIZE_FACTOR * used);

  size = new_size;
  entries = new TableEntry [new_size];
  for (int e = 0 ; e < new_size ; ++e)
    entries[e].used = false;
  used = 0;

  for (int e = 0 ; e < old_size ; ++e)
    if (old_entries[e].used)
      insert(old_entries[e].key, old_entries[e].data, old_entries[e].hash);

  delete [] old_entries;
}

// ----------------------------------------------------------------------------
//
unsigned long hash_combine(unsigned long h1, unsigned long h2)
{
  return (h1 + 1234567) ^ h2;
}

// ----------------------------------------------------------------------------
//
static bool equal_pointers(TableKey k1, TableKey k2)
  { return k1 == k2; }
static unsigned long hash_pointer(TableKey key)
{
  unsigned long k = (unsigned long) key;
  return k ^ (k << 3) ^ (k << 7);	// Spread out consecutive keys 1,2,3...
}
