// ----------------------------------------------------------------------------
//
#include <string.h>		// Use memcpy()

#include "subarray.h"		
#include "utility.h"		// Use fatal_error()

// ----------------------------------------------------------------------------
//
Subarray_Iterator::Subarray_Iterator(const IRegion &subreg,
				     const IRegion &region,
				     bool big_endian_indexing)
{
  if (!region.contains(subreg))
    fatal_error("Subarray_Iterator(): Bad subregion.\n");

  dim = region.dimension();
  dimfast = (big_endian_indexing ? dim - 1 : 0);
  dimstep = (big_endian_indexing ? -1 : 1);
  dimend = (big_endian_indexing ? -1 : dim);
  subregion = subreg;
  done = subregion.empty();
  if (!done)
    {
      pos = subregion.min;
      indx = position_index(pos, region, big_endian_indexing);
      step[dimfast] = 1;
      for (int a = dimfast+dimstep ; a != dimend ; a += dimstep)
	step[a] = step[a-dimstep] * region.size(a-dimstep);
    }
}

// ----------------------------------------------------------------------------
//
offset_type Subarray_Iterator::index() const
  { return indx; }
const IPoint &Subarray_Iterator::position() const
  { return pos; }

// ----------------------------------------------------------------------------
//
bool Subarray_Iterator::finished() const
{
  return done;
}

// ----------------------------------------------------------------------------
//
void Subarray_Iterator::next()
{
  int a;

  for (a = dimfast ; a != dimend ; a += dimstep)
    if (axis_step(a))
      break;

  done = (a == dimend);
}

// ----------------------------------------------------------------------------
//
int Subarray_Iterator::step_size(int a) const
{
  return step[a];
}

// ----------------------------------------------------------------------------
// Step one unit along an axis going back to beginning if at end.
// Return false if jump back to beginning occured.
//
bool Subarray_Iterator::axis_step(int a)
{
  pos[a] += 1;
  indx += step[a];
  if (pos[a] > subregion.max[a])
    {
      pos[a] = subregion.min[a];
      indx -= subregion.size(a) * step[a];
      return false;
    }

  return true;
}

// ----------------------------------------------------------------------------
//
IPoint index_position(offset_type index, const IRegion &region,
		      bool big_endian_indexing)
{
  IPoint p(region.dimension());

  int dimfast = (big_endian_indexing ? region.dimension() - 1 : 0);
  int dimstep = (big_endian_indexing ? -1 : 1);
  int dimend = (big_endian_indexing ? -1 : region.dimension());
  for (int a = dimfast ; a != dimend ; a += dimstep)
    {
      int size = region.size(a);
      p[a] = static_cast<int> (index % size);
      index /= size;
    }

  return p;
}

// ----------------------------------------------------------------------------
//
offset_type position_index(const IPoint &point, const IRegion &region,
			   bool big_endian_indexing)
{
  offset_type index = 0, step = 1;

  int dimfast = (big_endian_indexing ? region.dimension() - 1 : 0);
  int dimstep = (big_endian_indexing ? -1 : 1);
  int dimend = (big_endian_indexing ? -1 : region.dimension());
  for (int a = dimfast ; a != dimend ; a += dimstep)
    {
      index += (point[a] - region.min[a]) * step;
      step *= region.size(a);
    }

  return index;
}

// ----------------------------------------------------------------------------
//
int array_step(int axis, const IRegion &region, bool big_endian_indexing)
{
  int step = 1;

  int dimfast = (big_endian_indexing ? region.dimension() - 1 : 0);
  int dimstep = (big_endian_indexing ? -1 : 1);
  for (int a = dimfast ; a != axis ; a += dimstep)
    step *= region.size(a);

  return step;
}

// ----------------------------------------------------------------------------
//
void little_to_big_endian_indexing(const IRegion &region, float *le, float *be)
{
  //
  // Optimization: Copy data a row at a time along the longest axis.
  //
  IRegion base = region;
  int long_axis = region.longest_axis();
  base.max[long_axis] = base.min[long_axis];

  Subarray_Iterator li(base, region, false);

  int lstep = li.step_size(long_axis);
  int bstep = array_step(long_axis, region, true);
  int row_length = region.size(long_axis);

  for ( ; !li.finished() ; li.next())
    {
      float *from = le + li.index();
      float *to = be + position_index(li.position(), region, true);
      for (int k = 0 ; k < row_length ; ++k)
	to[k * bstep] = from[k * lstep];
    }
}

// ----------------------------------------------------------------------------
//
void big_to_little_endian_indexing(const IRegion &region, float *be, float *le)
{
  int dim = region.dimension();

  size_t volume = region.volume();
  size_t bi = 0, be_step = volume / region.size(0);
  IPoint p = region.min;
  for (size_t li = 0 ; li < volume ; ++li)
    {
      le[li] = be[bi];

      p[0] += 1;
      size_t bstep = be_step;
      bi += bstep;
      for (int a = 0 ; a < dim && p[a] > region.max[a] ; ++a)
	{
	  p[a] = region.min[a];
	  bi -= bstep * region.size(a);
	  if (a+1 == dim)
	    return;
	  p[a+1] += 1;
	  bstep /= region.size(a+1);
	  bi += bstep;
	}
    }
}

// ----------------------------------------------------------------------------
//
void copy_array(const void *source, void *destination, int element_size,
		const IRegion &rs, const IRegion &rd)
{
  IRegion common = rs;
  common.clip(rd);

  Subarray_Iterator si(common, rs, true), di(common, rd, true);
  for ( ; !si.finished() ; si.next(), di.next())
    memcpy((char *) destination + di.index() * element_size,
	   (char *) source + si.index() * element_size,
	   element_size);
}
