// ----------------------------------------------------------------------------
//
#include <stdlib.h>		// Use exit()
#include <fstream>		// use ifstream

#include "command.h"		// Use Command_Dispatcher
#include "contour.h"		// close_contour_cache()
#include "paths.h"		// Use installation_path(), SPARKY_LIB
#include "format.h"		// Use open_sparky_file(), save_project()
#include "memalloc.h"		// use new, allocation_report()
#include "notifier.h"		// use initialize_notifier(), close_notifier()
#include "ornament.h"		// use initialize_ornament_selection(),
				//     close_ornament_selection()
#include "print.h"		// Use print_init()
#include "project.h"		// Use Project
#include "reporter.h"		// use Reporter
#include "savefile.h"		// Use SaveFile
#include "session.h"
#include "stringc.h"		// Use Stringy
#include "system.h"		// Use core_dump(), catch_signal()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uimain.h"		// Use show_main_window(), query()
#include "uimode.h"		// Use show_mode_buttons()
#include "utility.h"		// Use fatal_error()
#include "winsystem.h"		// Use initialize_window_system(), event_loop()

extern "C" void delete_session_cb(CB_Data session);
static void show_windows(Session &);
static void load_commandline_files(Session &, int ac, char **av);
static void catch_signals();
static void directories_check(Session &s);
static bool CheckDirectories(const Stringy &home);
static bool BuildDirectories(const Stringy &home);
static bool create_directory(const Stringy &s);

// ----------------------------------------------------------------------------
//
class Session_Reporter : public Reporter
{
public:
  Session_Reporter(Session &);
  virtual ~Session_Reporter() {};
  virtual void message(const char *fmt, ...);
  virtual void warning(const char *fmt, ...);
private:
  Session &session;
};

// ----------------------------------------------------------------------------
//
Session::Session(int *argc, char *argv[])
{
  this->delete_on_exit = false;

  init_systems();

  Stringy resourcedir = installation_path(SPARKY_LIB);
  this->ws = new WinSys(SPARKY_RESOURCE, "sparky", resourcedir, argc, argv);
  this->proj = new Project(*this);

  show_windows(*this);
  load_commandline_files(*this, *argc, argv);
}

// ----------------------------------------------------------------------------
// When starting a session from Python, Tkinter creates the Tcl_Interp and
// passes it to Sparky to use.
//
Session::Session(void *tcl_interp)
{
  this->delete_on_exit = true;

  init_systems();

  Stringy resourcedir = installation_path(SPARKY_LIB);
  this->ws = new WinSys(SPARKY_RESOURCE, "sparky", resourcedir,	tcl_interp);
  this->proj = new Project(*this);

  show_windows(*this);
}

// ----------------------------------------------------------------------------
//
void Session::init_systems()
{
  catch_signals();

  this->delete_queued = false;
  this->nt = new Notifier();
  this->cd = new Command_Dispatcher(*this);
  this->dt = new Dialog_Table();
  this->rep = new Session_Reporter(*this);
}

// ----------------------------------------------------------------------------
//
Session::~Session()
{
  notifier().send_notice(nt_will_delete_session, this);

  SaveFile::remove_auto_backups();

  window_system().unshow_all_dialogs();

  //
  // Free all state.
  //
  project().unload();
  delete cd;				// close command dispatcher
  cd = NULL;
  delete rep;				// close reporter
  rep = NULL;
  window_system().delete_dialogs_except_main();
  delete_main_dialog(*this);
  delete dt;				// close dialog table
  dt = NULL;
  delete proj;				// close project
  proj = NULL;
  delete ws;				// close window system
  ws = NULL;
  delete nt;				// close notifier
  nt = NULL;

  //
  // Any memory still allocated is a memory leak.
  //
  allocation_report();
}

// ----------------------------------------------------------------------------
// The session can end in two ways depending on whether it is using Python
// or not.  If it is using Python then a callback is queued to delete the
// session.  I don't delete it here and now because I am being called by
// through a long string of session component methods functions.  If Python
// is not being used then the Session has been created as an automatic
// variable so I don't delete it.  Instead a request is made to exit the
// WinSys event loop, after which the automatic Session variable will be
// destroyed.
//
void Session::end_session()
{
  reporter().message("Quitting.\n");

  if (delete_on_exit)
    {
      if (!delete_queued)
	{
	  delete_queued = true;
	  window_system().queue_callback(delete_session_cb, this);
	}
    }
  else
    window_system().exit_event_loop();
}

// ----------------------------------------------------------------------------
//
extern "C" void delete_session_cb(CB_Data session)
{
  delete (Session *) session;
}

// ----------------------------------------------------------------------------
//
bool Session::ok_to_end_session()
{
  if (window_system().modal_dialog_shown())
    return false;

  if (!project().save_file().NeedsSaving() &&
      project().unsaved_spectra() == 0)
    return true;

  int answer = query(*this, "Do you want to save your changes?\n",
		     "Save", "Quit", "Cancel");
  if (answer == 1)
    {
      save_project(*this, false);
      return false;
    }
  else if (answer == 2)
    return true;
  else if (answer == 3 || answer == 0)
    return false;

  return false;		// no answer because query dialog destroyed
}

// ----------------------------------------------------------------------------
//
Notifier &Session::notifier()
  { return *nt; }
Project &Session::project()
  { return *proj; }
Dialog_Table &Session::dialog_table()
  { return *dt; }
WinSys &Session::window_system()
  { return *ws; }
Command_Dispatcher &Session::command_dispatcher()
  { return *cd; }
Reporter &Session::reporter()
  { return *rep; }

// ----------------------------------------------------------------------------
//
Session_Reporter::Session_Reporter(Session &s) : session(s)
{
}

// ----------------------------------------------------------------------------
//
void Session_Reporter::message(const char *format, ...)
{
  va_list args;
  va_start(args, format);
  Stringy msg = va_formatted_string(format, args);
  va_end(args);

  show_message(session, msg);
}

// ----------------------------------------------------------------------------
//
void Session_Reporter::warning(const char *format, ...)
{
  va_list args;
  va_start(args, format);
  Stringy msg = va_formatted_string(format, args);
  va_end(args);

  show_warning(session, msg);
}

// ----------------------------------------------------------------------------
//
static void show_windows(Session &s)
{
  show_main_window(s);
  show_mode_buttons(s);
  set_pointer_mode(s, MODE_ZOOM);

  directories_check(s);
}

// ----------------------------------------------------------------------------
//
static void load_commandline_files(Session &s, int ac, char **av)
{
  Stringy error_msg;
  for (int k = 1 ; k < ac ; ++k)
    if (!open_sparky_file(s, av[k], &error_msg))
      if (! error_msg.is_empty())
	s.reporter().warning(error_msg.cstring());
}

static void interrupt_signal_handler(int)
  { exit(2); }

// ----------------------------------------------------------------------------
//
static void core_dump_handler(int)
  { core_dump(); }

/*
 * Catch signals for nice cleanup
 */
static void catch_signals()
{
  catch_signal(SIGINT, interrupt_signal_handler);
  catch_fatal_errors(core_dump_handler);

  //
  // SIGPIPE is sent when a write is attempted on a pipe and no process
  // has the other end open for reading.  Sparky can start a postscript
  // previewer or molecular display child process with communication
  // through a pipe.  It is the responsibility of the controlling code
  // to handle write errors.  fprintf() returns < 0 with errno = EPIPE.
  //
  ignore_broken_pipes();
}

// ----------------------------------------------------------------------------
// Check that the Sparky home directory exists.
// If not, ask if user wishes to create it.
//
static void directories_check(Session &s)
{
  Stringy home = home_directory();

  Reporter &rr = s.reporter();
  if (home.is_empty()) {
    const char *msg =
      "Couldn't locate your Sparky home directory.\n\n"
      "When you open a file Sparky starts by showing you\n"
      "the contents of the Save or Projects or Lists\n"
      "subdirectory in your personal Sparky home directory.\n"
      "Under Unix this directory is usually named Sparky\n"
      "and is in your Unix home directory.  Under Windows\n"
      "there is no predefined location for the Sparky home\n"
      "directory.  You can specify where the Sparky home\n"
      "directory should be by setting the SPARKYHOME environment\n"
      "variable.  This is done in Windows by clicking on System\n"
      "under the Control Panel, choosing the Environment tab,\n"
      "and entering SPARKYHOME as the name of the variable, and\n"
      "a directory as the value.  If you don't set SPARKYHOME the\n"
      "only consequence will be that file dialogs will not start in\n"
      "your data directories by default.";
    rr.warning(msg);
  }
  else if (!CheckDirectories(home)) {
    Stringy msg = formatted_string(
		   "When you open a file Sparky starts by showing you\n"
		   "the contents of the Save or Projects or Lists\n"
		   "subdirectory in\n"
		   "\n"
		   "   %s\n"
		   "\n"
		   "These directories don't all exist.\n"
		   "Would you like to create the missing directories?\n"
		   "\n"
		   "(You can specify a different default directory by\n"
		   "setting the SPARKYHOME environment variable before\n"
		   "starting the program.)",
		   home.cstring());
    if (query(s, msg.cstring(), "Yes", "No") == 1)
      if (!BuildDirectories(home))
	rr.warning("Sparky could not create its directories.");
  }
}

// ----------------------------------------------------------------------------
//
static bool CheckDirectories(const Stringy &home)
{
  return (is_directory(home) 
	  && is_directory(file_path(home, SPARKY_SAVE))
	  && is_directory(file_path(home, SPARKY_LIST))
	  && is_directory(file_path(home, SPARKY_PROJECT)));
}

// ----------------------------------------------------------------------------
// Make Sparky home directory and Save, List and Projects subdirectories.
//
static bool BuildDirectories(const Stringy &home)
{
  return (create_directory(home) 
	  && create_directory(file_path(home, SPARKY_SAVE))
	  && create_directory(file_path(home, SPARKY_LIST))
	  && create_directory(file_path(home, SPARKY_PROJECT)));
}

// ----------------------------------------------------------------------------
//
static bool create_directory(const Stringy &s)
{
  return is_directory(s) || make_directory(s);
}

// ----------------------------------------------------------------------------
//
void request_coredump(Session &s)
{
  if (query(s, "Are you sure you want Sparky to dump core?\n",
	    "Cancel Coredump", "Dump Away") == 2)
    core_dump();
}
