/*
 * Project.C:		Implementation of the Project class.
 *
 * The Project class provides multiple-spectrum support to make it easier
 * to share information between spectra.
 *
 */
#include <stdlib.h>		// Use atof()
#include <string.h>		// Use strrchr(), strcat(), ...

#include "assigncopy.h"	// Use assignment_copy()
#include "atom.h"		// use Atom
#include "axismap.h"
#include "condition.h"
#include "format.h"		// Use save_project()
#include "group.h"		// Use Group
#include "label.h"		// Use Label()
#include "list.h"
#include "memalloc.h"		// use new()
#include "memcache.h"		// use Memory_Cache
#include "molecule.h"
#include "notifier.h"		// use Notifier
#include "num.h"
#include "ornament.h"		// use ornament_copy()
#include "paths.h"		// use sparky_directory()
#include "peak.h"		// Use Peak
#include "peakgp.h"		// Use PeakGp
#include "project.h"
#include "reporter.h"		// use Reporter
#include "resonance.h"		// Use Resonance
#include "session.h"		// use Session
#include "spectrum.h"
#include "system.h"		// Use remove_file(), file_path(),file_exists()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uimain.h"		// Use query()
#include "undo.h"		// Use Undo
#include "utility.h"		// Use fatal_error()
#include "uiview.h"		// Use View
#include "winsystem.h"		// Use add_timer(), remove_timer()

class Overlay;

#define MAX_LINE_LENGTH		4096

static void free_view_axes(List *view_axes);
static view_axis *find_view_axis(const List &view_axes, const view_axis &va);
static void synchronize_match_view_axis(const view_axis &from, List *to_axes);
static void match_axis(View *view, int axis, View *from, int from_axis);
static List *sublist_containing(const view_axis &va,
				const List &view_axis_lists);
static void remove_overlays(View *v, List &overlays);
static Overlay *find_overlay(View *from, View *onto, const List &overlays);
static void delete_from_list_cb(void *list, void *object);

/*
 * Construct a default project.
 */
Project::Project(Session &s) :
  view_regions(s.command_dispatcher()),
  ses(s),
  mSaveFile(s, SAVEFILE_PROJECT, true)
{
	backup_interval = 0;

	mInit();

	Notifier &n = s.notifier();
	n.notify_me(nt_created_ornament, ornament_created_cb, this);
	n.notify_me(nt_changed_ornament, ornament_changed_cb, this);
	n.notify_me(nt_will_delete_ornament, will_delete_ornament_cb, this);
	n.notify_me(nt_selected_ornament, ornament_selection_cb, this);
	n.notify_me(nt_unselected_ornament, ornament_selection_cb, this);
	n.notify_me(nt_created_view, view_created_cb, this);
	n.notify_me(nt_will_delete_view, will_delete_view_cb, this);
	n.notify_me(nt_deleted_view_of_spectrum, deleted_view_cb, this);
	n.notify_me(nt_created_spectrum, spectrum_created_cb, this);
	n.notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);
	n.notify_me(nt_created_molecule, molecule_created_cb, this);
	n.notify_me(nt_will_delete_molecule, will_delete_molecule_cb, this);
}

/*
 * Initialize a Project.
 * This routine is called from the constructor and unload()
 */
void Project::mInit()
{
	selection_changed = false;

	assignment_copy_from = NULL;

	mUndo = NULL;

	mem_cache = new Memory_Cache(0);
	set_cache_size(preferences.memory_cache_size);

	contours = create_contour_cache(mem_cache, session().notifier());

	use_auto_backups = false;
	ignore_auto_backups = false;
	remove_auto_backups = false;

	set_auto_backup(preferences.auto_backup_interval);
}

/*
 * There is only one Project and it gets reset when the current
 * Project is unloaded.
 */
void Project::unload()
{
	List slist = spectrum_list();			// Delete spectra
	for (int si = 0 ; si < slist.size() ; ++si)
	  delete (Spectrum *) slist[si];

	List mlist = molecule_list();			// Delete molecules
	for (int mi = 0 ; mi < mlist.size() ; ++mi)
	  delete (Molecule *) mlist[mi];

	mResetUndo();					// Reset the Undo state

	delete_contour_cache(contours);
	contours = NULL;

	delete mem_cache;
	mem_cache = NULL;

	view_regions.delete_regions();			// Erase named regions

	SaveFile::remove_auto_backups();

	save_file().set_path("");
	save_file().NeedsSaving(false);

	mInit();				// Reinitialize the Project
}

Project::~Project()
{
  Notifier &n = session().notifier();
  n.send_notice(nt_will_delete_project, this);

  unload();

  n.dont_notify_me(nt_created_ornament, ornament_created_cb, this);
  n.dont_notify_me(nt_changed_ornament, ornament_changed_cb, this);
  n.dont_notify_me(nt_will_delete_ornament, will_delete_ornament_cb, this);
  n.dont_notify_me(nt_selected_ornament, ornament_selection_cb, this);
  n.dont_notify_me(nt_unselected_ornament, ornament_selection_cb, this);
  n.dont_notify_me(nt_created_view, view_created_cb, this);
  n.dont_notify_me(nt_will_delete_view, will_delete_view_cb, this);
  n.dont_notify_me(nt_deleted_view_of_spectrum, deleted_view_cb, this);
  n.dont_notify_me(nt_created_spectrum, spectrum_created_cb, this);
  n.dont_notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);
  n.dont_notify_me(nt_created_molecule, molecule_created_cb, this);
  n.dont_notify_me(nt_will_delete_molecule, will_delete_molecule_cb, this);

  WinSys &ws = session().window_system();
  ws.remove_work_procedure(send_selection_changed_notice, this);
  ws.remove_work_procedure(send_ornaments_changed_notice, this);

  delete_contour_cache(contours);
  contours = NULL;

  delete mem_cache;
  mem_cache = NULL;
}

// ----------------------------------------------------------------------------
//
Session &Project::session() const
{
  return ses;
}

// ----------------------------------------------------------------------------
//
SaveFile &Project::save_file()
  { return mSaveFile; }

// ----------------------------------------------------------------------------
//
Stringy Project::sparky_path(const Stringy &dir, const Stringy &file) const
{
  Stringy projpath = mSaveFile.path();
  Stringy filedir = (projpath.is_empty() ?
		     sparky_directory(dir) :
		     file_path(file_directory(file_directory(projpath)), dir));
  return file_path(filedir, file);
}
Stringy Project::list_path(const Stringy &file) const
  { return sparky_path(SPARKY_LIST, file); }
Stringy Project::project_path(const Stringy &file) const
  { return sparky_path(SPARKY_PROJECT, file); }
Stringy Project::save_path(const Stringy &file) const
  { return sparky_path(SPARKY_SAVE, file); }

// ----------------------------------------------------------------------------
// The file name is the spectrum name with ".ucsf" stripped off
// concatenated with ".save".  A numeric suffix is added if
// necessary to make an unused file name.
//
Stringy Project::new_save_file_path(const Stringy &name) const
{
  Stringy file = remove_suffix(name, ".ucsf") +".save";
  Stringy basepath = save_path(file);
			       
  Stringy path = basepath;
  if (file_exists(path))
    {
      for (int i = 1; ; i++)
	{
	  path = basepath + formatted_string("%d", i);
	  if (! file_exists(path))
	    break;
	}
    }

  return path;
}

// ----------------------------------------------------------------------------
//
AttachedData &Project::saved_values()
  { return attached_data; }

/*
 * Return a count of the number of spectra that need saving.
 */
int Project::unsaved_spectra()
{
	int		count = 0;

	const List &slist = spectrum_list();
	for (int si = 0 ; si < slist.size() ; ++si)
		if (((Spectrum *) slist[si])->save_file().NeedsSaving())
			count++;

	return count;
}

/*
 * Make a new project and restore it (if <andRestore> is true).
 */
bool Project::unload_query()
{
	if (spectrum_list().size() > 0) {
		char	buf[MAX_LINE_LENGTH];

		sprintf(buf,
			"Are you sure you want to unload the current project?");
		int n = unsaved_spectra();
		if (n > 0) {
			sprintf(buf + strlen(buf),
				"\n\nThere %s %d unsaved spect%s.",
				n == 1 ? "is" : "are",
				n,
				n == 1 ? "um" : "a");
		}
		if (query(session(), buf,
			  "Yes, unload this project", "No, cancel unload")
		    != 1)
			return false;

		unload();
	}

	return true;
}

// ----------------------------------------------------------------------------
//
void Project::set_auto_backup(int seconds)
{
  WinSys &ws = session().window_system();
  if (backup_interval > 0)
    ws.remove_timer(backup_interval, auto_backup_cb, this);

  preferences.auto_backup_interval = seconds;
  backup_interval = 1000 * (unsigned long) seconds;  // milliseconds

  if (backup_interval > 0)
    ws.add_timer(backup_interval, auto_backup_cb, this);
}

// ----------------------------------------------------------------------------
//
void Project::auto_backup_cb(CB_Data project)
{
  Project *proj = (Project *) project;

  save_project(proj->session(), true);

  if (proj->backup_interval > 0)
    {
      WinSys &ws = proj->session().window_system();
      ws.add_timer(proj->backup_interval, auto_backup_cb, proj);
    }
}

// ----------------------------------------------------------------------------
//
bool Project::use_backup_file(const Stringy &path)
{
  if (!use_auto_backups && !ignore_auto_backups && !remove_auto_backups)
    {
      Stringy msg = formatted_string("Auto backup file found\n"
				     "  %s\n"
				     "What should be done with backup files?",
				     path.cstring());
      int choice = query(session(), msg,
			 "Use them", "Ignore them", "Remove them");
      if (choice == 1)
	use_auto_backups = true;
      else if (choice == 2 || choice == 0)
	ignore_auto_backups = true;
      else if (choice == 3)
	remove_auto_backups = true;
    }

  if (remove_auto_backups)
    {
      remove_file(path);
      Reporter &rr = session().reporter();
      rr.message("Removed auto backup file %s\n", path.cstring());
    }

  return use_auto_backups;
}

// ----------------------------------------------------------------------------
//
view_axis::view_axis(View *view, int axis)
{
  v = view;
  a = axis;
}

// ----------------------------------------------------------------------------
//
view_axis::view_axis(const view_axis &va)
{
  v = va.view();
  a = va.axis();
}

// ----------------------------------------------------------------------------
//
View *view_axis::view() const { return v; }
int view_axis::axis() const { return a; }
bool view_axis::operator==(const view_axis &va) const
  { return view() == va.view() && axis() == va.axis(); }

// ----------------------------------------------------------------------------
//
void free_view_axis_list_entries(List &view_axis_list)
{
  for (int vai = 0 ; vai < view_axis_list.size() ; ++vai)
    delete (view_axis *) view_axis_list[vai];
  view_axis_list.erase();
}

// ----------------------------------------------------------------------------
//
void Project::synchronize_view_axes(const view_axis &va1, const view_axis &va2)
{
  List *s1 = sublist_containing(va1, sync_lists);
  List *s2 = sublist_containing(va2, sync_lists);

  if (s1 && s2)
    {
      if (s1 != s2)
	{
	  s1->append(*s2);
	  sync_lists.erase(s2);
	  delete s2;
	}
    }
  else if (s1)
    s1->append(new view_axis(va2));
  else if (s2)
    s2->append(new view_axis(va1));
  else
    {
      List *s = new List();
      s->append(new view_axis(va1));
      s->append(new view_axis(va2));
      sync_lists.append(s);
    }

  save_file().NeedsSaving(true);
}

// ----------------------------------------------------------------------------
//
static void free_view_axes(List *view_axes)
{
  if (view_axes)
    for (int ai = 0 ; ai < view_axes->size() ; ++ai)
      delete (view_axis *) (*view_axes)[ai];
}

// ----------------------------------------------------------------------------
//
void Project::unsynchronize_view_axis(const view_axis &va)
{
  List *s = sublist_containing(va, sync_lists);

  if (s)
    if (s->size() == 2)
      {
	free_view_axes(s);
	sync_lists.erase(s);
	delete s;
      }
    else
      {
	view_axis *list_va = find_view_axis(*s, va);
	delete list_va;
	s->erase(list_va);
      }

  save_file().NeedsSaving(true);
}

// ----------------------------------------------------------------------------
//
static view_axis *find_view_axis(const List &view_axes, const view_axis &va)
{
  for (int ai = 0 ; ai < view_axes.size() ; ++ai)
    {
      view_axis *list_va = (view_axis *) view_axes[ai];
      if (va == *list_va)
	return list_va;
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
const List *Project::synchronized_view_axes(const view_axis &va)
{
  return sublist_containing(va, sync_lists);
}

// ----------------------------------------------------------------------------
//
const List &Project::synchronized_view_axis_sets() const
{
  return sync_lists;
}

// ----------------------------------------------------------------------------
//
void Project::synchronize_match_views(View *v)
{
  for (int a = 0 ; a < v->spectrum()->dimension() ; ++a)
    {
      view_axis from = view_axis(v, a);
      List *to_axes = sublist_containing(from, sync_lists);
      synchronize_match_view_axis(from, to_axes);
    }
}

// ----------------------------------------------------------------------------
//
static void synchronize_match_view_axis(const view_axis &from, List *to_axes)
{
  if (to_axes)
    for (int ai = 0 ; ai < to_axes->size() ; ++ai)
      {
	view_axis *to = (view_axis *) (*to_axes)[ai];
	if (!(*to == from))
	  match_axis(to->view(), to->axis(), from.view(), from.axis());
      }
}

// ----------------------------------------------------------------------------
// Adjust the pixel size of view to match that of another view.
// The pixel size along every axis of view is scaled by the same factor.
//
static void match_axis(View *view, int axis, View *from, int from_axis)
{
  SPoint center = view->center();
  center[axis] = from->center(from_axis);

  SPoint pixel_size = view->pixel_size();
  double factor = from->pixel_size(from_axis) / pixel_size[axis];
  pixel_size *= factor;

  view->View_Window::set_view(center, pixel_size, false);
}

// ----------------------------------------------------------------------------
//
static List *sublist_containing(const view_axis &va,
				const List &view_axis_lists)
{
  for (int i = 0 ; i < view_axis_lists.size() ; ++i)
    {
      List *view_axes = (List *) view_axis_lists[i];
      if (find_view_axis(*view_axes, va))
	return view_axes;
    }

  return NULL;
}

class Overlay
{
public:
  Overlay(View *from, View *onto)
    { this->from = from; this->onto = onto; }
  View *from, *onto;
};

// ----------------------------------------------------------------------------
//
void Project::add_overlay(View *from, View *onto)
{
  if (from != onto && find_overlay(from, onto, mOverlays) == NULL)
    {
      mOverlays.append(new Overlay(from, onto));
      onto->Drawing::erase();
    }
}

// ----------------------------------------------------------------------------
//
void Project::remove_overlay(View *from, View *onto)
{
  Overlay *overlay = find_overlay(from, onto, mOverlays);

  if (overlay)
    {
      mOverlays.erase(overlay);
      delete overlay;
      onto->Drawing::erase();
    }
}

// ----------------------------------------------------------------------------
//
static void remove_overlays(View *v, List &overlays)
{
  List olist = overlays;
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Overlay *overlay = (Overlay *) olist[oi];
      if (overlay->from == v || overlay->onto == v)
	{
	  overlays.erase(overlay);
	  delete overlay;
	}
    }
}

// ----------------------------------------------------------------------------
//
static Overlay *find_overlay(View *from, View *onto, const List &overlays)
{
  for (int oi = 0 ; oi < overlays.size() ; ++oi)
    {
      Overlay *overlay = (Overlay *) overlays[oi];
      if (overlay->from == from && overlay->onto == onto)
	return overlay;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
List Project::overlaid_views(View *onto)
{
  List vlist;
  for (int oi = 0 ; oi < mOverlays.size() ; ++oi)
    {
      Overlay *overlay = (Overlay *) mOverlays[oi];
      if (overlay->onto == onto)
	vlist.append(overlay->from);
    }
  return vlist;
}

// ----------------------------------------------------------------------------
// Install a work procedure that will generate the
// nt_selected_ornaments_changed notice.
// This notice is generally more useful than the nt_selected_ornament and
// nt_unselected_ornament notices because these generate a notice for each
// ornament.  Most dialogs do not want to update 3000 times if all 3000
// peaks in a spectrum are selected or unselected.
//
void Project::ornament_selection_cb(void *project, void *)
{
  Project *proj = (Project *) project;

  if (!proj->selection_changed)
    {
      proj->selection_changed = true;
      WinSys &ws = proj->session().window_system();
      ws.add_work_procedure(send_selection_changed_notice, project);
    }
}

// ----------------------------------------------------------------------------
//
void Project::send_selection_changed_notice(CB_Data project)
{
  Project *proj = (Project *) project;

  proj->selection_changed = false;
  Notifier &n = proj->session().notifier();
  n.send_notice(nt_selected_ornaments_changed, NULL);
}

// ----------------------------------------------------------------------------
// The deleted_ornaments table is used in conjunction with the
// changed_ornaments table to batch ornament changed notices.
// Simply deleting an ornament from the changed ornaments table when a
// will delete ornament notice is received is insufficient.  After the
// will delete notice the ornament also generates change notices which
// causes it to be added back into the changed table.  I could have
// remedied this by making not ornament changed notices occur after the
// will delete ornament notice.  But this would be very bug prone as
// the code is modified.
//
void Project::ornament_created_cb(void *project, void *ornament)
{
  Project *proj = (Project *) project;
  proj->deleted_ornaments.remove(ornament);
}

// ----------------------------------------------------------------------------
// Remember ornaments that have changed and install a work procedure
// to generate the nt_changed_ornaments notice.
//
void Project::ornament_changed_cb(void *project, void *ornament)
{
  Project *proj = (Project *) project;

  bool first_change = (proj->changed_ornaments.entry_count() == 0);
  proj->changed_ornaments.insert(ornament, NULL);
  if (first_change)
    {
      WinSys &ws = proj->session().window_system();
      ws.add_work_procedure(send_ornaments_changed_notice, project);
    }
}

// ----------------------------------------------------------------------------
//
void Project::send_ornaments_changed_notice(CB_Data project)
{
  Project *proj = (Project *) project;

  List olist = proj->changed_ornaments.keys();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (proj->deleted_ornaments.find(op, NULL))
	proj->changed_ornaments.remove(op);
    }
  olist = proj->changed_ornaments.keys();
  proj->changed_ornaments.erase();
  proj->deleted_ornaments.erase();

  Notifier &n = proj->session().notifier();
  n.notify_me(nt_will_delete_ornament, delete_from_list_cb, &olist);
  n.send_notice(nt_changed_ornaments, &olist);
  n.dont_notify_me(nt_will_delete_ornament, delete_from_list_cb, &olist);
}

// ----------------------------------------------------------------------------
//
static void delete_from_list_cb(void *list, void *object)
{
  ((List *) list)->erase(object);
}

// ----------------------------------------------------------------------------
// Remove references to ornament held by Project.
//
void Project::will_delete_ornament_cb(void *project, void *ornament)
{
  Project *proj = (Project *) project;
  Ornament *op = (Ornament *) ornament;

  proj->deleted_ornaments.insert(op, NULL);

  proj->ornament_copy_list.erase(op);

  if (is_crosspeak(op)) {
    CrossPeak	*xp = (CrossPeak *) op;
    proj->assignment_copy_list.erase(xp);
  }
}

// ----------------------------------------------------------------------------
// When a view is created add it to the project.
//
void Project::view_created_cb(void *project, void *view)
{
  Project *proj = (Project *) project;
  View *v = (View *) view;
  v->set_name(proj->unique_view_name(v->name(), v->is_top_level_window()));
  proj->mViews.append(v);
  Notifier &n = proj->session().notifier();
  n.send_notice(nt_added_view_to_project, v);
}

/*
 * When a View is deleted, remove any references the project is holding to it.
 */
void Project::will_delete_view_cb(void *project, void *view)
{
  Project *proj = (Project *) project;
  View *v = (View *) view;
  Spectrum *sp = v->spectrum();
 
  for (int a = 0 ; a < sp->dimension() ; ++a)
    proj->unsynchronize_view_axis(view_axis(v, a));

  remove_overlays(v, proj->mOverlays);

  proj->mViews.erase(v);
  Notifier &n = proj->session().notifier();
  n.send_notice(nt_removed_view_from_project, v);
}

// ----------------------------------------------------------------------------
// If spectrum has no views delete it.
// If it is not in the list of spectra then it has already been destroyed.
//
void Project::deleted_view_cb(void *project, void *spectrum)
{
  Project *proj = (Project *) project;
  Spectrum *sp = (Spectrum *) spectrum;

  if (proj->view_list(sp).size() == 0 && proj->spectrum_list().contains(sp))
    delete sp;
}

// ----------------------------------------------------------------------------
//
bool Project::assignment_copy(Spectrum *from)
{
  assignment_copy_list = from->selected_crosspeaks();

  Reporter &rr = session().reporter();
  rr.message("Assignment copy: %d peak%s selected.\n",
	     assignment_copy_list.size(),
	     (assignment_copy_list.size() > 1 ? "s" : ""));

  return assignment_copy_list.size() > 0;
}

// ----------------------------------------------------------------------------
//
void Project::assignment_paste(Spectrum *to, bool label)
{
  Reporter &rr = session().reporter();
  if (assignment_copy_list.size() == 0)
    rr.warning("Assignment paste: No assignments to copy.\n");
  else
    ::assignment_copy(assignment_copy_list, to, label, rr);
}

// ----------------------------------------------------------------------------
//
Spectrum *Project::assignment_copy_from_spectrum()
  { return assignment_copy_from; }
void Project::set_assignment_copy_from_spectrum(Spectrum *sp)
  { assignment_copy_from = sp; }

// ----------------------------------------------------------------------------
//
void Project::assignment_delete(Spectrum *sp)
{
  List list = sp->selected_crosspeaks();
  for (int pi = 0 ; pi < list.size() ; ++pi)
    ((CrossPeak *) list[pi])->DeleteAssignment();
}

// ----------------------------------------------------------------------------
//
bool Project::ornament_copy(Spectrum *from)
{
  ornament_copy_list = from->selected_ornaments();

  Reporter &rr = session().reporter();
  rr.message("Ornament copy: %d ornament%s selected.\n",
	     ornament_copy_list.size(),
	     (ornament_copy_list.size() > 1 ? "s" : ""));

  return ornament_copy_list.size() > 0;
}

// ----------------------------------------------------------------------------
//
static Spectrum *ornament_list_spectrum(const List &ornaments)
{
  return (ornaments.size() > 0 ?
	  ((Ornament *) ornaments[0])->spectrum() :
	  NULL);
}

// ----------------------------------------------------------------------------
//
void Project::ornament_paste(Spectrum *to)
{
  Axis_Map axismap;
  Spectrum *from = ornament_list_spectrum(ornament_copy_list);

  Reporter &rr = session().reporter();
  if (ornament_copy_list.size() == 0)
    rr.warning("Ornament paste: No ornaments to copy.\n");
  else if (identity_axis_map(from, to, &axismap) ||
	   unique_axis_map(from, to, &axismap))
    {
      unselect_all_ornaments();
      List copies = ::ornament_copy(ornament_copy_list, to, axismap);
      select_ornaments(copies);
    }
  else
    rr.warning("Ornament Copy: "
	       "Couldn't map axes of spectrum %s to spectrum %s\n",
	       from->name().cstring(), to->name().cstring());
}

/*
 * The general Undo object is used:
 *	o)	to undo ornament move
 *	o)	to undo ornament deletion
 *	o)	to undo peak integration
 */

/*
 * The Undo object is reset by deleting it.
 */
void Project::mResetUndo(void)
{
	if (mUndo) {
		delete mUndo;
		mUndo = NULL;
	}
}

/*
 * Return the UndoIntegrate object
 */
Undo *Project::MakeUndoMove()
{
	mResetUndo();
	mUndo = new Undo(session(), "", "moved back");
	return mUndo;
}

/*
 * Return the UndoIntegrate object
 */
Undo *Project::MakeUndoIntegrate(const List &peaks)
{
  mResetUndo();
  mUndo = new Undo(session(), "", "unintegrated");

  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    mUndo->add((Peak *) peaks[pi]);
  mUndo->finish();

  return mUndo;
}

/*
 * Return the UndoDelete object
 */
Undo *Project::MakeUndoDelete(void)
{
	mResetUndo();
	mUndo = new Undo(session(), "", "undeleted");
	return mUndo;
}

/*
 * Perform the undo operation, with the command called from View <fp>.
 */
bool Project::DoUndo()
{
	/*
	 * The undo only happens once: i.e. there's no undo, undo capability.
	 */
	bool status;
	if (mUndo)
	  {
	    status = mUndo->undo();
	    mResetUndo();
	  }
	else
	  {
	    Reporter &rr = session().reporter();
	    rr.message("Nothing to undo\n");
	    status = false;
	  }

	return status;
}

// ----------------------------------------------------------------------------
//
List Project::selected_ornaments()
{
  List ornaments;
  const List &slist = spectrum_list();
  for (int si = 0 ; si < slist.size() ; ++si)
    {
      Spectrum *sp = (Spectrum *) slist[si];
      ornaments.append(sp->selected_ornaments());
    }
  return ornaments;
}

/*
 * Unselect all ornaments in all spectra.
 */
void Project::unselect_all_ornaments()
{
  List olist = selected_ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    ((Ornament *) olist[oi])->select(false);
}

// ----------------------------------------------------------------------------
//
List Project::selected_crosspeaks()
{
  List xplist;
  List olist = selected_ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (is_crosspeak(op) && !is_peaklet(op))
	xplist.append((CrossPeak *) op);
    }
  return xplist;
}

//
// Return the first selected peak or peakgroup or, if none are found,
// the first selected label that points to a peak or peakgroup.
//
CrossPeak *Project::selected_peak()
{
  List	olist = selected_ornaments();

	for (int oi = 0 ; oi < olist.size() ; ++oi) {
		Ornament *op = (Ornament *) olist[oi];
		if (is_crosspeak(op))
			return (CrossPeak *) op;
	}

	for (int oi = 0 ; oi < olist.size() ; ++oi) {
		Ornament *op = (Ornament *) olist[oi];
		if (op->type() == label)
		  {
		    CrossPeak *xp = ((Label *)op)->attached();
		    if (xp)
		      return xp;
		  }
	}

	return NULL;
}

// ----------------------------------------------------------------------------
// If a unique peak is selected return it otherwise return NULL.
// A peak can be chosen by having it's label chosen.
//
CrossPeak *Project::single_peak_selected()
{
  CrossPeak *xp = NULL;

  List selected = selected_ornaments();
  for (int oi = 0 ; oi < selected.size() ; ++oi)
    {
      Ornament *op = (Ornament *) selected[oi];
      CrossPeak *selected_xp = NULL;

      if (is_crosspeak(op))
	selected_xp = (CrossPeak *) op;
      else if (op->type() == label)
	selected_xp = ((Label *)op)->attached();

      if (selected_xp)
	if (xp && xp != selected_xp)
	  return NULL;			// More than one peak.
	else
	  xp = selected_xp;
    }

  return xp;
}

/*
 * Return the pointer to the named view.
 */
View *Project::find_view(const Stringy &name)
{
  const List &views = view_list();
  for (int vi = 0 ; vi < views.size() ; ++vi)
    {
      View *view = (View *) views[vi];
      if (view->name() == name)
	return view;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
View *Project::spectrum_view(Spectrum *sp) const
{
  const List &views = view_list();
  for (int vi = 0 ; vi < views.size() ; ++vi)
    {
      View *view = (View *) views[vi];
      if (view->spectrum() == sp)
	return view;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
const List &Project::spectrum_list(void) const
  { return mSpectra; }

// ----------------------------------------------------------------------------
//
List Project::view_list() const
  { return mViews; }

// ----------------------------------------------------------------------------
//
List Project::view_list(Spectrum *sp) const
{
  List vlist;

  for (int vi = 0 ; vi < mViews.size() ; ++vi)
    if (((View *) mViews[vi])->spectrum() == sp)
      vlist.append(mViews[vi]);

  return vlist;
}

// ----------------------------------------------------------------------------
//
List Project::view_list(const Condition *c) const
{
  List cviews;

  for (int vi = 0 ; vi < mViews.size() ; ++vi)
    if (((View *) mViews[vi])->spectrum()->condition() == c)
      cviews.append(mViews[vi]);

  return cviews;
}

// ----------------------------------------------------------------------------
//
List Project::top_level_views() const
{
  List vlist;

  for (int vi = 0 ; vi < mViews.size() ; ++vi)
    if (((View *) mViews[vi])->is_top_level_window())
      vlist.append(mViews[vi]);

  return vlist;
}

// ----------------------------------------------------------------------------
//
List Project::molecule_list() const
{
  return mMolecules;
}

// ----------------------------------------------------------------------------
//
List Project::condition_list() const
{
  List slist;
  const List &mlist = molecule_list();
  for (int mi = 0 ; mi < mlist.size() ; ++mi)
    slist.append(((Molecule *) mlist[mi])->condition_list());

  return slist;
}

// ----------------------------------------------------------------------------
// If the proposed view name is already in use, strip the numeric suffix
// and add a new numeric suffix to make an unused name.  For top level views
// find the lowest available suffix number.  The algorithm for this uses
// O(n^2) time.  For non top level views just use a suffix number higher
// than any previously used view suffix number.
//
#define VIEWSEP		'/'	// separator between view and number suffix
Stringy Project::unique_view_name(const Stringy &n, bool top_level)
{
  int suffix;
  Stringy basename;
  number_suffix(n, VIEWSEP, &basename, &suffix);

  Stringy name;
  if (top_level)
    {
      if (find_view(n) == NULL)
	name = n;
      else
	{
	  name = basename;
	  for (suffix = 1 ; find_view(name) ; ++suffix)
	    name = formatted_string("%s%c%d",
				    basename.cstring(), VIEWSEP, suffix);
	}
    }
  else
    {
      Dialog_Table &dt = session().dialog_table();
      int suffix = (int) (long) dt.get_data("highest_used_view_suffix");
      suffix = max(9, suffix);	// reserve suffixes 1-9 for top levels
      suffix += 1;
      dt.set_data("highest_used_view_suffix", (void *) suffix);
      name = formatted_string("%s%c%d", basename.cstring(), VIEWSEP, suffix);
    }

  return name;
}

// ----------------------------------------------------------------------------
//
#define SPECTSEP ':'
Stringy	Project::unique_spectrum_name(const Stringy &name)
{
  if (find_spectrum(name) == NULL)
    return name;

  int suffix;
  Stringy basename;
  number_suffix(name, SPECTSEP, &basename, &suffix);
  
  Stringy new_name = basename;
  for (suffix = 1 ; find_spectrum(new_name) ; ++suffix)
    new_name = formatted_string("%s%c%d", basename.cstring(),
				SPECTSEP, suffix);
  return new_name;
}

/*
 * Return the project entry corresponding to a save file.
 */
Spectrum *Project::find_savefile_spectrum(const Stringy &savefile) const
{
	for (int si = 0 ; si < mSpectra.size() ; ++si) {
		Spectrum *sp = (Spectrum *) mSpectra[si];
		if (same_file(savefile, sp->save_file().path()))
			return sp;
	}

	return NULL;
}

// ----------------------------------------------------------------------------
// When a spectrum is created add it to the project.
//
void Project::spectrum_created_cb(void *project, void *spectrum)
{
  Project *proj = (Project *) project;
  Spectrum *sp = (Spectrum *) spectrum;
  sp->set_name(proj->unique_spectrum_name(sp->name()));
  proj->mSpectra.append(sp);
  proj->save_file().NeedsSaving(true);
  Notifier &n = proj->session().notifier();
  n.send_notice(nt_added_spectrum_to_project, sp);
}

/*
 * When a View is deleted, remove any references the project is holding to it.
 */
void Project::will_delete_spectrum_cb(void *project, void *spectrum)
{
  Project *proj = (Project *) project;
  Spectrum *sp = (Spectrum *) spectrum;

  proj->save_file().NeedsSaving(true);
  if (sp == proj->assignment_copy_from)
    proj->assignment_copy_from = NULL;

  //
  // Remove spectrum from list before deleting views.
  // This prevents the deletion of the last view from
  // redestroying the spectrum.
  //
  proj->mSpectra.erase(sp);

  List vlist = proj->view_list(sp);
  for (int vi = 0 ; vi < vlist.size() ; ++vi)
    delete (View *) vlist[vi];

  Notifier &n = proj->session().notifier();
  n.send_notice(nt_removed_spectrum_from_project, sp);
}

// ----------------------------------------------------------------------------
// When a molecule is created add it to the project.
//
void Project::molecule_created_cb(void *project, void *molecule)
{
  Project *proj = (Project *) project;
  Molecule *m = (Molecule *) molecule;

  if (proj->find_molecule(m->name()))
    fatal_error("Project::add_molecule(): Duplicate molecule.\n");

  proj->mMolecules.append(m);
  proj->save_file().NeedsSaving(true);
  Notifier &n = proj->session().notifier();
  n.send_notice(nt_added_molecule_to_project, m);
}

// ----------------------------------------------------------------------------
// When a molecule is created add it to the project.
//
void Project::will_delete_molecule_cb(void *project, void *molecule)
{
  Project *proj = (Project *) project;
  Molecule *m = (Molecule *) molecule;

  proj->mMolecules.erase(m);
  proj->save_file().NeedsSaving(true);
  Notifier &n = proj->session().notifier();
  n.send_notice(nt_removed_molecule_from_project, m);
}

// ----------------------------------------------------------------------------
//
Condition *Project::default_condition()
{
  return define_condition("", "");
}

// ----------------------------------------------------------------------------
//
Condition *Project::define_condition(const Stringy &molecule_name,
				     const Stringy &condition_name)
{
  Molecule *m = define_molecule(molecule_name);
  Condition *c = m->define_condition(condition_name);

  return c;
}

// ----------------------------------------------------------------------------
//
Condition *Project::find_condition(const Stringy &molecule_name,
				   const Stringy &condition_name)
{
  if (molecule_name.is_empty() && condition_name.is_empty())
    return default_condition();

  Reporter &rr = session().reporter();
  Molecule *m = find_molecule(molecule_name);
  if (!m)
    {
      rr.warning("Molecule %s not found.\n"
		 "Will use default molecule and condition.\n",
		 molecule_name.cstring());
      return default_condition();
    }

  Condition *c = m->find_condition(condition_name);
  if (!c)
    {
      rr.warning("Molecule %s has no condition %s.\n"
		 "Will use default molecule and condition.\n",
		 molecule_name.cstring(), condition_name.cstring());
      return default_condition();
    }

  return c;
}

// ----------------------------------------------------------------------------
//
Molecule *Project::find_molecule(const Stringy &molecule_name)
{
  List molecules = molecule_list();
  for (int mi = 0 ; mi < molecules.size() ; ++mi)
    {
      Molecule *m = (Molecule *) molecules[mi];
      if (m->name() == molecule_name)
	return m;
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
Molecule *Project::define_molecule(const Stringy &molecule_name)
{
  Molecule *m = find_molecule(molecule_name);
  if (m == NULL)
    m = new Molecule(session(), molecule_name);

  return m;
}

// ----------------------------------------------------------------------------
//
Spectrum *Project::find_spectrum(const Stringy &sname) const
{
  const List &slist = spectrum_list();
  for (int si = 0 ; si < slist.size() ; ++si)
    {
      Spectrum *sp = (Spectrum *) slist[si];
      if (sp->fullname() == sname || sp->name() == sname)
	return sp;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
Contour_Cache *Project::contour_cache() const
  { return contours; }
Memory_Cache *Project::memory_cache() const
  { return mem_cache; }

// ----------------------------------------------------------------------------
//
void Project::set_cache_size(int mbytes)
{
  preferences.memory_cache_size = mbytes;

  mbytes = max(1, mbytes);			// Minimum size is 1 Mbyte
  int bytes = (1 << 20) * mbytes;
  mem_cache->resize(bytes);
}

// ----------------------------------------------------------------------------
// Set default preferences.
//
Preferences::Preferences()
{
  prompt_before_overwrite = true;
  key_timeout_interval = 3000;
  auto_backup_interval = 0;
  resize_views = false;
  memory_cache_size = 4;	// Mbytes
  contour_graying = true;
}
