// ----------------------------------------------------------------------------
// Replace new and delete with versions that keep track of all allocated
// blocks for debugging memory leaks.
//
#include <iostream>	// use cerr, endl
#include <stdlib.h>	// use malloc(), free()
#include <string.h>	// use strcmp()

#include "list.h"	// use List
#include "memalloc.h"	// use TRACKING_MEMORY
#include "table.h"	// use Table
#include "utility.h"	// use fatal_error()

#undef new
#undef delete

// ----------------------------------------------------------------------------
//
struct alloc_record
{
  void *pointer;
  size_t size;
  const char *file;
  int line;
};

// ----------------------------------------------------------------------------
//
static bool tracking_memory = TRACKING_MEMORY;		// state variable
static Table *alloc_table = NULL;	// memory tracking state variable
static bool alloc_table_locked = false;	// memory tracking state variable

// ----------------------------------------------------------------------------
//
static void *allocate(size_t size);
static void remember(void *p, size_t size, const char *file, int line);
static void forget(void *p);
static int alloc_record_compare(const void *rec1, const void *rec2);

// ----------------------------------------------------------------------------
//
void *operator new(size_t size) throw(std::bad_alloc)
{
  void *value = allocate(size);
  if (tracking_memory)
    remember(value, size, "unknown", 0);
  return value;
}

// ----------------------------------------------------------------------------
//
static void *allocate(size_t size)
{
  //
  // malloc(0) returns 0.
  // But new of size 0 must return non-zero pointer.
  //
  void *value = malloc(size == 0 ? 1 : size);
  if (value == NULL)
    fatal_error("Sparky: out of memory: allocate(%lu) failed\n", size);
  return value;
}

// ----------------------------------------------------------------------------
//
void *operator new(size_t size, const char *file, int line)
{
  void *value = allocate(size);
  remember(value, size, file, line);
  return value;
}

// ----------------------------------------------------------------------------
//
void *operator new[](size_t size) throw(std::bad_alloc)
{
  void *value = allocate(size);
  if (tracking_memory)
    remember(value, size, "unknown", 0);
  return value;
}

// ----------------------------------------------------------------------------
//
void *operator new[](size_t size, const char *file, int line)
{
  void *value = allocate(size);
  remember(value, size, file, line);
  return value;
}

// ----------------------------------------------------------------------------
//
void operator delete(void *p) throw()
{
  if (tracking_memory)
    forget(p);
  free(p);
}

// ----------------------------------------------------------------------------
//
void operator delete[](void *p) throw()
{
  if (tracking_memory)
    forget(p);
  free(p);
}

// ----------------------------------------------------------------------------
//
static void remember(void *p, size_t size, const char *file, int line)
{
  if (alloc_table_locked)
    return;

  if (p == NULL)
    return;

  alloc_table_locked = true;

  if (alloc_table == NULL)
    alloc_table = new Table();

  alloc_record *r = (alloc_record *) malloc(sizeof(alloc_record));
  r->pointer = p;
  r->size = size;
  r->file = file;
  r->line = line;
  alloc_table->insert(p, r);

  alloc_table_locked = false;
}

// ----------------------------------------------------------------------------
//
static void forget(void *p)
{
  if (alloc_table_locked)
    return;

  if (p == NULL)
    return;

  alloc_table_locked = true;

  TableData r;
  bool found = alloc_table->find(p, &r);
  if (found)
    {
      alloc_table->remove(p);
      free(r);
    }

  alloc_table_locked = false;
}

// ----------------------------------------------------------------------------
//
void allocation_report()
{
  if (alloc_table == NULL)
    return;

  alloc_table_locked = true;

  std::cerr << "Memory allocation using new" << std::endl;

  List records = alloc_table->values();
  records.sort(alloc_record_compare);

  size_t total = 0;
  for (int ri = 0 ; ri < records.size() ; )
    {
      alloc_record *r = (alloc_record *) records[ri];
      size_t count = 0;
      size_t size = 0;
      while (ri < records.size())
	{
	  alloc_record *r2 = (alloc_record *) records[ri];
	  if (r2->line == r->line && r2->file == r->file)
	    {
	      count += 1;
	      size += r2->size;
	      total = total + r2->size;
	    }
	  else
	    break;
	  ri += 1;
	}
      std::cerr << r->file << ":" << r->line
		<< "    " << count << " blocks    "
		<< size << " bytes"
		<< std::endl;
    }
  std::cerr << "Total bytes allocated using new: " << total << std::endl;

  alloc_table_locked = false;
}

// ----------------------------------------------------------------------------
//
static int alloc_record_compare(const void *rec1, const void *rec2)
{
  alloc_record *r1 = (alloc_record *) rec1;
  alloc_record *r2 = (alloc_record *) rec2;

  int fcmp = strcmp(r1->file, r2->file);
  if (fcmp)
    return fcmp;

  int lcmp = compare_ints(r1->line, r2->line);
  if (lcmp)
    return lcmp;

  return compare_ints((int) r1->size, (int) r2->size);
}
