// ----------------------------------------------------------------------------
// Felix NMR data file format
//

#include "binaryIO.h"		// Use read_s32(), ...
#include "blockfile.h"		// Use Block_File
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "memcache.h"		// use Memory_Cache
#include "nmrdata.h"		// Use NMR_Data
#include "spoint.h"		// Use IPoint, SPoint
#include "stringc.h"		// Use Stringy
#include "system.h"		// use seek_position()
#include "utility.h"		// Use fatal_error()

#define MAXFELIXDIM 6
#define FELIX_DATA_START 16384

// ----------------------------------------------------------------------------
// The header format for Felix files has been inferred from examples,
// not documentation.  So it may be wrong.
//
struct Felix_Header
{
  int dim;
  bool complex;
  IPoint start;				// Not certain about this param.
  IPoint data_size;			// Not certain about this param.
  IPoint block_count;			// Not certain about this param.
  IPoint block_step;			// Not certain about this param.
  IPoint block_size;
  IPoint data_step;			// Not certain about this param.
  SPoint observe_freq;			// MHz
  SPoint spectral_width;		// Hz
  SPoint ref_point;			// Index
  SPoint ref_shift;			// Hz
  IPoint axis_display_type;		// 0=none, 1=points, 2=hz, 3=ppm
  List axis_labels;
};

static bool read_felix_header(FILE *felixfile, Felix_Header *h);
static bool read_felix_ipoint(FILE *felixfile, int dim, IPoint *p);
static bool read_felix_spoint(FILE *felixfile, int dim, SPoint *p);
static bool read_string(FILE *fp, int length, Stringy *s);
static SPoint felix_origin_ppm(const Felix_Header &fh);

// ----------------------------------------------------------------------------
//
NMR_Data *felix_nmr_data(const Stringy &path, Memory_Cache *mcache)
{
  FILE *fp = fopen(path.cstring(), "rb");
  if (!fp)
    return NULL;

  Felix_Header fh;
  bool readable = read_felix_header(fp, &fh);

  fclose(fp);
  if (!readable)
    return NULL;

  Block_File *bf = packed_block_file(path, "rb", fh.data_size, fh.block_size,
				     FELIX_DATA_START, false, mcache);
  if (bf == NULL)
    return NULL;

  return block_file_nmr_data(bf, fh.observe_freq, fh.spectral_width,
			     felix_origin_ppm(fh), fh.axis_labels);
}

// ----------------------------------------------------------------------------
//
bool is_felix_nmr_data(const Stringy &path)
{
  FILE *fp = fopen(path.cstring(), "rb");
  if (!fp)
    return false;

  Felix_Header fh;
  bool read_header = read_felix_header(fp, &fh);
  fclose(fp);

  return read_header;
}

// ----------------------------------------------------------------------------
//
static bool read_felix_header(FILE *felixfile, Felix_Header *h)
{
  int dim;
  if (!read_s32(&dim, felixfile) || dim < 2 || dim > MAXFELIXDIM)
    return false;
  h->dim = dim;

  int dtype;
  if (!read_s32(&dtype, felixfile) || (dtype != 1 && dtype != 2))
    return false;
  h->complex = (dtype == 2);

  if (!seek_position(felixfile, 80, 0) ||
      !read_felix_ipoint(felixfile, dim, &h->start) ||
      !read_felix_ipoint(felixfile, dim, &h->data_size) ||
      !read_felix_ipoint(felixfile, dim, &h->block_count) ||
      !read_felix_ipoint(felixfile, dim, &h->block_step) ||
      !read_felix_ipoint(felixfile, dim, &h->block_size) ||
      !read_felix_ipoint(felixfile, dim, &h->data_step) ||
      !read_felix_spoint(felixfile, dim, &h->observe_freq) ||
      !read_felix_spoint(felixfile, dim, &h->spectral_width) ||
      !read_felix_spoint(felixfile, dim, &h->ref_point) ||
      !read_felix_spoint(felixfile, dim, &h->ref_shift) ||
      !read_felix_ipoint(felixfile, dim, &h->axis_display_type))
    return false;

  if (!seek_position(felixfile, 880, 0))
    return false;

  Stringy axis_label;
  for (int a = 0 ; a < dim ; ++a)
    if (read_string(felixfile, 8, &axis_label))
      h->axis_labels.append(new Stringy(standard_nucleus_name(axis_label)));
  else
    return false;

  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_felix_ipoint(FILE *felixfile, int dim, IPoint *p)
{
  int value;
  IPoint values(dim);

  for (int a = 0 ; a < dim ; ++a)
    if (read_s32(&value, felixfile))
      values[a] = value;
    else
      return false;

  *p = values;

  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_felix_spoint(FILE *felixfile, int dim, SPoint *p)
{
  float value;
  SPoint values(dim);

  for (int a = 0 ; a < dim ; ++a)
    if (read_f32(&value, felixfile))
      values[a] = value;
    else
      return false;

  *p = values;

  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_string(FILE *fp, int length, Stringy *s)
{
  char *string = new char [length+1];
  string[length] = '\0';
  int c;
  bool read = true;
  for (int k = 0 ; k < length && read ; ++k)
    {
      read = read_s32(&c, fp);
      string[k] = (char) c;
    }
  if (read)
    *s = trim_white(string);
  delete [] string;

  return read;
}

// ----------------------------------------------------------------------------
//
static SPoint felix_origin_ppm(const Felix_Header &fh)
{
  SPoint origin_ppm(fh.dim);

  for (int a = 0 ; a < fh.dim ; ++a)
    {
      double sfreq = fh.observe_freq[a];
      double ppm_per_data_point =
	(fh.spectral_width[a] / fh.data_size[a]) / sfreq;
      double ppm_ref = fh.ref_shift[a] / sfreq;
      double ref_pos = fh.ref_point[a] - 1;	// 1 = origin in felix
      origin_ppm[a] = ppm_ref + ppm_per_data_point * ref_pos;
    }

  return origin_ppm;
}
