/*
 * ContourStream.C:	Implementation of the ContourStream class
 *
 * The ContourStream class is a "stream" of calculated contours, with
 * command such as moveto and lineto embedded in the stream. The
 * stream knows how to display() itself.
 *
 */
#include "contour.h"		// Use Drawing_Routines
#include "contourstream.h"
#include "memalloc.h"		// use new()
#include "num.h"

#define CS_SIZE		256		// initial size
#define CS_RESIZE	256		// resize size

//
// Add this contour to the stream
//
void ContourStream::addContour(float level, int cnt, float *xp, float *yp)
{
  float *f = _allocate(2 + cnt + cnt);
  f[0] = level;
  f[1] = (float) cnt;

  f = f + 2;
  for (int k = 0 ; k < cnt ; ++k)
    {
      f[2*k] = xp[k];
      f[2*k+1] = yp[k];
    }
}

//
// Allocate float array.
//
float *ContourStream::_allocate(int floats)
{
	if (mStream == NULL || mSize < mUsed + floats)
	  {
	    //
	    // Reallocate array
	    //
		int size = mSize;
		while (size < mUsed + floats)
			size += CS_RESIZE;
		resize_array(size);
	  }


	//
	// Get the start of the allocated memory, increment the size,
	// then return the start of memory.
	//
	float	*sp = mStream + mUsed;
	mUsed += floats;

	return sp;
}

//
// resize the allocated memory down to the actual amount used.
//
void ContourStream::free_extra_memory()
{
  if (mUsed < mSize)
    resize_array(mUsed);
}

// ----------------------------------------------------------------------------
// Must have size > mUsed
//
void ContourStream::resize_array(int size)
{
  float *copy = new float [size];
  for (int k = 0 ; k < mUsed ; ++k)
    copy[k] = mStream[k];
  delete [] mStream;
  mSize = size;
  mStream = copy;
}

//
// Construct a new ContourStream
//
ContourStream::ContourStream()
{
	mUsed = mSize = 0;
	mStream = NULL;
}

ContourStream::~ContourStream()
{
	if (mStream)
		delete [] mStream;
}

// ----------------------------------------------------------------------------
//
void ContourStream::display(Drawing_Routines &dr)
{
  if (mStream == NULL)
    return;

	float	current_level = (mStream[0] == 0 ? 1 : 0);
	float	*endOfStream = mStream + mUsed;

	for (float *sp = mStream ; sp < endOfStream ; ) {
	  float level = sp[0];
	  if (level != current_level)
	    {
	      dr.set_contour_level(level);
	      current_level = level;
	    }
	  int len = (int) sp[1];
	  float *xy = sp + 2;
	  dr.draw_contour_path(xy, len);
	  sp += 2 + len + len;
	}
}

// ----------------------------------------------------------------------------
//
size_t ContourStream::memory_use() const
{
  return sizeof(ContourStream) + mSize * sizeof(float);
}
