// ----------------------------------------------------------------------------
// A condition holds a set of resonances.  Each spectrum has a condition
// which physically corresponds to the temperature, pH, and other physical
// parameters that effect chemical shift values.  The condition name will
// typically be something like "30 deg, ph 7" describing these physical
// characteristics.  The condition object allows Sparky to maintain separate
// sets of resonance chemical shifts for spectra acquired under different
// conditions.  The resonance frequencies for a condition are calculated
// by averaging peak positions for all assigned peaks in spectra that were
// acquired under that condition.
//
#ifndef _Condition_h
#define _Condition_h

#include "list.h"		// Use List
#include "stringc.h"		// Use Stringy
#include "table.h"		// use Table

class Atom;
class Molecule;
class Resonance;

class Condition
{

public:
  Condition(Molecule *, const Stringy &name);
  virtual		~Condition();

  Stringy	name() const;
  Stringy	fullname(void) const;
  Molecule	*molecule() const;

  //
  // Resonances
  //

  const List &	resonance_list() const;
  void		add_resonance(Resonance *r);
  void		remove_resonance(Resonance *r);
  Resonance *	define_resonance(const Stringy &group, const Stringy &atom,
				 const Stringy &nucleus);
  Resonance *	find_resonance(const Stringy &group,
			       const Stringy &atom) const;
  Resonance *	find_resonance(const Atom &atom) const;
  Resonance *	find_resonance(const Stringy &resname) const;
  List		interval_resonances(double ppm_min, double ppm_max,
				    const Stringy &nucleus,
				    double alias_min, double alias_max) const;

private:
  Stringy	mName;
  Molecule	*mMolecule;		// the Condition's Molecule
  List		mResonance;		// the list of Resonances
  Table		atom_to_resonance;
};

#endif
