/*
 * Condition.C - Implementation of the Condition class 
 *
 */
#include <stdlib.h>		// Use atof(), atoi()
#include <string.h>		// Use strcmp()

#include "atom.h"
#include "condition.h"
#include "constant.h"
#include "crosspeak.h"		// Use CrossPeak
#include "group.h"		// use Group
#include "memalloc.h"		// use new()
#include "molecule.h"
#include "num.h"
#include "resonance.h"
#include "session.h"		// use Session
#include "spectrum.h"
#include "table.h"		// use Table
#include "notifier.h"		// use Notifier
#include "utility.h"		// Use same_file(), message()

// ----------------------------------------------------------------------------
//
Condition::Condition(Molecule *m, const Stringy &name)
{
  mMolecule = m;			// set the Molecule
  mName = name;
  m->add_condition(this);

  m->session().notifier().send_notice(nt_created_condition, this);
}

/*
 * Destructor
 *
 * A Condition that has been loaded from the DB cannot be destroyed,
 * so we don't do any cleanup
 */
Condition::~Condition()
{
  Molecule *m = molecule();
  m->session().notifier().send_notice(nt_will_delete_condition, this);

  m->remove_condition(this);

  List rlist = resonance_list();
  mResonance.erase();	// Avoid O(n^2) 1 by 1 erasing of resonances
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    delete (Resonance *) rlist[ri];
}

/*
 * Create a Resonance in this Condition.
 */
Resonance *Condition::define_resonance(const Stringy &gpname,
				       const Stringy &atname,
				       const Stringy &nucleus)
{
  Resonance *rp = find_resonance(gpname, atname);
  if (!rp)
    {
      Molecule *m = molecule();
      Atom *ap = m->define_atom(gpname, atname, nucleus);
      if (ap)
	rp = new Resonance(this, ap);
    }
  return rp;
}

// ----------------------------------------------------------------------------
//
void Condition::add_resonance(Resonance *r)
{
  mResonance.append(r);
  atom_to_resonance.insert(r->atom(), r);
  Notifier &n = molecule()->session().notifier();
  n.send_notice(nt_added_resonance_to_condition, r);
}

/*
 * Destroy a Resonance in this Condition.
 */
void Condition::remove_resonance(Resonance *r)
{
  mResonance.erase(r);
  atom_to_resonance.remove(r->atom());
  Notifier &n = molecule()->session().notifier();
  n.send_notice(nt_removed_resonance_from_condition, r);
}

// ----------------------------------------------------------------------------
//
Resonance *Condition::find_resonance(const Stringy &group,
				     const Stringy &atom) const
{
  Atom *a = molecule()->find_atom(group, atom);
  return find_resonance(*a);

  //
  // Replaced this linear search with table lookup because
  // it consumed 40% of the time to load a small project.
  //

  /*
  const List &rlist = resonance_list();
  for (List::iterator ri = rlist.begin() ; ri != rlist.end() ; ++ri)
    {
      Resonance *r = (Resonance *) *ri;
      if (r->group()->name() == group && r->atom()->name() == atom)
	return r;
    }
  return NULL;
  */
}

// ----------------------------------------------------------------------------
//
Resonance *Condition::find_resonance(const Atom &atom) const
{
  TableData r;
  return (atom_to_resonance.find(&atom, &r) ? (Resonance *) r : NULL);
}

// ----------------------------------------------------------------------------
// Slow -- uses linear search.
//
Resonance *Condition::find_resonance(const Stringy &resname) const
{
  const List &rlist = resonance_list();
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      Resonance *r = (Resonance *) rlist[ri];
      if (r->name() == resname)
	return r;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
// Return sorted list of resonances in interval (ppm_min, ppm_max).
// The resonances are first aliased into the range (alias_min, alias_max).
//
List Condition::interval_resonances(double ppm_min, double ppm_max,
				    const Stringy &nucleus,
				    double alias_min, double alias_max) const
{
  List reslist;
  const List &rlist = resonance_list();
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      Resonance *r = (Resonance *) rlist[ri];
      if (r->atom()->nucleus() == nucleus)
	{
	  double ppm_pos = interval_mod(r->frequency(), alias_min, alias_max);
	  if (ppm_pos >= ppm_min && ppm_pos <= ppm_max)
	    reslist.append(r);
	}
    }
  sort_aliased_resonances(reslist, alias_min, alias_max);

  return reslist;
}

/*
 * Return this Condition's name.
 */
Stringy Condition::name() const
{
  return mName;
}

//
// Return the fully qualified name based on the molecule and condition names
//
Stringy Condition::fullname() const
{
  Stringy molname = molecule()->name();
  if (molname.is_empty())
    return name();

  return molname + " / " + name();
}

// ----------------------------------------------------------------------------
//
Molecule *Condition::molecule() const
  { return mMolecule; }
const List &Condition::resonance_list() const
  { return mResonance; }
