// ----------------------------------------------------------------------------
// Guess peak assignments using currently known resonances.
// A sequence index constraint can be applied.  For example the w1 and w2
// axis assignments can be required to be to groups i-1 and i respectively.
// The atom name for each axis can also be specified.  Or a list of allowed
// atom names can be specified, for example HB1|HB2.  A maximum ppm range
// from peak position to resonance position can be specified for each axis.
//
#include <ctype.h>		// Use isspace()
#include <math.h>		// use fabs()
#include <stdlib.h>		// use size_t
#include <string.h>		// Use strchr(), strncmp()

#include "assignguess.h"
#include "atom.h"		// Use Atom
#include "condition.h"		// Use Condition
#include "crosspeak.h"		// Use CrossPeak
#include "group.h"		// Use Group
#include "memalloc.h"		// use new()
#include "num.h"		// use interval_mod()
#include "resonance.h"		// use Resonance
#include "spectrum.h"		// Use Spectrum
#include "utility.h"		// use fatal_error()

static SRegion guess_ppm_region(const Guess_Assignments &ga,
				const SPoint &center);
static double guess_error(const Guess_Assignments &ga, Assignment_Guess *ag,
			  const SPoint &center, const SRegion &alias_onto);
static List make_guess_list(const Guess_Assignments &ga, const List &reslist,
			    const SRegion &alias_onto, const List &nuclei,
			    const SRegion &ppm_region, int max_axis_guesses);
static int compare_guesses(const void *guess1, const void *guess2);
static void axis_resonance_lists(const Guess_Assignments &ga,
				 const List &rlist,
				 const SRegion &alias_onto, const List &nuclei,
				 const SRegion &ppm_region, List *rlists);
static List find_resonances(const List &rlist,
			    double alias_min, double alias_max,
			    double ppm_min, double ppm_max,
			    const Stringy &nucleus, const Stringy &atom_spec);
static bool index_constrained(const Guess_Assignments &ga,
			      List *rlists, List *index_list);
static List intersect_lists(const List &list1, const List &list2);
static bool is_index_constraint(const char *type);
static void add_acceptable_indices(const Guess_Assignments &ga,
				   Resonance *r, int a, List &ilist);
static void append_unique(int i, List &list);
static bool acceptable_index(const Guess_Assignments &ga,
			     Resonance *r, int a, int index);
static bool acceptable_index(const Guess_Assignments &ga,
			     Resonance *r, int a, const List &ilist);
static void index_guess_combinations(const Guess_Assignments &ga, int index,
				     List *rlists, List *guesses);
static void all_guess_combinations(const Guess_Assignments &ga, List *rlists,
				   List *guesses);
static bool first_multi_list_index(List *lists, bool *use, int dim, int *i);
static bool next_multi_list_index(List *lists, bool *use, int dim, int *i);
static bool assignment_is_atom_match(const char *pat, const char *nm);
static void guessable_resonances(const Guess_Assignments &ga,
				 const List &reslist,
				 const SRegion &alias_onto, const List &nuclei,
				 const SRegion &region, List *rlists);

// ----------------------------------------------------------------------------
//
const char *Guess_Assignments::any_group = "any";
const char *Guess_Assignments::this_group = "i";
const char *Guess_Assignments::previous_group = "i-1";
const char *Guess_Assignments::this_or_previous_group = "i|i-1";
const char *Guess_Assignments::no_group = "none";

// ----------------------------------------------------------------------------
//
const char *Guess_Assignments::constraint_names[] =
{
  Guess_Assignments::any_group,
  Guess_Assignments::this_group,
  Guess_Assignments::previous_group,
  Guess_Assignments::this_or_previous_group,
  Guess_Assignments::no_group,
  NULL
};

// ----------------------------------------------------------------------------
//
Assignment_Guess::Assignment_Guess(int dim, Resonance *rp[])
{
  this->dim = dim;
  this->mRes = new Resonance *[dim];
  for (int i = 0; i < dim; i++)
    mRes[i] = rp[i];
}

// ----------------------------------------------------------------------------
//
Assignment_Guess::~Assignment_Guess()
{
  delete [] mRes;
}

// ----------------------------------------------------------------------------
//
int Assignment_Guess::dimension() const { return dim; }
Resonance *Assignment_Guess::resonance(int i) const { return mRes[i]; }

// ----------------------------------------------------------------------------
//
class assignment_guess_compare : public ListOrder
{
public:
  assignment_guess_compare(const Guess_Assignments &g, CrossPeak *xp) : ga(g)
    {
      this->center = xp->position();
      this->alias_onto = xp->spectrum()->alias_region();
    }
  virtual ~assignment_guess_compare() {};
  virtual int operator()(const void *a, const void *b) const;
private:
  const Guess_Assignments &ga;
  SPoint center;
  SRegion alias_onto;
};

// ----------------------------------------------------------------------------
//
int assignment_guess_compare::operator()(const void *a, const void *b) const
{
  double dev1 = guess_error(ga, (Assignment_Guess *) a, center, alias_onto);
  double dev2 = guess_error(ga, (Assignment_Guess *) b, center, alias_onto);
  return compare_doubles(dev1, dev2);
}

// ----------------------------------------------------------------------------
//
static double guess_error(const Guess_Assignments &ga, Assignment_Guess *ag,
			  const SPoint &center, const SRegion &alias_onto)
{
  double dev = 0;
  int dim = center.dimension();
  for (int a = 0 ; a < dim ; ++a)
    {
      Resonance *r = ag->resonance(a);
      if (r)
	{
	  double threshold = ga.ppm_range[a];
	  if (threshold > 0)
	    {
	      double pos = interval_mod(r->frequency(),
					alias_onto.min[a], alias_onto.max[a]);
	      dev += fabs(pos - center[a]) / threshold;
	    }
	}
      }
  return dev;
}

// ----------------------------------------------------------------------------
//
Guess_Assignments::Guess_Assignments(int dim)
{
  this->dimension = dim;
  this->sequence_constraint = new const char *[dim];
  this->atom_constraint = new Stringy [dim];
  this->ppm_range = SPoint(dim);

  for (int i = 0; i < dim; ++i)
    {
      sequence_constraint[i] = no_group;
      atom_constraint[i] = "";
    }
}

// ----------------------------------------------------------------------------
//
Guess_Assignments::Guess_Assignments(const Guess_Assignments &ga)
{
  int dim = ga.dimension;
  this->dimension = dim;
  this->sequence_constraint = new const char *[dim];
  this->atom_constraint = new Stringy [dim];
  this->ppm_range = SPoint(dim);
  *this = ga;
}

// ----------------------------------------------------------------------------
//
Guess_Assignments &Guess_Assignments::operator=(const Guess_Assignments &ga)
{
  if (ga.dimension != dimension)
    fatal_error("Guess_Assignments::operator=(): Dimension mismatch.\n");

  for (int i = 0; i < dimension; ++i)
    {
      sequence_constraint[i] = ga.sequence_constraint[i];
      atom_constraint[i] = ga.atom_constraint[i];
    }
  ppm_range = ga.ppm_range;

  return *this;
}

// ----------------------------------------------------------------------------
//
Guess_Assignments::~Guess_Assignments()
{
  delete [] sequence_constraint;
  delete [] atom_constraint;

  sequence_constraint = NULL;
  atom_constraint = NULL;
}

// ----------------------------------------------------------------------------
//
#define MAX_AXIS_GUESSES 10

List Guess_Assignments::guesses(CrossPeak *xp) const
{
  Spectrum *sp = xp->spectrum();
  Condition *c = sp->condition();
  List glist = make_guess_list(*this, c->resonance_list(),
			       sp->alias_region(), sp->nucleus_types(),
			       guess_ppm_region(*this, xp->position()),
			       MAX_AXIS_GUESSES);

  glist.sort(assignment_guess_compare(*this, xp));

  return glist;
}

// ----------------------------------------------------------------------------
//
static SRegion guess_ppm_region(const Guess_Assignments &ga,
				const SPoint &center)
{
  int dim = center.dimension();
  SRegion ppm_region = SRegion(dim);
  for (int a = 0 ; a < dim ; ++a)
    {
      double pos_ppm = center[a];
      double range_ppm = ga.ppm_range[a];
      ppm_region.min[a] = pos_ppm - range_ppm;
      ppm_region.max[a] = pos_ppm + range_ppm;
    }
  return ppm_region;
}

/* 
 * Reclaim the assignment guess storage
 */
void free_assignment_guesses(List &glist)
{
	for (int g = 0 ; g < glist.size() ; ++g)
		delete (Assignment_Guess *) glist[g];
	glist.erase();
}

// ----------------------------------------------------------------------------
// Return a list of assignment guesses.
//
static List make_guess_list(const Guess_Assignments &ga, const List &reslist,
			    const SRegion &alias_onto, const List &nuclei,
			    const SRegion &ppm_region, int max_axis_guesses)
{
  int dim = ga.dimension;
  List *rlists = new List [dim];

  axis_resonance_lists(ga, reslist, alias_onto, nuclei, ppm_region, rlists);
  for (int a = 0 ; a < dim ; ++a)
    rlists[a] = closest_n_resonances(rlists[a],
				     alias_onto.min[a], alias_onto.max[a],
				     ppm_region.center(a), max_axis_guesses);

  List guesses;

  List index_list;
  if (index_constrained(ga, rlists, &index_list))
    {
      for (int ii = 0 ; ii < index_list.size() ; ++ii)
	{
	  int index = (int) (long) index_list[ii];
	  index_guess_combinations(ga, index, rlists, &guesses);
	}
      guesses.sort_removing_duplicates(compare_guesses);
    }
  else
    all_guess_combinations(ga, rlists, &guesses);

  delete [] rlists;

  return guesses;
}

// ----------------------------------------------------------------------------
// Used for removing duplicates.
//
static int compare_guesses(const void *guess1, const void *guess2)
{
  Assignment_Guess *g1 = (Assignment_Guess *) guess1;
  Assignment_Guess *g2 = (Assignment_Guess *) guess2;

  int order = compare_ints(g1->dimension(), g2->dimension());
  if (order)
    return order;

  int dim = g1->dimension();
  for (int a = 0 ; a < dim ; ++a)
    {
      order = compare_pointers(g1->resonance(a), g2->resonance(a));
      if (order)
	return order;
    }

  return 0;
}

// ----------------------------------------------------------------------------
// Find the list of resonances for each axis in the desired ppm range and
// having the guess atom names.
//
static void axis_resonance_lists(const Guess_Assignments &ga,
				 const List &rlist,
				 const SRegion &alias_onto, const List &nuclei,
				 const SRegion &ppm_region, List *rlists)
{
  for (int a = 0 ; a < ppm_region.dimension() ; ++a)
    rlists[a] = find_resonances(rlist, alias_onto.min[a], alias_onto.max[a],
				ppm_region.min[a], ppm_region.max[a],
				*(Stringy *)nuclei[a], ga.atom_constraint[a]);
				

}

// ----------------------------------------------------------------------------
//
static List find_resonances(const List &rlist,
			    double alias_min, double alias_max,
			    double ppm_min, double ppm_max,
			    const Stringy &nucleus, const Stringy &atom_spec)
{
  List res;
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      Resonance *r = (Resonance *) rlist[ri];
      double res_position = interval_mod(r->frequency(), alias_min, alias_max);
      if (res_position >= ppm_min &&
	  res_position <= ppm_max &&
	  r->atom()->nucleus() == nucleus &&
	  assignment_is_atom_match(atom_spec.cstring(),
				   r->atom()->name().cstring()))
	res.append(r);
    }
  return res;
}

// ----------------------------------------------------------------------------
//
static bool index_constrained(const Guess_Assignments &ga,
			      List *rlists, List *index_list)
{
  int dim = ga.dimension;
  bool index_constraint_found = false;

  for (int a = 0 ; a < dim ; ++a)
    if (is_index_constraint(ga.sequence_constraint[a]))
      {
	List ilist;
	for (int ri = 0 ; ri < rlists[a].size() ; ++ri)
	  add_acceptable_indices(ga, (Resonance *) rlists[a][ri], a, ilist);

	if (!index_constraint_found)
	  {
	    *index_list = ilist;
	    index_constraint_found = true;
	  }
	else
	  *index_list = intersect_lists(*index_list, ilist);
      }

  return index_constraint_found;
}

// ----------------------------------------------------------------------------
//
static List intersect_lists(const List &list1, const List &list2)
{
  List ilist;
  for (int it = 0 ; it < list1.size() ; ++it)
    if (list2.contains(list1[it]))
      ilist.append(list1[it]);

  return ilist;
}

// ----------------------------------------------------------------------------
//
static bool is_index_constraint(const char *type)
{
  return (type == Guess_Assignments::this_group ||
	  type == Guess_Assignments::previous_group ||
	  type == Guess_Assignments::this_or_previous_group);
}

// ----------------------------------------------------------------------------
//
static void add_acceptable_indices(const Guess_Assignments &ga,
				   Resonance *r, int a, List &ilist)
{
  int gnum = r->group()->number();
  if (gnum > 0)
    {
      const char *sc = ga.sequence_constraint[a];
      if (sc == Guess_Assignments::this_group)
	append_unique(gnum, ilist);
      else if (sc == Guess_Assignments::previous_group)
	append_unique(gnum+1, ilist);
      else if (sc == Guess_Assignments::this_or_previous_group)
	{
	  append_unique(gnum, ilist);
	  append_unique(gnum+1, ilist);
	}
    }
}

// ----------------------------------------------------------------------------
//
static void append_unique(int i, List &list)
  { if (!list.contains((void *) i)) list.append((void *) i); }

// ----------------------------------------------------------------------------
//
static bool acceptable_index(const Guess_Assignments &ga,
			     Resonance *r, int a, int index)
{
  int gnum = r->group()->number();
  if (gnum == 0)
    return false;

  const char *sc = ga.sequence_constraint[a];
  bool accept = false;
  if (sc == Guess_Assignments::this_group)
    accept = (gnum == index);
  else if (sc == Guess_Assignments::previous_group)
    accept = (gnum+1 == index);
  else if (sc == Guess_Assignments::this_or_previous_group)
    accept = ((gnum == index) || (gnum+1 == index));
  else if (sc == Guess_Assignments::any_group)
    accept = true;

  return accept;
}

// ----------------------------------------------------------------------------
//
static bool acceptable_index(const Guess_Assignments &ga,
			     Resonance *r, int a, const List &ilist)
{
  int gnum = r->group()->number();
  if (gnum == 0)
    return false;

  const char *sc = ga.sequence_constraint[a];
  bool accept = false;
  if (sc == Guess_Assignments::this_group)
    accept = ilist.contains((void *) gnum);
  else if (sc == Guess_Assignments::previous_group)
    accept = ilist.contains((void *) (gnum+1));
  else if (sc == Guess_Assignments::this_or_previous_group)
    accept = (ilist.contains((void *) gnum) ||
	      ilist.contains((void *) (gnum+1)));
  else if (sc == Guess_Assignments::any_group)
    accept = true;

  return accept;
}

// ----------------------------------------------------------------------------
//
static void index_guess_combinations(const Guess_Assignments &ga, int index,
				     List *rlists, List *guesses)
{
  int dim = ga.dimension;
  List *rilists = new List [dim];

  for (int a = 0 ; a < dim ; ++a)
    {
      const char *t = ga.sequence_constraint[a];
      if (t == Guess_Assignments::any_group)
	rilists[a] = rlists[a];
      else if (is_index_constraint(t))
	{
	  for (int ri = 0 ; ri < rlists[a].size() ; ++ri)
	    {
	      Resonance *r = (Resonance *) rlists[a][ri];
	      if (acceptable_index(ga, r, a, index))
		rilists[a].append(r);
	    }
	}
    }

  all_guess_combinations(ga, rilists, guesses);
  delete [] rilists;
}

// ----------------------------------------------------------------------------
//
static void all_guess_combinations(const Guess_Assignments &ga,
				   List *rlists, List *guesses)
{
  int dim = ga.dimension;
  bool *assign = new bool [dim];
  for (int a = 0 ; a < dim ; ++a)
    assign[a] = (ga.sequence_constraint[a] != Guess_Assignments::no_group);

  int *ri = new int [dim];
  Resonance **res = new Resonance *[dim];
  if (first_multi_list_index(rlists, assign, dim, ri))
    do
      {
	for (int a = 0 ; a < dim ; ++a)
	  res[a] = (Resonance *) (assign[a] ? rlists[a][ri[a]] : NULL);
	guesses->append(new Assignment_Guess(dim, res));
      }
    while (next_multi_list_index(rlists, assign, dim, ri));

  delete [] res;
  delete [] ri;
  delete [] assign;
}

// ----------------------------------------------------------------------------
//
static bool first_multi_list_index(List *lists, bool *use, int dim, int *i)
{
  bool used_one = false;

  for (int a = 0 ; a < dim ; ++a)
    if (use[a])
      {
	used_one = true;
	if (lists[a].empty())
	  return false;
	else
	  i[a] = 0;
      }

  return used_one;
}

// ----------------------------------------------------------------------------
//
static bool next_multi_list_index(List *lists, bool *use, int dim, int *i)
{
  for (int a = 0 ; a < dim ; ++a)
    if (use[a])
      {
	i[a] += 1;
	if (i[a] < lists[a].size())
	  return true;
	i[a] = 0;
      }

  return false;
}

/*
 * Return true if <nm> is a match for any part of pattern <pat>.
 * The <pat> may be a single atom name or may be atom names
 * separated by '|'. Whitespace around each '|' is ignored.
 */
static bool assignment_is_atom_match(const char *pat, const char *nm)
{
	/*
	 * Pattern may be: atomName1 | atomName2 | atomName3
	 */
	const char *cp = strchr(pat, '|');
	if (cp) {
		size_t	nmLen = strlen(nm);

		while (pat) {
			if (strncmp(pat, nm, nmLen) == 0
			&&  (pat[nmLen] == '\0'
			  || pat[nmLen] == '|'
			  || isspace(pat[nmLen])))
				return true;

			pat = cp ? cp + 1 : NULL;
			if (pat) {
				pat = skip_white(pat);
				cp = strchr(pat, '|');
			}
		}
		return false;
	}
	return *pat == '\0' || strcmp(pat, nm) == 0;
}

/*
 * Make the assignment for CrossPeak <xp> using guess <ap>.
 */
static void _makeGuessAssign(CrossPeak *xp, Assignment_Guess *ap)
{
	for (int a = 0; a < xp->dimension(); a++)
	  xp->SetResonance(a, ap->resonance(a));
}

/*
 * Guess the assignment of selected crosspeaks. The guess uses the
 * assignment relationships and thresholds to determine if
 * more than one assignment is possible. If only one is possible,
 * that assignment is applied.
 */
List Guess_Assignments::apply_unique_guesses(const List &xplist) const
{
  List assigned;
	for (int pi = 0 ; pi < xplist.size() ; ++pi) {
		CrossPeak *xp = (CrossPeak *) xplist[pi];

		/*
		 * Only assign peaks that have no assignment yet.
		 */
		if (xp->IsAssigned())
			continue;
		
		List glist = guesses(xp);
		if (glist.size() == 1)
		  {
		    _makeGuessAssign(xp, (Assignment_Guess *) glist[0]);
		    assigned.append(xp);
		  }
		free_assignment_guesses(glist);
	}

	return assigned;
}

// ----------------------------------------------------------------------------
// Return lists of resonances for each axis that could be used in
// assignment guesses for peaks in the given region.
//
static void guessable_resonances(const Guess_Assignments &ga,
				 const List &reslist,
				 const SRegion &alias_onto, const List &nuclei,
				 const SRegion &region, List *rlists)
{
  axis_resonance_lists(ga, reslist, alias_onto, nuclei, region, rlists);

  List index_list;
  if (index_constrained(ga, rlists, &index_list))
    {
      for (int a = 0 ; a < region.dimension() ; ++a)
	if (is_index_constraint(ga.sequence_constraint[a]))
	  {
	    List res;
	    for (int ri = 0 ; ri < rlists[a].size() ; ++ri)
	      {
		Resonance *r = (Resonance *) rlists[a][ri];
		if (acceptable_index(ga, r, a, index_list))
		  res.append(r);
	      }
	    rlists[a] = res;
	  }
    }
}

// ----------------------------------------------------------------------------
//
List Guess_Assignments::guessable_resonances(Spectrum *sp,
					     const SRegion &region,
					     int axis) const
{
  List *rlists = new List [region.dimension()];
  ::guessable_resonances(*this, sp->condition()->resonance_list(),
			 sp->alias_region(), sp->nucleus_types(),
			 region, rlists);
  List axis_rlist = rlists[axis];
  delete [] rlists;

  SRegion alias_onto = sp->alias_region();
  sort_aliased_resonances(axis_rlist,
			  alias_onto.min[axis], alias_onto.max[axis]);
  return axis_rlist;
}
