// ----------------------------------------------------------------------------
//
#include "assigncopy.h"
#include "assignguess.h"
#include "atom.h"
#include "axismap.h"		// Use Axis_Map
#include "condition.h"
#include "crosspeak.h"
#include "group.h"
#include "label.h"
#include "memalloc.h"		// use new()
#include "molecule.h"
#include "num.h"
#include "peak.h"
#include "peakgp.h"
#include "reporter.h"		// use Reporter
#include "resonance.h"
#include "spectrum.h"

static void axis_map_warning(Spectrum *from, Spectrum *to, Reporter &);
static void copy_assignment(CrossPeak *xp, Spectrum *to, bool label,
			    Reporter &);
static void copy_assignment(Spectrum *from, CrossPeak *xp, bool label,
			    Reporter &);
static void copy_assignment(CrossPeak *xpfrom, CrossPeak *xpto,
			    const Axis_Map &axismap, bool label, Reporter &);
static bool assignments_match(CrossPeak *xp1, CrossPeak *xp2,
			      const Axis_Map &axismap);
static bool resonances_match(Resonance *r1, Resonance *r2);
static CrossPeak *unique_nearby_peak(CrossPeak *xp, Spectrum *to,
				     const Axis_Map &axismap, Reporter &);
static CrossPeak *unique_nearby_peak(Spectrum *from, CrossPeak *xp,
				     const Axis_Map &axismap, Reporter &);
static List crosspeak_spectra(const List &crosspeaks);

// ----------------------------------------------------------------------------
//
void assignment_copy(const List &frompeaks, Spectrum *to, bool label,
		     Reporter &rr)
{
  //
  // Generate warnings when axismaps between spectra can't be generated.
  //
  List slist = crosspeak_spectra(frompeaks);
  for (int si = 0 ; si < slist.size() ; ++si)
    axis_map_warning((Spectrum *) slist[si], to, rr);

  for (int pi = 0 ; pi < frompeaks.size() ; ++pi)
    copy_assignment((CrossPeak *) frompeaks[pi], to, label, rr);
}

// ----------------------------------------------------------------------------
//
void assignment_copy(Spectrum *from, const List &topeaks, bool label,
		     Reporter &rr)
{
  //
  // Generate warnings when axismaps between spectra can't be generated.
  //
  List slist = crosspeak_spectra(topeaks);
  for (int si = 0 ; si < slist.size() ; ++si)
    axis_map_warning(from, (Spectrum *) slist[si], rr);

  for (int pi = 0 ; pi < topeaks.size() ; ++pi)
    copy_assignment(from, (CrossPeak *) topeaks[pi], label, rr);
}

// ----------------------------------------------------------------------------
//
void crossdiagonal_assignment_copy(const List &to_peaks, bool label,
				   Reporter &rr)
{
  IPoint swap(2);
  swap[0] = 1;
  swap[1] = 0;

  Axis_Map swap_axes;
  swap_axes.set_map(swap);

  for (int pi = 0 ; pi < to_peaks.size() ; ++pi)
    {
      CrossPeak *to_xp = (CrossPeak *) to_peaks[pi];
      Spectrum *sp = to_xp->spectrum();
      if (sp->dimension() == 2 && sp->is_homonuclear(0, 1))
	{
	  CrossPeak *from_xp = unique_nearby_peak(sp, to_xp, swap_axes, rr);
	  if (from_xp)
	    copy_assignment(from_xp, to_xp, swap_axes, label, rr);
	}
    }
}

// ----------------------------------------------------------------------------
//
static void axis_map_warning(Spectrum *from, Spectrum *to, Reporter &rr)
{
  Axis_Map amap;
  if (!identity_axis_map(from, to, &amap) &&
      !unique_axis_map(from, to, &amap))
    rr.warning("Assignment Copy: "
	       "Couldn't map axes of spectrum %s to spectrum %s\n",
	       from->name().cstring(), to->name().cstring());
}

// ----------------------------------------------------------------------------
//
static void copy_assignment(CrossPeak *xp, Spectrum *to, bool label,
			    Reporter &rr)
{
  Axis_Map axismap;
  if (!identity_axis_map(xp->spectrum(), to, &axismap) &&
      !unique_axis_map(xp->spectrum(), to, &axismap))
    return;

  CrossPeak *xpto = unique_nearby_peak(xp, to, axismap, rr);

  if (xpto)
    copy_assignment(xp, xpto, axismap, label, rr);
}

// ----------------------------------------------------------------------------
//
static void copy_assignment(Spectrum *from, CrossPeak *xp, bool label,
			    Reporter &rr)
{
  Axis_Map axismap;
  if (!identity_axis_map(from, xp->spectrum(), &axismap) &&
      !unique_axis_map(from, xp->spectrum(), &axismap))
    return;

  CrossPeak *xpfrom = unique_nearby_peak(from, xp, axismap, rr);

  if (xpfrom)
    copy_assignment(xpfrom, xp, axismap, label, rr);
}

// ----------------------------------------------------------------------------
//
static void copy_assignment(CrossPeak *xpfrom, CrossPeak *xpto,
			    const Axis_Map &axismap, bool label, Reporter &rr)
{
  if (!assignments_match(xpfrom, xpto, axismap))
    rr.message("Assignment paste: "
	       "Existing assignments %s and %s don't match.\n",
	       xpfrom->assignment_name().cstring(),
	       xpto->assignment_name().cstring());
  else
    {
      bool made_change = false;
      for (int a = 0 ; a < xpto->spectrum()->dimension() ; ++a)
	{
	  Resonance *r = xpfrom->resonance(a);
	  int to_axis = axismap.map(a);
	  if (r && xpto->resonance(to_axis) == NULL)
	    {
	      xpto->ChangeAssignment(to_axis,
				     r->atom()->name(), r->group()->name());
	      made_change = true;
	    }
	}

      if (made_change)
	{
	  rr.message("Assignment Copy: Copied %s\n",
		     xpto->assignment_name().cstring());

	  if (label && xpto->label() == NULL)
	    (void) new Label(xpto);
	}
    }
}

// ----------------------------------------------------------------------------
//
static bool assignments_match(CrossPeak *xp1, CrossPeak *xp2,
			      const Axis_Map &axismap)
{
  if (xp1->spectrum()->dimension() != xp2->spectrum()->dimension())
    return false;

  for (int a = 0 ; a < xp1->spectrum()->dimension() ; ++a)
    {
      Resonance *r1 = xp1->resonance(a);
      int axis2 = axismap.map(a);
      Resonance *r2 = xp2->resonance(axis2);
      if (r1 && r2 && !resonances_match(r1, r2))
	return false;
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static bool resonances_match(Resonance *r1, Resonance *r2)
{
  return (r1->atom()->name() == r2->atom()->name() &&
	  r1->group()->name() == r2->group()->name());
}

// ----------------------------------------------------------------------------
//
static CrossPeak *unique_nearby_peak(CrossPeak *xp, Spectrum *to,
				     const Axis_Map &axismap, Reporter &rr)
{
  Spectrum *from = xp->spectrum();
  SPoint loc = axismap.map(xp->position());
  List nearto = to->nearby_crosspeaks(loc);

  if (nearto.size() == 0)
    {
      rr.message("Assignment paste: No destination peak for %s\n",
		 xp->assignment_name().cstring());
      return NULL;
    }
  else if (nearto.size() > 1)
    {
      rr.message("Assignment paste: Multiple destination peaks for %s\n",
		 xp->assignment_name().cstring());
      return NULL;
    }

  CrossPeak *xpto = (CrossPeak *) nearto[0];
  loc = axismap.invert(xpto->position());
  List nearfrom = from->nearby_crosspeaks(loc);

  if (nearfrom.size() > 1)
    {
      rr.message("Assignment paste: Multiple source peaks near %s\n",
		 xp->assignment_name().cstring());
      return NULL;
    }

  return xpto;
}

// ----------------------------------------------------------------------------
//
static CrossPeak *unique_nearby_peak(Spectrum *from, CrossPeak *xp,
				     const Axis_Map &axismap, Reporter &rr)
{
  Spectrum *to = xp->spectrum();
  SPoint loc = axismap.invert(xp->position());
  List nearfrom = from->nearby_crosspeaks(loc);

  if (nearfrom.size() == 0)
    {
      rr.message("Assignment paste: No source peak near %s ppm\n",
		 point_format(loc, "%.3f", false));
      return NULL;
    }
  else if (nearfrom.size() > 1)
    {
      rr.message("Assignment paste: Multiple source peaks near %s ppm\n",
		  point_format(loc, "%.3f", false));
      return NULL;
    }

  CrossPeak *xpfrom = (CrossPeak *) nearfrom[0];
  loc = axismap.map(xpfrom->position());
  List nearto = to->nearby_crosspeaks(loc);

  if (nearto.size() > 1)
    {
      rr.message("Assignment paste: Multiple destination peaks for %s\n",
		 xpfrom->assignment_name().cstring());
      return NULL;
    }

  return xpfrom;
}

// ----------------------------------------------------------------------------
//
static List crosspeak_spectra(const List &crosspeaks)
{
  List spectra;
  for (int pi = 0 ; pi < crosspeaks.size() ; ++pi)
    {
      Spectrum *sp = ((CrossPeak *)crosspeaks[pi])->spectrum();
      if (!spectra.contains(sp))
	spectra.append(sp);
    }
  return spectra;
}
