/*
* Copyright (c) 2012-2013 Spotify AB
*
* Licensed under the Apache License, Version 2.0 (the "License"); you may not
* use this file except in compliance with the License. You may obtain a copy of
* the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
* WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
* License for the specific language governing permissions and limitations under
* the License.
*/
#include <stdlib.h>
#include <inttypes.h>
#include <stdio.h>
#include <string.h>

#include "MurmurHash3.h"

void assert_murmurhash3_x86_32(uint32_t expected, const char *s, uint32_t seed) {
  uint32_t actual = murmurhash32_hash((uint8_t *) s, strlen(s), seed);
  if (expected != actual) {
    printf(" failed!\n");
    printf("Expected %"PRIu32" but got %"PRIu32"\n", expected, actual);
    exit(1);
  }
}

void assert_murmurhash3_x64_64(uint64_t expected, const char *s, uint32_t seed) {
  uint64_t actual = murmurhash64_hash((uint8_t *) s, strlen(s), seed);
  if (expected != actual) {
    printf(" failed!\n");
    printf("Expected %"PRIu64" but got %"PRIu64"\n", expected, actual);
    exit(1);
  }
}

static int parsehex(char c) {
  if ('0' <= c && c <= '9') return c - '0';
  if ('a' <= c && c <= 'f') return 10 + (c - 'a');
  printf("illegal char %c\n", c);
  exit(1);
}

static void hex(char *s) {
  int n = strlen(s);
  int i, j;
  for (i = 0, j = 0; i < n; i += 2, j++) {
    int high = parsehex(s[i]);
    int low = parsehex(s[i+1]);
    s[j] = (high * 16 + low) & 0xFF;
  }
  s[j] = 0;
}

int main() {
printf("Running hash test... ");
assert_murmurhash3_x86_32(0x5af6cd1b, "z", 0x5942ad3d);
assert_murmurhash3_x86_32(0xa9489813, "2", 0x71905ea7);
assert_murmurhash3_x86_32(0x77f4f434, "d", 0x64ee4d11);
assert_murmurhash3_x86_32(0x60fe8162, "L", 0x29fc249d);
assert_murmurhash3_x86_32(0xaaf896b2, "K", 0x14aae517);
assert_murmurhash3_x86_32(0xab2cb048, "N", 0x7ce81cf5);
assert_murmurhash3_x86_32(0x768beec5, "C", 0x743d7979);
assert_murmurhash3_x86_32(0xf15a77be, "G", 0x29bc6a00);
assert_murmurhash3_x86_32(0xeec4866a, "h", 0x53dd72a4);
assert_murmurhash3_x86_32(0x8426f89d, "u", 0x74fd834f);
assert_murmurhash3_x86_32(0x680b80bc, "87", 0x500dc5a2);
assert_murmurhash3_x86_32(0x7b18c4e1, "Nx", 0x10d5412e);
assert_murmurhash3_x86_32(0x48e0a968, "cI", 0x142f8840);
assert_murmurhash3_x86_32(0xc520f6b6, "hW", 0x6a059ad5);
assert_murmurhash3_x86_32(0xd63c105f, "EZ", 0x5b95f97d);
assert_murmurhash3_x86_32(0x758c1adb, "RE", 0x64d5803e);
assert_murmurhash3_x86_32(0x6c3d90a4, "Jo", 0x79806555);
assert_murmurhash3_x86_32(0xbf55dfa4, "wp", 0x673e5532);
assert_murmurhash3_x86_32(0x845dfd0e, "X5", 0x10fabf32);
assert_murmurhash3_x86_32(0xd0fd9be9, "eX", 0x3e3f0006);
assert_murmurhash3_x86_32(0x1a5fd660, "t5g", 0x696736c5);
assert_murmurhash3_x86_32(0x4d7530a2, "RLH", 0x20132ae9);
assert_murmurhash3_x86_32(0x5d35be0a, "tVk", 0x282bd870);
assert_murmurhash3_x86_32(0x28d2607c, "LYn", 0x54079486);
assert_murmurhash3_x86_32(0x10215d71, "O1k", 0x46d645a5);
assert_murmurhash3_x86_32(0x52eaf10d, "n43", 0x26abfa0c);
assert_murmurhash3_x86_32(0x2ac6ce75, "oZR", 0x2a968e7e);
assert_murmurhash3_x86_32(0xe72558bd, "dPb", 0x3dbae557);
assert_murmurhash3_x86_32(0x4d6cccf7, "Spe", 0x54657cad);
assert_murmurhash3_x86_32(0x7c745209, "8UM", 0x3e48db7c);
assert_murmurhash3_x86_32(0x8bf5674b, "P4Yy", 0x1eaef7eb);
assert_murmurhash3_x86_32(0x85f4a325, "jIPi", 0x3520d897);
assert_murmurhash3_x86_32(0x7aa5be50, "kF5l", 0x6afc31e1);
assert_murmurhash3_x86_32(0x52b0eaa5, "rkgt", 0x2e97cd16);
assert_murmurhash3_x86_32(0xe98b1bc8, "3CfW", 0x79e981e1);
assert_murmurhash3_x86_32(0x94e69772, "rD8l", 0x4525aef4);
assert_murmurhash3_x86_32(0xa56544ba, "GbqE", 0x0cda2629);
assert_murmurhash3_x86_32(0xdb4bcc63, "rVoe", 0x5f826d97);
assert_murmurhash3_x86_32(0xc5c013ab, "pyQs", 0x2dc2c7d0);
assert_murmurhash3_x86_32(0x9597719c, "fTev", 0x5287e7bc);
assert_murmurhash3_x86_32(0xb686d4ac, "MxZZr", 0x60f7a89e);
assert_murmurhash3_x86_32(0xa4a5df0d, "r44Kq", 0x25c6baae);
assert_murmurhash3_x86_32(0x064d9dc2, "VRmBi", 0x37b23d99);
assert_murmurhash3_x86_32(0x066b6073, "Zxso5", 0x3b07509c);
assert_murmurhash3_x86_32(0x112dfc59, "QXxHc", 0x21626c57);
assert_murmurhash3_x86_32(0x57912558, "3Wy0U", 0x2fa53851);
assert_murmurhash3_x86_32(0xdd28cf06, "Xl113", 0x20af4caa);
assert_murmurhash3_x86_32(0xc4debd5a, "KXj6y", 0x136fd981);
assert_murmurhash3_x86_32(0xd9e85fa5, "ox6QT", 0x57c7dc38);
assert_murmurhash3_x86_32(0x95bf2145, "7H7Eo", 0x5330848d);
assert_murmurhash3_x86_32(0xfc2267da, "Dpcbq6", 0x62179576);
assert_murmurhash3_x86_32(0x6b79bc0c, "lpcksC", 0x12185758);
assert_murmurhash3_x86_32(0xcc289239, "pVbXuz", 0x212382bc);
assert_murmurhash3_x86_32(0xf4821c5c, "znlHu2", 0x16a5a1f8);
assert_murmurhash3_x86_32(0xe77ffaed, "7q4KQg", 0x0f82524e);
assert_murmurhash3_x86_32(0xba2c718b, "ek4xJe", 0x1b09ee75);
assert_murmurhash3_x86_32(0x7c7a1ae5, "bQWzJ7", 0x2c3267c8);
assert_murmurhash3_x86_32(0x1ad4b45d, "cEXLZi", 0x73d20e88);
assert_murmurhash3_x86_32(0xd82230a5, "sjczJg", 0x33971a37);
assert_murmurhash3_x86_32(0x985ab014, "xubJMe", 0x5c48c98c);
assert_murmurhash3_x86_32(0xeb404e51, "vq8Fh2c", 0x1f66134c);
assert_murmurhash3_x86_32(0x36338e61, "9GaLBTK", 0x06e59da6);
assert_murmurhash3_x86_32(0x4e9a1213, "lcbwJSM", 0x57465553);
assert_murmurhash3_x86_32(0x5e52230d, "ofa7Zom", 0x5a7118db);
assert_murmurhash3_x86_32(0x47b027d7, "Ps84uk2", 0x5b0a53bf);
assert_murmurhash3_x86_32(0xa6ceb252, "YbkA44s", 0x1f5eee28);
assert_murmurhash3_x86_32(0x4ed59cbd, "fDUYvEX", 0x14c22e1f);
assert_murmurhash3_x86_32(0x78b8c94f, "R8XRujm", 0x6dab91f5);
assert_murmurhash3_x86_32(0xe0dc1077, "JsldMmM", 0x540d2fa0);
assert_murmurhash3_x86_32(0x5f039b2f, "x6kB8MX", 0x67e8ea83);
assert_murmurhash3_x86_32(0xa76df046, "nRkRwhIn", 0x207e8cf7);
assert_murmurhash3_x86_32(0x36450e8b, "FeSWy9F0", 0x5924b4c3);
assert_murmurhash3_x86_32(0x77534a7e, "QKRCtmGN", 0x2a50bd07);
assert_murmurhash3_x86_32(0xc1aeb318, "PyTaTKsK", 0x27d1cc44);
assert_murmurhash3_x86_32(0xa9a238c4, "KrC56OX3", 0x25307b55);
assert_murmurhash3_x86_32(0x5f0cbf3c, "7GlAx4p9", 0x2c6329e5);
assert_murmurhash3_x86_32(0x104b3222, "LOiEBQxL", 0x03cc6c96);
assert_murmurhash3_x86_32(0x0f5585c3, "gesuWT41", 0x493952a8);
assert_murmurhash3_x86_32(0x38c2e077, "RXZc2FvC", 0x712c0c42);
assert_murmurhash3_x86_32(0x559c4fae, "x1k4K6nm", 0x21252bd5);
assert_murmurhash3_x86_32(0xc647453d, "bUqT8CmCy", 0x679bf501);
assert_murmurhash3_x86_32(0x250fbe16, "Fy9wt8GvM", 0x2a8c58d5);
assert_murmurhash3_x86_32(0x2cac3c9d, "7WH6eLOk6", 0x4c1c6c2c);
assert_murmurhash3_x86_32(0x83423803, "1i2sba4vL", 0x3aeb8b07);
assert_murmurhash3_x86_32(0x61c28378, "Hq7OUAXy5", 0x3ee5d761);
assert_murmurhash3_x86_32(0x43a253e4, "HcDmgQ62a", 0x1dc3649c);
assert_murmurhash3_x86_32(0xeed79977, "ccudCEnE7", 0x025dfa90);
assert_murmurhash3_x86_32(0x6d328322, "fMwmBoUwU", 0x2b5b80d5);
assert_murmurhash3_x86_32(0xec7c731b, "dA9OWpE2r", 0x23cbd625);
assert_murmurhash3_x86_32(0xc6461907, "fCXHPzjad", 0x0813cf01);
assert_murmurhash3_x86_32(0x44016c59, "hrbLLM8Gg2", 0x60061445);
assert_murmurhash3_x86_32(0xe62f751c, "rCNexcQxCD", 0x1d31c96c);
assert_murmurhash3_x86_32(0x9ba3b339, "cAs17J9j8q", 0x6c7b6c7a);
assert_murmurhash3_x86_32(0x03d95171, "kct4k9jkNK", 0x05d4ab2d);
assert_murmurhash3_x86_32(0x7be2bea3, "AYsKPXkqIw", 0x601c1b35);
assert_murmurhash3_x86_32(0xa9ba5109, "8v6RFp037i", 0x4d084779);
assert_murmurhash3_x86_32(0xd7ac229c, "BkxIDSvmN5", 0x7335fb60);
assert_murmurhash3_x86_32(0x438af124, "fO7c8ySDx0", 0x37f2a5ae);
assert_murmurhash3_x86_32(0xccc7967c, "DSJ3QN860J", 0x14b012c8);
assert_murmurhash3_x86_32(0x1d0fcdae, "ys2qVmdZGq", 0x49d63415);
assert_murmurhash3_x86_32(0x4c775fad, "MATituNtBqj", 0x6b5dc0f4);
assert_murmurhash3_x86_32(0x17d4e9d5, "r75Ab6hAEHA", 0x133fadea);
assert_murmurhash3_x86_32(0xc4c7ed03, "DWfCADpkdgC", 0x1f054942);
assert_murmurhash3_x86_32(0x5c27acf1, "Lp6mOpEdvIL", 0x6be5f8a6);
assert_murmurhash3_x86_32(0x2858694c, "O3eqIOslKFX", 0x1882c9f7);
assert_murmurhash3_x86_32(0xc0fa3d3f, "ikDtzg8kDeE", 0x0bbf7d11);
assert_murmurhash3_x86_32(0x2d373d9a, "BgD4YoJMpXa", 0x73745d2c);
assert_murmurhash3_x86_32(0x5ae23c24, "MCrufnNxHp1", 0x6b2df5de);
assert_murmurhash3_x86_32(0x35942600, "F7ygjcGUtjZ", 0x27d7afaf);
assert_murmurhash3_x86_32(0x3615e0ae, "FIa3Dk9nVAS", 0x6c05663c);
assert_murmurhash3_x86_32(0xa5b3a4d7, "XDGD2FRFKqW3", 0x4a8e0f3c);
assert_murmurhash3_x86_32(0xe6d34f5d, "AX1RUrwAzxeT", 0x1963c5de);
assert_murmurhash3_x86_32(0xdd0415c1, "BMdrVa3BL5eD", 0x14ecaabb);
assert_murmurhash3_x86_32(0x19e00c1c, "Y1F1zdWB4P0y", 0x65f40099);
assert_murmurhash3_x86_32(0x35273cf7, "z6BqIbbVA66D", 0x21ff80d3);
assert_murmurhash3_x86_32(0xad457e84, "PHTqqPr3OZgK", 0x658ce6ce);
assert_murmurhash3_x86_32(0x1cacbbbb, "kzySX5WwvFHE", 0x4595c4e4);
assert_murmurhash3_x86_32(0xaf2ea51c, "gKETpuAgU05k", 0x6acce2b7);
assert_murmurhash3_x86_32(0x493b2673, "TjhCRB9ns6so", 0x1e29369f);
assert_murmurhash3_x86_32(0xff3d954a, "V1gACwkeqIG0", 0x796c1370);
assert_murmurhash3_x86_32(0x6cfdb1fb, "zZ2T0TaitZRDm", 0x27d44dd7);
assert_murmurhash3_x86_32(0x8978de82, "HOGgJHxhh3Awt", 0x409bb938);
assert_murmurhash3_x86_32(0xd0d8bf39, "cseLreCp6MHzL", 0x6cc32945);
assert_murmurhash3_x86_32(0x16c69bd0, "a5QQTw6BeDSvH", 0x7238e3b6);
assert_murmurhash3_x86_32(0x509e85d5, "ZaIcAXXZ9ypfk", 0x42e1c174);
assert_murmurhash3_x86_32(0x16a4cf34, "P4BX9sN1YUAax", 0x0c043706);
assert_murmurhash3_x86_32(0xdcb3aba7, "GeTFnzPVuLVEj", 0x201dc2fc);
assert_murmurhash3_x86_32(0x543191bb, "TBmGFWENorMkl", 0x4a750e68);
assert_murmurhash3_x86_32(0x4935e8ee, "vRM95DMqbAmFl", 0x1970e23f);
assert_murmurhash3_x86_32(0x113cd28c, "jExbfRRTNvE9V", 0x4259aaa0);
assert_murmurhash3_x86_32(0x984cd50c, "ugKX7w6b8QzjqV", 0x617caf92);
assert_murmurhash3_x86_32(0x0a25ae7e, "JagnFrBxgukM8s", 0x04a5cbc1);
assert_murmurhash3_x86_32(0x922f28b4, "wWVgT1KXaSLJ9c", 0x73be353f);
assert_murmurhash3_x86_32(0xa73e1a95, "8VNmhrNSY1kgNq", 0x6cd0c52c);
assert_murmurhash3_x86_32(0xf46d4e0c, "sWCmatlUqvLceU", 0x71116526);
assert_murmurhash3_x86_32(0xc9406d50, "R0fDmu47k09EeU", 0x7b028cc6);
assert_murmurhash3_x86_32(0xdf922105, "VlYxF6101ptMzH", 0x2e1098a4);
assert_murmurhash3_x86_32(0x53017bf2, "TxF9aZBc5WccAq", 0x0c852728);
assert_murmurhash3_x86_32(0xd874ca90, "DjpzR2GQ0FftpP", 0x5ac63851);
assert_murmurhash3_x86_32(0x4d5a5b39, "3IEGPOGrYJlaVT", 0x7874e0f4);
assert_murmurhash3_x86_32(0x58c88556, "ZuSn4HnIxonMPMz", 0x039f6a92);
assert_murmurhash3_x86_32(0xb24c9a68, "NiAu5m8ukRPsKJZ", 0x06adf34b);
assert_murmurhash3_x86_32(0x679f4ac0, "dP4fwsX1OiLD4lA", 0x093208b2);
assert_murmurhash3_x86_32(0x532c6433, "BakGxs9PhYiZFhG", 0x7bb93d13);
assert_murmurhash3_x86_32(0x2a7819c0, "6L6Ab4BZkmApF8f", 0x3cb78b3a);
assert_murmurhash3_x86_32(0x74c59cae, "hxVOX4Bc0TbG8Rx", 0x1d8110d8);
assert_murmurhash3_x86_32(0xba356ecc, "ADMLFlI0GhnlpAA", 0x4332771a);
assert_murmurhash3_x86_32(0xc69a4b08, "5vs2z4exVnd4eIg", 0x5f5cfda7);
assert_murmurhash3_x86_32(0xc1de77d7, "k1yZkfX0wkkVSUA", 0x4d2576a5);
assert_murmurhash3_x86_32(0xc0005f6d, "p2XW49t0wF2Lngy", 0x5acc225d);
assert_murmurhash3_x64_64(0xa6fca4d63e0eb97eL, "z", 0x5942ad3d);
assert_murmurhash3_x64_64(0xe0f5dbd0eda15dcaL, "z", 0xd942ad3d);
assert_murmurhash3_x64_64(0xf9b1243c1294ad79L, "2", 0x71905ea7);
assert_murmurhash3_x64_64(0x2db3aa2944cf7005L, "2", 0xf1905ea7);
assert_murmurhash3_x64_64(0xf621d456b28f4de5L, "d", 0x64ee4d11);
assert_murmurhash3_x64_64(0x3d2b3a5d19c9c3e8L, "d", 0xe4ee4d11);
assert_murmurhash3_x64_64(0x70a655b76fee736aL, "L", 0x29fc249d);
assert_murmurhash3_x64_64(0x4696cfcba0e6b671L, "L", 0xa9fc249d);
assert_murmurhash3_x64_64(0x7f9f62611e52dc35L, "K", 0x14aae517);
assert_murmurhash3_x64_64(0x3b02b688e1a50d2cL, "K", 0x94aae517);
assert_murmurhash3_x64_64(0x936de9d1ed11cbb4L, "N", 0x7ce81cf5);
assert_murmurhash3_x64_64(0x1319696ad094b1f2L, "N", 0xfce81cf5);
assert_murmurhash3_x64_64(0x1e407e07821067abL, "C", 0x743d7979);
assert_murmurhash3_x64_64(0xe78c8b0cbacee127L, "C", 0xf43d7979);
assert_murmurhash3_x64_64(0xb24065097ec5b7e3L, "G", 0x29bc6a00);
assert_murmurhash3_x64_64(0x7b69fd3800d69a81L, "G", 0xa9bc6a00);
assert_murmurhash3_x64_64(0xd33b3d98a1749a67L, "h", 0x53dd72a4);
assert_murmurhash3_x64_64(0x3c6fc6135d0dd21fL, "h", 0xd3dd72a4);
assert_murmurhash3_x64_64(0xd52442dbe5e0fe10L, "u", 0x74fd834f);
assert_murmurhash3_x64_64(0x3a9553995d0912b9L, "u", 0xf4fd834f);
assert_murmurhash3_x64_64(0xe1b85aed290b7fc3L, "87", 0x500dc5a2);
assert_murmurhash3_x64_64(0x76df971d54e2ef9fL, "87", 0xd00dc5a2);
assert_murmurhash3_x64_64(0xa31e32bd5bba6652L, "Nx", 0x10d5412e);
assert_murmurhash3_x64_64(0xe0f8e90fcdc280caL, "Nx", 0x90d5412e);
assert_murmurhash3_x64_64(0xae61ed4bdc278b13L, "cI", 0x142f8840);
assert_murmurhash3_x64_64(0x6e2109f4053b7f0eL, "cI", 0x942f8840);
assert_murmurhash3_x64_64(0xca6c3a9f91bd1741L, "hW", 0x6a059ad5);
assert_murmurhash3_x64_64(0xbf7ead591645e979L, "hW", 0xea059ad5);
assert_murmurhash3_x64_64(0x1df8a2648ad52718L, "EZ", 0x5b95f97d);
assert_murmurhash3_x64_64(0xbc91749523f6e674L, "EZ", 0xdb95f97d);
assert_murmurhash3_x64_64(0x81ceb1f55acb717cL, "RE", 0x64d5803e);
assert_murmurhash3_x64_64(0xa909328fe4649b9dL, "RE", 0xe4d5803e);
assert_murmurhash3_x64_64(0x86c1d8de579f327cL, "Jo", 0x79806555);
assert_murmurhash3_x64_64(0x87fff1e71e202c6bL, "Jo", 0xf9806555);
assert_murmurhash3_x64_64(0x32a2d468afca0343L, "wp", 0x673e5532);
assert_murmurhash3_x64_64(0xe8885c5e0b829c6dL, "wp", 0xe73e5532);
assert_murmurhash3_x64_64(0x85fb2007b85b6499L, "X5", 0x10fabf32);
assert_murmurhash3_x64_64(0xb3305b8e92599504L, "X5", 0x90fabf32);
assert_murmurhash3_x64_64(0x80810e50c192fa49L, "eX", 0x3e3f0006);
assert_murmurhash3_x64_64(0x02826fdc5218c1b7L, "eX", 0xbe3f0006);
assert_murmurhash3_x64_64(0xe5eba93b34ba7a71L, "t5g", 0x696736c5);
assert_murmurhash3_x64_64(0x9ddf2efcbd53ffc5L, "t5g", 0xe96736c5);
assert_murmurhash3_x64_64(0x05a626d6bbab60f1L, "RLH", 0x20132ae9);
assert_murmurhash3_x64_64(0x8c262ba98d20cf67L, "RLH", 0xa0132ae9);
assert_murmurhash3_x64_64(0xfc7e2dc47f835549L, "tVk", 0x282bd870);
assert_murmurhash3_x64_64(0xa25622c830b03c08L, "tVk", 0xa82bd870);
assert_murmurhash3_x64_64(0xafc167ef6438e24cL, "LYn", 0x54079486);
assert_murmurhash3_x64_64(0x619238dde5708ee0L, "LYn", 0xd4079486);
assert_murmurhash3_x64_64(0x12e2fb07c3b06731L, "O1k", 0x46d645a5);
assert_murmurhash3_x64_64(0xaeb8272b127922c4L, "O1k", 0xc6d645a5);
assert_murmurhash3_x64_64(0xfab82b0fe61905d9L, "n43", 0x26abfa0c);
assert_murmurhash3_x64_64(0x7453001b4a81f9ecL, "n43", 0xa6abfa0c);
assert_murmurhash3_x64_64(0x293c940111f90711L, "oZR", 0x2a968e7e);
assert_murmurhash3_x64_64(0xd3fecbecab749d3cL, "oZR", 0xaa968e7e);
assert_murmurhash3_x64_64(0xbd76c681714bf8beL, "dPb", 0x3dbae557);
assert_murmurhash3_x64_64(0xa87db71cabb91ba9L, "dPb", 0xbdbae557);
assert_murmurhash3_x64_64(0x84d4b82090f419d3L, "Spe", 0x54657cad);
assert_murmurhash3_x64_64(0x8466e5a48b22a8f9L, "Spe", 0xd4657cad);
assert_murmurhash3_x64_64(0xd14e92954ec8c017L, "8UM", 0x3e48db7c);
assert_murmurhash3_x64_64(0x9aea37faf4edfce1L, "8UM", 0xbe48db7c);
assert_murmurhash3_x64_64(0xd82220a684d5575cL, "P4Yy", 0x1eaef7eb);
assert_murmurhash3_x64_64(0xea6c917d161a7a63L, "P4Yy", 0x9eaef7eb);
assert_murmurhash3_x64_64(0xc34e712999784f82L, "jIPi", 0x3520d897);
assert_murmurhash3_x64_64(0x146fde98c7a3294aL, "jIPi", 0xb520d897);
assert_murmurhash3_x64_64(0x1a5fc0f13e792a06L, "kF5l", 0x6afc31e1);
assert_murmurhash3_x64_64(0x0204837951fbc519L, "kF5l", 0xeafc31e1);
assert_murmurhash3_x64_64(0x3539ddb891c78c08L, "rkgt", 0x2e97cd16);
assert_murmurhash3_x64_64(0xbe43ba6f025b3478L, "rkgt", 0xae97cd16);
assert_murmurhash3_x64_64(0x08ad0a28633be94fL, "3CfW", 0x79e981e1);
assert_murmurhash3_x64_64(0xad1de795a1df1c42L, "3CfW", 0xf9e981e1);
assert_murmurhash3_x64_64(0x7bd0ff697ae22885L, "rD8l", 0x4525aef4);
assert_murmurhash3_x64_64(0xeac5efaaaefd4828L, "rD8l", 0xc525aef4);
assert_murmurhash3_x64_64(0xa3257abe0c0885d9L, "GbqE", 0x0cda2629);
assert_murmurhash3_x64_64(0x1ed402ed5b2b0815L, "GbqE", 0x8cda2629);
assert_murmurhash3_x64_64(0xbdf698c6823e81ffL, "rVoe", 0x5f826d97);
assert_murmurhash3_x64_64(0xc44c9cf9489fcf78L, "rVoe", 0xdf826d97);
assert_murmurhash3_x64_64(0xea82324d005f0c6fL, "pyQs", 0x2dc2c7d0);
assert_murmurhash3_x64_64(0xd132db91c4c1e580L, "pyQs", 0xadc2c7d0);
assert_murmurhash3_x64_64(0x441b0eb457158bf5L, "fTev", 0x5287e7bc);
assert_murmurhash3_x64_64(0x8f5e336a7f70b2edL, "fTev", 0xd287e7bc);
assert_murmurhash3_x64_64(0xa53e4231fed0bdd9L, "MxZZr", 0x60f7a89e);
assert_murmurhash3_x64_64(0x573c2c9f8ab82ab7L, "MxZZr", 0xe0f7a89e);
assert_murmurhash3_x64_64(0xe9b910b63470660cL, "r44Kq", 0x25c6baae);
assert_murmurhash3_x64_64(0x9c846c3f5e33cf60L, "r44Kq", 0xa5c6baae);
assert_murmurhash3_x64_64(0xf172670df26e8562L, "VRmBi", 0x37b23d99);
assert_murmurhash3_x64_64(0xcda4f1a71751933fL, "VRmBi", 0xb7b23d99);
assert_murmurhash3_x64_64(0x3924aa3337159204L, "Zxso5", 0x3b07509c);
assert_murmurhash3_x64_64(0xa43ed8c0b8f1364fL, "Zxso5", 0xbb07509c);
assert_murmurhash3_x64_64(0x937ddd28efa02344L, "QXxHc", 0x21626c57);
assert_murmurhash3_x64_64(0x957f4e5098f3e3f3L, "QXxHc", 0xa1626c57);
assert_murmurhash3_x64_64(0x8b3298b69045640dL, "3Wy0U", 0x2fa53851);
assert_murmurhash3_x64_64(0xe557ada9e2523294L, "3Wy0U", 0xafa53851);
assert_murmurhash3_x64_64(0x84f7bf31ee8cfadbL, "Xl113", 0x20af4caa);
assert_murmurhash3_x64_64(0x1d092bd7649cf56fL, "Xl113", 0xa0af4caa);
assert_murmurhash3_x64_64(0xd297767b6638ca69L, "KXj6y", 0x136fd981);
assert_murmurhash3_x64_64(0xf43aa25e5c126aa1L, "KXj6y", 0x936fd981);
assert_murmurhash3_x64_64(0x9d41e07d41827a06L, "ox6QT", 0x57c7dc38);
assert_murmurhash3_x64_64(0xa1d32016b95c58ceL, "ox6QT", 0xd7c7dc38);
assert_murmurhash3_x64_64(0xd9b7810b8498b5fcL, "7H7Eo", 0x5330848d);
assert_murmurhash3_x64_64(0x5c1b9f873f27ff33L, "7H7Eo", 0xd330848d);
assert_murmurhash3_x64_64(0x6f0a2a4380dbf274L, "Dpcbq6", 0x62179576);
assert_murmurhash3_x64_64(0xdd0ea1f8da26f2e9L, "Dpcbq6", 0xe2179576);
assert_murmurhash3_x64_64(0xe54d8f8ab88b7171L, "lpcksC", 0x12185758);
assert_murmurhash3_x64_64(0xd0b1045daed16381L, "lpcksC", 0x92185758);
assert_murmurhash3_x64_64(0xc72b9c75b10c9963L, "pVbXuz", 0x212382bc);
assert_murmurhash3_x64_64(0x6e250815ba726fa8L, "pVbXuz", 0xa12382bc);
assert_murmurhash3_x64_64(0xc990fd7ef463f19dL, "znlHu2", 0x16a5a1f8);
assert_murmurhash3_x64_64(0x81cea4f63297028dL, "znlHu2", 0x96a5a1f8);
assert_murmurhash3_x64_64(0x97d15eadb4c92183L, "7q4KQg", 0x0f82524e);
assert_murmurhash3_x64_64(0xa89be99538633821L, "7q4KQg", 0x8f82524e);
assert_murmurhash3_x64_64(0xdd553de0a93acb70L, "ek4xJe", 0x1b09ee75);
assert_murmurhash3_x64_64(0x707807340f350e42L, "ek4xJe", 0x9b09ee75);
assert_murmurhash3_x64_64(0xde378b5af10b4ac1L, "bQWzJ7", 0x2c3267c8);
assert_murmurhash3_x64_64(0x2d098433d609f857L, "bQWzJ7", 0xac3267c8);
assert_murmurhash3_x64_64(0xa712cd195611cd35L, "cEXLZi", 0x73d20e88);
assert_murmurhash3_x64_64(0x2cc031739638870eL, "cEXLZi", 0xf3d20e88);
assert_murmurhash3_x64_64(0x16c6e0bc6db1acefL, "sjczJg", 0x33971a37);
assert_murmurhash3_x64_64(0xcdf52c033c7564c4L, "sjczJg", 0xb3971a37);
assert_murmurhash3_x64_64(0x0a7f710bc18e083eL, "xubJMe", 0x5c48c98c);
assert_murmurhash3_x64_64(0x973f12fcec9bb11dL, "xubJMe", 0xdc48c98c);
assert_murmurhash3_x64_64(0x5e97a97a5868665dL, "vq8Fh2c", 0x1f66134c);
assert_murmurhash3_x64_64(0x01d299986f35c3adL, "vq8Fh2c", 0x9f66134c);
assert_murmurhash3_x64_64(0x0d42a63dd5b55c31L, "9GaLBTK", 0x06e59da6);
assert_murmurhash3_x64_64(0x1a7a4c790e6e16c4L, "9GaLBTK", 0x86e59da6);
assert_murmurhash3_x64_64(0x7c2fbd63ffd0bd54L, "lcbwJSM", 0x57465553);
assert_murmurhash3_x64_64(0xf2628d6cc4c53034L, "lcbwJSM", 0xd7465553);
assert_murmurhash3_x64_64(0x934d4166dd44587cL, "ofa7Zom", 0x5a7118db);
assert_murmurhash3_x64_64(0xf0d52864cfe7d01fL, "ofa7Zom", 0xda7118db);
assert_murmurhash3_x64_64(0x1a1dd5b72e593f4aL, "Ps84uk2", 0x5b0a53bf);
assert_murmurhash3_x64_64(0xfbdbcad39912360fL, "Ps84uk2", 0xdb0a53bf);
assert_murmurhash3_x64_64(0x8d4b8e4f0a514088L, "YbkA44s", 0x1f5eee28);
assert_murmurhash3_x64_64(0x5d91a1769f945163L, "YbkA44s", 0x9f5eee28);
assert_murmurhash3_x64_64(0x798754262fcd2ba3L, "fDUYvEX", 0x14c22e1f);
assert_murmurhash3_x64_64(0x3ec159b798555b2fL, "fDUYvEX", 0x94c22e1f);
assert_murmurhash3_x64_64(0x3b85a4e7149aee3cL, "R8XRujm", 0x6dab91f5);
assert_murmurhash3_x64_64(0xacacc0adb2264e0bL, "R8XRujm", 0xedab91f5);
assert_murmurhash3_x64_64(0xa8e4f6f60020b4dfL, "JsldMmM", 0x540d2fa0);
assert_murmurhash3_x64_64(0x9f35ac7e0197fb32L, "JsldMmM", 0xd40d2fa0);
assert_murmurhash3_x64_64(0x3a0a8ba6bdbb8a0eL, "x6kB8MX", 0x67e8ea83);
assert_murmurhash3_x64_64(0x8f898f2976331671L, "x6kB8MX", 0xe7e8ea83);
assert_murmurhash3_x64_64(0x93247c2b01725d60L, "nRkRwhIn", 0x207e8cf7);
assert_murmurhash3_x64_64(0x10418c5a4d8d128bL, "nRkRwhIn", 0xa07e8cf7);
assert_murmurhash3_x64_64(0xcba7635172d8c7ddL, "FeSWy9F0", 0x5924b4c3);
assert_murmurhash3_x64_64(0x17498cb7a9ca60bdL, "FeSWy9F0", 0xd924b4c3);
assert_murmurhash3_x64_64(0xc90245457a8885a7L, "QKRCtmGN", 0x2a50bd07);
assert_murmurhash3_x64_64(0xbf0e2d7e93f5586fL, "QKRCtmGN", 0xaa50bd07);
assert_murmurhash3_x64_64(0x7165b165c8360211L, "PyTaTKsK", 0x27d1cc44);
assert_murmurhash3_x64_64(0xe5e87aed270f9ce3L, "PyTaTKsK", 0xa7d1cc44);
assert_murmurhash3_x64_64(0x7ddf3fa5ab52cec0L, "KrC56OX3", 0x25307b55);
assert_murmurhash3_x64_64(0xd7f8ff97735deb59L, "KrC56OX3", 0xa5307b55);
assert_murmurhash3_x64_64(0xb5a9279e82aac1b3L, "7GlAx4p9", 0x2c6329e5);
assert_murmurhash3_x64_64(0xac7590a8d766cc42L, "7GlAx4p9", 0xac6329e5);
assert_murmurhash3_x64_64(0xd9ea08745dd2d53fL, "LOiEBQxL", 0x03cc6c96);
assert_murmurhash3_x64_64(0x93983e4806963ae4L, "LOiEBQxL", 0x83cc6c96);
assert_murmurhash3_x64_64(0x4d186387e5d8b881L, "gesuWT41", 0x493952a8);
assert_murmurhash3_x64_64(0x4d85476289fdecb8L, "gesuWT41", 0xc93952a8);
assert_murmurhash3_x64_64(0x658f5950973e3f81L, "RXZc2FvC", 0x712c0c42);
assert_murmurhash3_x64_64(0x7be2648eac3384f5L, "RXZc2FvC", 0xf12c0c42);
assert_murmurhash3_x64_64(0xb5243b0035e097c1L, "x1k4K6nm", 0x21252bd5);
assert_murmurhash3_x64_64(0x913dde647a98c973L, "x1k4K6nm", 0xa1252bd5);
assert_murmurhash3_x64_64(0xf856f841e957285dL, "bUqT8CmCy", 0x679bf501);
assert_murmurhash3_x64_64(0x618e3d1f4308c7a2L, "bUqT8CmCy", 0xe79bf501);
assert_murmurhash3_x64_64(0xa8e29343f374cb67L, "Fy9wt8GvM", 0x2a8c58d5);
assert_murmurhash3_x64_64(0xbb86409aff0c6537L, "Fy9wt8GvM", 0xaa8c58d5);
assert_murmurhash3_x64_64(0xfdb02a7f0874b643L, "7WH6eLOk6", 0x4c1c6c2c);
assert_murmurhash3_x64_64(0xf494040e5bbc2257L, "7WH6eLOk6", 0xcc1c6c2c);
assert_murmurhash3_x64_64(0xa2af5b940b5b3d0fL, "1i2sba4vL", 0x3aeb8b07);
assert_murmurhash3_x64_64(0x1cdb89f381589944L, "1i2sba4vL", 0xbaeb8b07);
assert_murmurhash3_x64_64(0x398a2392bb4acfccL, "Hq7OUAXy5", 0x3ee5d761);
assert_murmurhash3_x64_64(0xd4e06f80c1450d28L, "Hq7OUAXy5", 0xbee5d761);
assert_murmurhash3_x64_64(0x91bb48ca86628216L, "HcDmgQ62a", 0x1dc3649c);
assert_murmurhash3_x64_64(0x7d47f53189777091L, "HcDmgQ62a", 0x9dc3649c);
assert_murmurhash3_x64_64(0x58927263c2de3207L, "ccudCEnE7", 0x025dfa90);
assert_murmurhash3_x64_64(0x8f178d4c8e5b3323L, "ccudCEnE7", 0x825dfa90);
assert_murmurhash3_x64_64(0x426d002e5784c30aL, "fMwmBoUwU", 0x2b5b80d5);
assert_murmurhash3_x64_64(0x9d0fcc51347742bcL, "fMwmBoUwU", 0xab5b80d5);
assert_murmurhash3_x64_64(0x6b15280a9bd9cb8bL, "dA9OWpE2r", 0x23cbd625);
assert_murmurhash3_x64_64(0x798af9ce6a756070L, "dA9OWpE2r", 0xa3cbd625);
assert_murmurhash3_x64_64(0xeb385ccc7eb7eb6aL, "fCXHPzjad", 0x0813cf01);
assert_murmurhash3_x64_64(0xce4e81da118b8a50L, "fCXHPzjad", 0x8813cf01);
assert_murmurhash3_x64_64(0xe27e904a557952d9L, "hrbLLM8Gg2", 0x60061445);
assert_murmurhash3_x64_64(0x8ce1b6b858f45d44L, "hrbLLM8Gg2", 0xe0061445);
assert_murmurhash3_x64_64(0x8d58cff892b329acL, "rCNexcQxCD", 0x1d31c96c);
assert_murmurhash3_x64_64(0x0cc0bb7c3a7d8d22L, "rCNexcQxCD", 0x9d31c96c);
assert_murmurhash3_x64_64(0x7d4323af83bec0d2L, "cAs17J9j8q", 0x6c7b6c7a);
assert_murmurhash3_x64_64(0x3d002ef23d0a4273L, "cAs17J9j8q", 0xec7b6c7a);
assert_murmurhash3_x64_64(0x040faa0771b49b03L, "kct4k9jkNK", 0x05d4ab2d);
assert_murmurhash3_x64_64(0x9c61df2b7417c21bL, "kct4k9jkNK", 0x85d4ab2d);
assert_murmurhash3_x64_64(0x969e15e57d384309L, "AYsKPXkqIw", 0x601c1b35);
assert_murmurhash3_x64_64(0x4db267abe2fdcfa8L, "AYsKPXkqIw", 0xe01c1b35);
assert_murmurhash3_x64_64(0x054ab6fd1bc43751L, "8v6RFp037i", 0x4d084779);
assert_murmurhash3_x64_64(0xead5447b57226b4bL, "8v6RFp037i", 0xcd084779);
assert_murmurhash3_x64_64(0x55a92a79fe8ee484L, "BkxIDSvmN5", 0x7335fb60);
assert_murmurhash3_x64_64(0x6525ed107f720c5cL, "BkxIDSvmN5", 0xf335fb60);
assert_murmurhash3_x64_64(0x32254dc61e3b3d4bL, "fO7c8ySDx0", 0x37f2a5ae);
assert_murmurhash3_x64_64(0xe5d799d77b18b5a3L, "fO7c8ySDx0", 0xb7f2a5ae);
assert_murmurhash3_x64_64(0xa145f08ec7eb16c6L, "DSJ3QN860J", 0x14b012c8);
assert_murmurhash3_x64_64(0x47b5b44970828481L, "DSJ3QN860J", 0x94b012c8);
assert_murmurhash3_x64_64(0x42d89a2c08ac44a4L, "ys2qVmdZGq", 0x49d63415);
assert_murmurhash3_x64_64(0xf1c31424a57037ddL, "ys2qVmdZGq", 0xc9d63415);
assert_murmurhash3_x64_64(0x27bffa3e20d9cc88L, "MATituNtBqj", 0x6b5dc0f4);
assert_murmurhash3_x64_64(0x1a48ee09c9f4cc06L, "MATituNtBqj", 0xeb5dc0f4);
assert_murmurhash3_x64_64(0x5691f7473736f3fbL, "r75Ab6hAEHA", 0x133fadea);
assert_murmurhash3_x64_64(0xe0b487a8fd596fc2L, "r75Ab6hAEHA", 0x933fadea);
assert_murmurhash3_x64_64(0xfc48349bfd6b56e1L, "DWfCADpkdgC", 0x1f054942);
assert_murmurhash3_x64_64(0x483bf6382101e279L, "DWfCADpkdgC", 0x9f054942);
assert_murmurhash3_x64_64(0xdb31bdb2b45ff351L, "Lp6mOpEdvIL", 0x6be5f8a6);
assert_murmurhash3_x64_64(0x6b35dcd7aa11ecbbL, "Lp6mOpEdvIL", 0xebe5f8a6);
assert_murmurhash3_x64_64(0xe9bc3761d239f10bL, "O3eqIOslKFX", 0x1882c9f7);
assert_murmurhash3_x64_64(0x0431b34d64a49d52L, "O3eqIOslKFX", 0x9882c9f7);
assert_murmurhash3_x64_64(0x989908e6994fc13bL, "ikDtzg8kDeE", 0x0bbf7d11);
assert_murmurhash3_x64_64(0x3552567355a13f6bL, "ikDtzg8kDeE", 0x8bbf7d11);
assert_murmurhash3_x64_64(0x3d1784d85d024eefL, "BgD4YoJMpXa", 0x73745d2c);
assert_murmurhash3_x64_64(0xd0c627dade12c706L, "BgD4YoJMpXa", 0xf3745d2c);
assert_murmurhash3_x64_64(0xa31353064e8e5581L, "MCrufnNxHp1", 0x6b2df5de);
assert_murmurhash3_x64_64(0x1bcd40da7dbab150L, "MCrufnNxHp1", 0xeb2df5de);
assert_murmurhash3_x64_64(0x5c14dd1fa0c27931L, "F7ygjcGUtjZ", 0x27d7afaf);
assert_murmurhash3_x64_64(0xc29b543577d847f3L, "F7ygjcGUtjZ", 0xa7d7afaf);
assert_murmurhash3_x64_64(0xad6a60ebc626f696L, "FIa3Dk9nVAS", 0x6c05663c);
assert_murmurhash3_x64_64(0x980f15f3ff2828d3L, "FIa3Dk9nVAS", 0xec05663c);
assert_murmurhash3_x64_64(0xe5a30dc9f2da30f9L, "XDGD2FRFKqW3", 0x4a8e0f3c);
assert_murmurhash3_x64_64(0x5cd8e71f8af5003fL, "XDGD2FRFKqW3", 0xca8e0f3c);
assert_murmurhash3_x64_64(0x28bc443294717831L, "AX1RUrwAzxeT", 0x1963c5de);
assert_murmurhash3_x64_64(0xfa7764a42c55a0ecL, "AX1RUrwAzxeT", 0x9963c5de);
assert_murmurhash3_x64_64(0x4a65b3241838e348L, "BMdrVa3BL5eD", 0x14ecaabb);
assert_murmurhash3_x64_64(0x164c8004603b4af3L, "BMdrVa3BL5eD", 0x94ecaabb);
assert_murmurhash3_x64_64(0x7ecb780f5d634a43L, "Y1F1zdWB4P0y", 0x65f40099);
assert_murmurhash3_x64_64(0x7339ac6e5f56670eL, "Y1F1zdWB4P0y", 0xe5f40099);
assert_murmurhash3_x64_64(0x7143a45da501791bL, "z6BqIbbVA66D", 0x21ff80d3);
assert_murmurhash3_x64_64(0xf4f2ce640d890109L, "z6BqIbbVA66D", 0xa1ff80d3);
assert_murmurhash3_x64_64(0xa4fe7b81c2e25675L, "PHTqqPr3OZgK", 0x658ce6ce);
assert_murmurhash3_x64_64(0x1c4daf0d87bf08adL, "PHTqqPr3OZgK", 0xe58ce6ce);
assert_murmurhash3_x64_64(0x0e988bc61802bfcbL, "kzySX5WwvFHE", 0x4595c4e4);
assert_murmurhash3_x64_64(0x4f70a6c01b14fc24L, "kzySX5WwvFHE", 0xc595c4e4);
assert_murmurhash3_x64_64(0x84313ce5fc15d1ccL, "gKETpuAgU05k", 0x6acce2b7);
assert_murmurhash3_x64_64(0xbe9bf4d2a6f37d63L, "gKETpuAgU05k", 0xeacce2b7);
assert_murmurhash3_x64_64(0x669b5b4d6f6f11beL, "TjhCRB9ns6so", 0x1e29369f);
assert_murmurhash3_x64_64(0x6d1d2c43022f29d1L, "TjhCRB9ns6so", 0x9e29369f);
assert_murmurhash3_x64_64(0xc455102e0ad07346L, "V1gACwkeqIG0", 0x796c1370);
assert_murmurhash3_x64_64(0xfe3c12c8ce846f66L, "V1gACwkeqIG0", 0xf96c1370);
assert_murmurhash3_x64_64(0x00c4a5d789d3f23fL, "zZ2T0TaitZRDm", 0x27d44dd7);
assert_murmurhash3_x64_64(0x40fac12de28a3a96L, "zZ2T0TaitZRDm", 0xa7d44dd7);
assert_murmurhash3_x64_64(0xd90008a6c547750bL, "HOGgJHxhh3Awt", 0x409bb938);
assert_murmurhash3_x64_64(0x7c4a36878e848577L, "HOGgJHxhh3Awt", 0xc09bb938);
assert_murmurhash3_x64_64(0x6796bd87956bae76L, "cseLreCp6MHzL", 0x6cc32945);
assert_murmurhash3_x64_64(0xeb231c6435da66c1L, "cseLreCp6MHzL", 0xecc32945);
assert_murmurhash3_x64_64(0x50772f03b5967e80L, "a5QQTw6BeDSvH", 0x7238e3b6);
assert_murmurhash3_x64_64(0xf08313506907c922L, "a5QQTw6BeDSvH", 0xf238e3b6);
assert_murmurhash3_x64_64(0x58780f031b56728cL, "ZaIcAXXZ9ypfk", 0x42e1c174);
assert_murmurhash3_x64_64(0xf2d17b059be60f6dL, "ZaIcAXXZ9ypfk", 0xc2e1c174);
assert_murmurhash3_x64_64(0xa21ca79e36a00547L, "P4BX9sN1YUAax", 0x0c043706);
assert_murmurhash3_x64_64(0x4bd55054f3af2305L, "P4BX9sN1YUAax", 0x8c043706);
assert_murmurhash3_x64_64(0x45a7eb0c5ea3c357L, "GeTFnzPVuLVEj", 0x201dc2fc);
assert_murmurhash3_x64_64(0x889c693e2c15394bL, "GeTFnzPVuLVEj", 0xa01dc2fc);
assert_murmurhash3_x64_64(0xb01828a6fe78a106L, "TBmGFWENorMkl", 0x4a750e68);
assert_murmurhash3_x64_64(0x5376a244eb63c0baL, "TBmGFWENorMkl", 0xca750e68);
assert_murmurhash3_x64_64(0xc4d53d2457664ac1L, "vRM95DMqbAmFl", 0x1970e23f);
assert_murmurhash3_x64_64(0x38ed69c36c656058L, "vRM95DMqbAmFl", 0x9970e23f);
assert_murmurhash3_x64_64(0xbc7af5ea01f6c73fL, "jExbfRRTNvE9V", 0x4259aaa0);
assert_murmurhash3_x64_64(0xaa2f4ab8eae105a2L, "jExbfRRTNvE9V", 0xc259aaa0);
assert_murmurhash3_x64_64(0x81aa0c10ccfc51eeL, "ugKX7w6b8QzjqV", 0x617caf92);
assert_murmurhash3_x64_64(0xca18ce46bb02557cL, "ugKX7w6b8QzjqV", 0xe17caf92);
assert_murmurhash3_x64_64(0x333582819e748166L, "JagnFrBxgukM8s", 0x04a5cbc1);
assert_murmurhash3_x64_64(0xef9a69a82f80173eL, "JagnFrBxgukM8s", 0x84a5cbc1);
assert_murmurhash3_x64_64(0xbd3af10da4a6115bL, "wWVgT1KXaSLJ9c", 0x73be353f);
assert_murmurhash3_x64_64(0xf335f23f01ee5d82L, "wWVgT1KXaSLJ9c", 0xf3be353f);
assert_murmurhash3_x64_64(0x1007b9a7485fd7e1L, "8VNmhrNSY1kgNq", 0x6cd0c52c);
assert_murmurhash3_x64_64(0xe1fed56825abafddL, "8VNmhrNSY1kgNq", 0xecd0c52c);
assert_murmurhash3_x64_64(0xf253fe6bfc592334L, "sWCmatlUqvLceU", 0x71116526);
assert_murmurhash3_x64_64(0x76bd34604bb515abL, "sWCmatlUqvLceU", 0xf1116526);
assert_murmurhash3_x64_64(0x6f9f8c055effbdc6L, "R0fDmu47k09EeU", 0x7b028cc6);
assert_murmurhash3_x64_64(0x696f3aa83e91287cL, "R0fDmu47k09EeU", 0xfb028cc6);
assert_murmurhash3_x64_64(0x6479a8bc4d651a44L, "VlYxF6101ptMzH", 0x2e1098a4);
assert_murmurhash3_x64_64(0x3f2914bdef0ec144L, "VlYxF6101ptMzH", 0xae1098a4);
assert_murmurhash3_x64_64(0x157c82fc2819e16fL, "TxF9aZBc5WccAq", 0x0c852728);
assert_murmurhash3_x64_64(0xaa5ab7803bbd6694L, "TxF9aZBc5WccAq", 0x8c852728);
assert_murmurhash3_x64_64(0x02bd1a031be179ccL, "DjpzR2GQ0FftpP", 0x5ac63851);
assert_murmurhash3_x64_64(0x0766eb88c44e6350L, "DjpzR2GQ0FftpP", 0xdac63851);
assert_murmurhash3_x64_64(0xad90c1bf00873010L, "3IEGPOGrYJlaVT", 0x7874e0f4);
assert_murmurhash3_x64_64(0xab5b7c94d99dcffbL, "3IEGPOGrYJlaVT", 0xf874e0f4);
assert_murmurhash3_x64_64(0x6e0f0ccd77412f93L, "ZuSn4HnIxonMPMz", 0x039f6a92);
assert_murmurhash3_x64_64(0x7fcbad3919158bf5L, "ZuSn4HnIxonMPMz", 0x839f6a92);
assert_murmurhash3_x64_64(0x4b279e7cd792bd97L, "NiAu5m8ukRPsKJZ", 0x06adf34b);
assert_murmurhash3_x64_64(0x4d083a391fe0eeb6L, "NiAu5m8ukRPsKJZ", 0x86adf34b);
assert_murmurhash3_x64_64(0x695430997a2111d5L, "dP4fwsX1OiLD4lA", 0x093208b2);
assert_murmurhash3_x64_64(0x922a8e6aa1b122f3L, "dP4fwsX1OiLD4lA", 0x893208b2);
assert_murmurhash3_x64_64(0x5685705d159d03e5L, "BakGxs9PhYiZFhG", 0x7bb93d13);
assert_murmurhash3_x64_64(0x3d7b838461f5aa9bL, "BakGxs9PhYiZFhG", 0xfbb93d13);
assert_murmurhash3_x64_64(0x67bb062df1ccd97fL, "6L6Ab4BZkmApF8f", 0x3cb78b3a);
assert_murmurhash3_x64_64(0x5650d71bd6ca83ddL, "6L6Ab4BZkmApF8f", 0xbcb78b3a);
assert_murmurhash3_x64_64(0x512c7aac3a980e48L, "hxVOX4Bc0TbG8Rx", 0x1d8110d8);
assert_murmurhash3_x64_64(0xcfcfc91f28fa5a5eL, "hxVOX4Bc0TbG8Rx", 0x9d8110d8);
assert_murmurhash3_x64_64(0x4d609092cedd0e54L, "ADMLFlI0GhnlpAA", 0x4332771a);
assert_murmurhash3_x64_64(0x0ffc0a4d2b43a073L, "ADMLFlI0GhnlpAA", 0xc332771a);
assert_murmurhash3_x64_64(0x0f5765db241c49feL, "5vs2z4exVnd4eIg", 0x5f5cfda7);
assert_murmurhash3_x64_64(0x8e4b377b9c5d4ac8L, "5vs2z4exVnd4eIg", 0xdf5cfda7);
assert_murmurhash3_x64_64(0x3fe66ab1872e26afL, "k1yZkfX0wkkVSUA", 0x4d2576a5);
assert_murmurhash3_x64_64(0xdd6e9eb0f80beed3L, "k1yZkfX0wkkVSUA", 0xcd2576a5);
assert_murmurhash3_x64_64(0x2bf925b511dc279aL, "p2XW49t0wF2Lngy", 0x5acc225d);
assert_murmurhash3_x64_64(0x23ea58fd357bd3dcL, "p2XW49t0wF2Lngy", 0xdacc225d);

char test[] = "cf0a875f177d4977b7b9119d7e4ee4ec";
hex(test);
assert_murmurhash3_x64_64(7646390503503309584L, test, 88057744);
printf("Success!\n");
}

