// -*- c++ -*-
/***************************************************************************
                         creature.h  -  description
                             -------------------
    begin                : ven fv 22 22:27:00 CET 2002
    copyright            : (C) 2002 by Romain Vinot
    email                : vinot@aist.enst.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef _CREATURE_H_
#define _CREATURE_H_

#include <iostream>
#include <map>
#include <qstring.h>

#include "variables.h"
#include "object.h"

class Board;

class GUICreature;
class Creature;
class GUICommandLine;

class ExtensibleObject : public Object {
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  ExtensibleObject(GUICreature *guic, GUICommandLine *guicmd, int position);
  ExtensibleObject(const ExtensibleObject& copy);
  virtual ~ExtensibleObject(void);
  virtual Object* getCopy() const;

  virtual bool isAttackable(void);
  virtual bool isFireable(void);
  virtual bool isShootable(void);
  virtual int canBeTaken(void);
  virtual int canBeDropped(void);
  virtual int canBeTransfered(void);
  virtual bool shootHasKilledObject(void);

  virtual void hasBeenTaken(void);
  virtual void hasBeenFree(void);

  bool isEmpty(void);
  virtual int ThrowDice(bool reThrow=false, int value=0);

  void printInfo(int loc);
  QString getDesc(void);
  char getChar(void);
  void *getPixmap(bool big);

  void setDesc(QString &d);
  void setGraphic(QString &g);
  void setEmpty(bool isEmpty);
  void setAttackable(bool isAttackable);
  void setShootable(int isShootable);
  void setFireable(bool isFireable);
  void setTaken(int t);
  void setDropable(int r);
  void setTransferable(int t);
  void setShootWith(bool sw);
  void setUseDoorWith(bool udw);
private:
  QString desc, graphic;
  
  bool stateIsEmpty;
  bool stateIsAttackable;
  int stateIsShootable; // 0 means not shootable, more indicates nb of times 
                        // it needs to be shoot at before destroyed.
  bool stateIsFireable;

  int stateCanBeTaken; // 0 means not takeable, more indicates nb of AP needed
  int stateDropable; // number of APs needed to drop it.
  int stateTransferable; // number of APs to transfer it to other marine.
  bool stateShootWith; // Only needed if can be taken.
  bool stateUseDoorWith; // Only needed if can be taken.

  bool isTaken;
};


class EntryZone : public Object {
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  EntryZone(GUICreature *guic, GUICommandLine *guicmd, int position, 
	    Direction dir, int maxblip);
  EntryZone(const EntryZone& copy);
  virtual ~EntryZone(void);

  virtual Object* getCopy() const;

  void printInfo(int loc);
  char getChar(void);
  void *getPixmap(bool big);
  int getMaxBlip(void);
private:
  Direction graphicalDir;
  int maxblip;
};

class Door : public Object {
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  Door(GUICreature *guic, GUICommandLine *guicmd, 
       int position, bool vertical, DoorState s=CLOSE);
  Door(const Door& copy);
  virtual ~Door(void);

  virtual Object* getCopy() const;

  bool isEmpty(void);
  bool Open(void);
  bool Close(void);
  bool canBeOpened(void);
  bool canBeClosed(void);

  virtual bool isAttackable(void);
  virtual bool isFireable(void);
  virtual bool isShootable(void);
  virtual int ThrowDice(bool reThrow=false, int value=0);

  void printInfo(int loc);
  char getChar(void);
  void *getPixmap(bool big);
private:
  DoorState state;
  bool isVertical;
};

class Bulkhead : public Object {
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  Bulkhead(GUICreature *guic, GUICommandLine *guicmd, int position,
	   DoorState s=OPEN, bool vertical=true,
	   bool opened=false, bool closed=true);
  Bulkhead(const Bulkhead& copy);
  virtual ~Bulkhead(void);

  virtual Object* getCopy() const;

  bool isEmpty(void);
  bool Open(void);
  bool Close(void);
  bool canBeOpened(void);
  bool canBeClosed(void);
  void setCanBeOpened(bool op);
  void setCanBeClosed(bool cl);

  void printInfo(int loc);
  char getChar(void);
  void *getPixmap(bool big);
private:
  DoorState state;
  bool isVertical;
  bool stateCanBeOpened;
  bool stateCanBeClosed;
};


class Exit : public Object {
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  Exit(GUICreature *guic, GUICommandLine *guicmd, 
       int position, Turn passingTurn, Direction dir);
  Exit(const Exit& copy);
  virtual ~Exit(void);

  virtual Object* getCopy() const;

  bool isEmpty(void);

  void addCreatureOut(ObjectType ty, int id);

  void printInfo(int loc);
  char getChar(void);
  void *getPixmap(bool big);
private:
  Direction graphicalDir;
  map<ObjectType, list<int> > out;
  Turn passingTurn;
};

class Flame : public Object {
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  Flame(GUICreature *guic, GUICommandLine *guicmd, int position);
  Flame(const Flame& copy);
  virtual ~Flame(void);

  virtual Object* getCopy() const;
  bool isEmpty(void);
  void addBigFlame(void);
  bool reduceFlame(void);
  bool isFireable(void);

  void setTransparency(bool yes);
  
  void printInfo(int loc);
  char getChar(void);
  void *getPixmap(bool big);
private:
  int nbFlames;
  bool isTransparent;
};

class Creature : public Object
{
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  Creature(GUICreature *xcrea, GUICommandLine *guicmd, int position, int id);
  Creature(const Creature& copy);
  virtual ~Creature(void);

  //  void getInfo(QDomElement &unitlist)=0;
  int getId(void);

  virtual bool select(void)=0;
  virtual void deselect(void)=0;
  virtual void beginNewTurn (Turn turn);
  virtual void begin(Turn turn);
  virtual void setInitTurn(int turn);
  virtual int getInitTurn(void);
  
  bool MoveToCase(int pos);

  virtual Direction getOrientation(void);
  virtual void setOrientation(Direction d);
  virtual bool TurnOrientation(Direction d);
  virtual bool Move(FullDirection d);
  virtual bool Enter(void);
  bool GenestealerEnter(void);
  virtual bool Revert(void);
  virtual bool takeObject(void);
  virtual bool hasAnObject(void);
  virtual Object * transferObject(int newpos);

  virtual bool OpenDoor(int caseId=-1)=0;
  virtual bool CloseDoor(int caseId=-1)=0;

  virtual AttackRes Attack(void);
  virtual AttackRes Shoot(int caseId);
  virtual bool Overwatch(void);
  virtual bool OverwatchShoot(int caseId);
  virtual bool Unjam(void);
  virtual bool Fire(int c);
  virtual bool Reload(void);

  virtual bool canViewAt(int caseId);
  virtual bool canShootAt(int caseId);
  bool NoObstacleTo(int targetPos);
  virtual bool isMarinePlayer(void)=0;
  virtual int EndTurnActionPts();
  virtual int getActionPts(void);
  virtual void setActionPts(int ap);
protected:
  int id;
  int initTurn; // used for enter.
  bool selected;
  bool hasDoneSomething;
  int actionPts;
  int lastActionPts; // used for levelcond with APs.
  Turn currentTurn;
  Object* taken;
};

class NonOrientedCreature : public Creature {
  friend class XCreature;
  friend class QCreature;
public:
  NonOrientedCreature(GUICreature *gc, GUICommandLine *gcmd, int pos,int id);
  NonOrientedCreature(const NonOrientedCreature& copy);
  virtual ~NonOrientedCreature(void);

  virtual bool OpenDoor(int caseId=-1);
  virtual bool CloseDoor(int caseId=-1);
};

class ModuleCAT : public NonOrientedCreature {
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  ModuleCAT(GUICreature *guic, GUICommandLine *guicmd, int position, int id);
  ModuleCAT(const ModuleCAT& copy);
  virtual ~ModuleCAT(void);
  virtual Object* getCopy() const;

  //  void getInfo(QDomElement &unitlist);

  bool isMarinePlayer(void);
  bool isEmpty(void);
  bool isAttackable(void);
  bool isShootable(void);

  virtual bool Move(FullDirection d);
  virtual bool OpenDoor(int caseId=-1);
  virtual bool CloseDoor(int caseId=-1);

  virtual int canBeTaken(void);
  virtual int canBeDropped(void);
  virtual int canBeTransfered(void);
  virtual void hasBeenTaken(void);
  virtual void hasBeenFree(void);

  virtual bool select(void);
  virtual bool isSelectable(void);
  virtual void deselect(void);
  virtual void beginNewTurn (Turn turn);

  void printInfo(int loc);
  char getChar(void);
  void *getPixmap(bool big);
private:
  bool isTaken;
};

class Blip : public NonOrientedCreature
{
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  Blip(GUICreature *gcrea, GUICommandLine *guicmd, int position, int id, int nb);
  Blip(const Blip& copy);
  virtual ~Blip(void);

  virtual Object* getCopy() const;

  bool select(void);
  void deselect(void);
  virtual void beginNewTurn (Turn turn);

  virtual bool Move(FullDirection d);
  virtual bool Enter(void);
  virtual bool Revert(void);
  virtual int getActionPts(void);
  virtual bool shootHasKilledObject(void);

  virtual bool OpenDoor(int caseId=-1);
  virtual bool CloseDoor(int caseId=-1);
  virtual AttackRes Attack(void);

  void printInfo(int loc);
  char getChar(void);
  void *getPixmap(bool big);
  virtual bool isMarinePlayer(void);
  virtual bool isSelectable(void);
  int getNbGen(void);
protected:
  int nbGen;
};

class RealCreature : public Creature
{
  friend class XCreature;
  friend class QCreature;
public: 
  RealCreature(GUICreature *xcrea, GUICommandLine *guicmd,int position,int id,
	       Direction ori);
  RealCreature(const RealCreature& copy);
  virtual ~RealCreature(void);

  void getFrontalCases(set<int> *cases);
  Direction getOrientation(void);
  virtual void turnOrientation(Direction dir);
  virtual void setOrientation(Direction d);
  virtual bool canTurn(void);
  virtual RelativeDirection CardinalToRelativeDirection(FullDirection dir);

  virtual bool OpenDoor(int caseId=-1);
  virtual bool CloseDoor(int caseId=-1);

  pair<bool, Object*> PreliminaryAttack(void);
  virtual AttackRes Attack(void);
  virtual bool canAttack(Direction dir);

  virtual bool canViewAt(int caseId);
  virtual bool canShootAt(int caseId);
protected:
  Direction orientation;
  historyCmd history;
};


class Genestealer : public RealCreature
{
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  Genestealer(GUICreature *xcrea, GUICommandLine *guicmd, int position, int id, 
	      Direction ori, int actionpts=6);
  Genestealer(const Genestealer& copy);
  ~Genestealer(void);
  
  virtual Object* getCopy() const;

  bool select(void);
  void deselect(void);
  virtual void beginNewTurn (Turn turn);

  virtual bool TurnOrientation(Direction d);
  virtual bool Move(FullDirection dir);
  virtual bool Enter(void);

  virtual bool OpenDoor(int caseId=-1);
  virtual bool CloseDoor(int caseId=-1);

  virtual bool isFireable(void);
  virtual bool isAttackable(void);
  virtual bool isShootable(void);

  virtual int ThrowDice(bool reThrow=false, int value=0);
  virtual AttackRes Attack(void);

  void printInfo(int loc);
  char getChar(void);
  void *getPixmap(bool big);
  virtual bool isMarinePlayer(void);
  virtual bool isSelectable(void);
};

class Marine : public RealCreature
{
  friend class XCreature;
  friend class QCreature;
public:
  Marine(GUICreature *xcrea, GUICommandLine *guicmd, int position, int id, 
	 Direction ori, int *cmd);
  Marine(const Marine& copy);
  ~Marine(void);
  virtual bool isMarinePlayer(void);
  
  int EndTurnActionPts(void);

  bool select(void);
  void deselect(void);
  virtual void beginNewTurn (Turn turn);

  virtual bool TurnOrientation(Direction d);
  virtual bool Move(FullDirection dir);

  virtual bool OpenDoor(int caseId=-1);
  virtual bool CloseDoor(int caseId=-1);

  virtual bool takeObject(void);
  virtual Object * transferObject(int newpos);
  virtual void automaticTakeObject(Object *obj);
  virtual bool dropTakenObject(bool fromDead=false);

  virtual bool isSelectable(void);
  virtual bool isAttackable(void);
  virtual bool isShootable(void);
  virtual int ThrowDice(bool reThrow=false, int value=0);
  virtual AttackRes Attack(void);
  virtual void isAttacked(void);

 protected:
  void CheckCmdPts(void);

  int *cmdPts;
  int previousTargetCase;
};

class BolterMarine : public Marine {
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  BolterMarine(GUICreature *xcrea, GUICommandLine *guicmd, int position, int id, 
	       Direction ori, int *cmd);
  BolterMarine(const BolterMarine& copy);
  ~BolterMarine(void);

  virtual Object* getCopy() const;

  virtual AttackRes Shoot(int c);
  int ThrowShootDice(int c);
  bool ThrowJammedDice(void);
  virtual bool Overwatch(void);
  virtual bool OverwatchShoot(int caseId);
  virtual bool Unjam(void);
  virtual bool onOverwatch(void);
  virtual bool isJammed(void);
  
  virtual void printInfo(int loc);
  virtual char getChar(void);
  void *getPixmap(bool big);
protected:
  bool jammed;
};

class Sergeant : public BolterMarine {
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  Sergeant(GUICreature *xcrea, GUICommandLine *guicmd, int position, int id, 
	   Direction ori, int *cmd);
  Sergeant(const Sergeant& copy);
  ~Sergeant(void);

  virtual Object* getCopy() const;

  virtual int ThrowDice(bool reThrow=false, int value=0);

  virtual void printInfo(int loc);
  virtual char getChar(void);
  void *getPixmap(bool big);
};

class FlamerMarine : public Marine
{
  friend class XCreature;
  friend class QCreature;
  friend class DataStoring;
public:
  FlamerMarine(GUICreature *xcrea,GUICommandLine *guicmd, int position,int id, 
	       Direction ori, int *cmd, int charger);
  FlamerMarine(const FlamerMarine& copy);
  ~FlamerMarine(void);

  virtual Object* getCopy() const;

  virtual int getAllAmmunition(void);

  virtual bool Fire(int c);
  bool ThrowFireDice(void);
  virtual bool Reload(void);

  virtual void printInfo(int loc);
  virtual char getChar(void);
  void *getPixmap(bool big);
private:
  int ammunition;
  int charger;

  list<Flame *> currentFiring;
};

#endif
