/**
 * @file   player_list.h
 * @author Devin Smittle <pandagoat@gmail.com>
 * @date   Wed Jun  4 16:55:41 2008
 * 
 * @brief This contains a structure for the definition of a player, a
 * glib hash of them, a list of pending player connections, and
 * methods to manipulate the hash and list.
 * 
 * 
 */

/***************************************************************************
 *   Copyright (C) 2008 by Devin Smittle   *
 *   pandagoat@gmail.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef __PLAYER_LIST_H_
#define __PLAYER_LIST_H_ 1

#include <sys/types.h>
#include <netinet/in.h>
#include "packet_join_request.h"
#include <glib.h>

enum COLOR{RED=0, BLUE=1, TEAL=2, PURPLE=3, YELLOW=4, ORANGE=5, GREEN=6, PINK=7, GRAY=8, LIGHTBLUE=9, DARKGREEN=10, BROWN=11};
int massive_color_parser(char* str);


/* A structure for a player */
typedef struct
{
#define PLAYER_SIZE(p) (strlen((p)->name) + 3*sizeof(u_char) + sizeof(float) + sizeof(struct sockaddr_in) + sizeof(struct in_addr) + sizeof(u_short))
  int pid; 			/* size = 1 */
  float ping;			/* size = 4 */
  struct sockaddr_in ext_addr; 	/* size = 16 */ 

  /* Needed info to construct eth,tcp,and ip headers to send to them */
  u_char mac_addr[6]; 		/* Dest mac */
  u_char my_mac_addr[6];     	/* To save it */
  struct in_addr ip_to_me;
  u_short port_to_me; 		/* Src port */
  /* ip_info contains dest ip and port */

  u_char* name;
  u_char color;
}player;

typedef struct
{
#define GET_PING_MS(pp) (((pp)->end_echo - (pp)->start_echo)*1000)
  u_long id; 			/* sin_addr + sin_port */
  gdouble ping;
  gdouble start_echo;
  gdouble end_echo;
} player_ping;

typedef struct
{
  /* GHashTable* players; */
  GList* players;
  GList* pending;
  GList* pings;
  /* GHashTable* pending; */
  /* GHashTable* pings; */
} player_list;



player_list* new_player_list();

/* Access the pending list */
void players_add_pending(player_join_request_packet* pjrp, player_list* pl);
void players_remove_pending(struct sockaddr_in ext_ip, player_list* pl);
player* players_get_player_pending(struct sockaddr_in ext_ip, player_list* pl);


/* Access the player list */
void players_add(player* p, player_list* pl);
void players_remove(int pid, player_list* pl);
player* players_get(int pid, player_list* pl);
player* players_get_addr(struct in_addr ip, u_short port, player_list* pl);
player* players_get_color(enum COLOR color, player_list* pl);
player* players_get_ts(int pid, player_list* pl); /* thread safe */

typedef void (*players_loop_callback)(u_char pid, player* p, player_list* pl);

struct player_loop_thread_args
{
  players_loop_callback cb;
  player_list* pl;
};

void players_start_loop(players_loop_callback cb,  player_list* pl);


void players_clear_players(player_list* pl);
void players_clear_pending(player_list* pl);
void players_clear_pings(player_list* pl);


/* Access the ping list */
player_ping* players_add_ping(struct in_addr ip, u_short port, player_list* pl);
void players_remove_ping(struct in_addr ip, u_short port, player_list* pl);
player_ping* players_ping_get(struct in_addr ip, u_short port, player_list* pl);

player_list* player_list_shallow_copy(player_list* pl);

/* deprecated */
gchar* players_get_name_pending(const struct sockaddr_in* ipinfo, player_list* pl);

#endif /* __PLAYER_LIST_H_ */

