////////////////////////////////////////////////////////////////////////////////
/// widgets.js
///
/// DISCLAIMER
///
/// Copyright 2010 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
///////////////////////////////////////////////////////////////////////////////
//
//

// widgets: 
//
// paginator 
// logTable
// configEditor
// chartTypeEditor
// flotChart
// restForm
// keyList
// httpRequestList
// slidelist
// notifier
// extendedKeyVals

if(!window['jQuery']){ throw "-1 not enough jQuery."; } 

(function($){ 

// helpers -------------------------------------------------- 
// these are only used in this file and are not visible from outside 
// because of functional scope

function ucfirst(str) { 
  // returns str with the first character uppercased
  return str.charAt(0).toUpperCase() + str.slice(1);
}

function makeSelect(name, options, cond, scope){ 
  // this makes html for a select-field
  // Arguments: 
  //   name: the name attribute and classname of the field
  //   options: an array of options. 
  //            if the options are strings the value will be the same string
  //            in lowercase 
  //   cond: a function that get an option as argument and should return true 
  //         when this options should be selected.
  //   scope: optional. an objet that will be used as this for cond
  scope = scope || this;
  var sel = $('<select />', {
    className: name,
    name: name 
  }); 
  for(var i=0; i<options.length; i++){ 
    var val = options[i].toLowerCase ? options[i].toLowerCase() : options[i]; 
    $('<option />', {
      value: val,
      selected: cond.call(scope, options[i]), 
      text: options[i] 
    }).appendTo(sel); 
  } 
  return sel;
}

function ISODateString(d){
 function pad(n){return n<10 ? '0'+n : n}
 return d.getUTCFullYear()+'-'
      + pad(d.getUTCMonth()+1)+'-'
      + pad(d.getUTCDate())+'T'
      + pad(d.getUTCHours())+':'
      + pad(d.getUTCMinutes())+':'
      + pad(d.getUTCSeconds()) //+'Z';
}

function lbl(label, elm){
  var lbl = $('<label />'); 
  $('<span />', { text: label + ':' }).appendTo(lbl);
  lbl.append(elm); 
  return lbl;
} 


// widgets ---------------------------------------------------------
/* 
  TODO: 

  // some info in jquery ui-widgets: 
  http://blog.petersendidit.com/post/jquery-ui-1-8-widget-factory/
  http://nemikor.com/presentations/jQuery-UI-Widget-Factory.pdf
  http://bililite.com/blog/understanding-jquery-ui-widgets-a-tutorial/
  http://westhoffswelt.de/data/portfolio/webtechcon2010_jquery_ui_widget_development.pdf
  http://ajpiano.com/widgetfactory/#slide1
  http://www.erichynds.com/jquery/tips-for-developing-jquery-ui-widgets/

*/

$.widget('ui.paginator', {
  // a widget that displays a pagination ui 
  // this assumes that there is a certain amount of items and 
  // each page contains a number of items.  
  // the number of pages is calculated automatically. 
  // << < n/m > >>  
  options: {
    size: 10, 
    offset: 0, 
  }, 
  _create: function(){ 
    this.totalAmount = 0;
    this.numPages = 0;
    
    $(this.getHTML()).appendTo(this.element); 
    this.element.addClass('paginator'); 

    // connects event listeners for the pagination-controll
    this.element.find('.first').bind('click.paginator', function(e){
      $(this).parents('.paginator').paginator('first');
    }); 

    this.element.find('.prev').bind('click.paginator', function(e){
      $(this).parents('.paginator').paginator('prev');
    }); 

    this.element.find('.next').bind('click.paginator', function(e){
      $(this).parents('.paginator').paginator('next');
    }); 

    this.element.find('.last').bind('click.paginator', function(e){
      $(this).parents('.paginator').paginator('last');
    }); 
  },  
  setTotalAmount: function(ta){
    this.totalAmount = parseInt(ta, 10);
    this.updateUI(); 
  }, 
  setSize: function(si){
    this.options.size = parseInt(si, 10);
    this.updateUI(); 
  }, 
  getParams: function(){ 
    return {
      offset: this.options.offset, 
      size: this.options.size,
    }; 
  }, 
  prev: function(){ 
     // jumps to the previous page
     var nextOffset = parseInt(this.options.offset,10) - parseInt(this.options.size);
    if(nextOffset > 0){ 
      this.options.offset = nextOffset; 
    } else { 
      this.options.offset = 0;
    } 
    this._change();
    this.updateUI(); 
  }, 
  next: function(){ 
     // jumps to the following page
    var nextOffset = parseInt(this.options.offset,10) + parseInt(this.options.size, 10);
    if(this.totalAmount && this.totalAmount > nextOffset){ 
      this.options.offset = nextOffset; 
      // emit page changed event
      this._change();
      this.updateUI(); 
    }    
  }, 
  first: function(){
    this.options.offset = 0;
    this._change();
    this.updateUI(); 
  }, 
  last: function(){
    this.options.offset = this.totalAmount - this.options.size;
    this._change();
    this.updateUI(); 
  }, 
  _change: function(){ 
    this._trigger('change', null, { 
      offset: this.options.offset, 
      size: this.options.size 
    }); 
  },  
  updateUI: function(){ 
    // dis- or en-ables buttons of the pager element and
    // updates the page n of m thingy. 
    //
    // last page? 
    if(this.options.offset >= this.totalAmount - this.options.size){
      this.element.find('.next').addClass('inactive');
      this.element.find('.last').addClass('inactive');
    } else { 
      this.element.find('.next').removeClass('inactive');
      this.element.find('.last').removeClass('inactive');
    } 
    
    // first page? 
    if(this.options.offset < this.options.size){ 
      this.element.find('.first').addClass('inactive');
      this.element.find('.prev').addClass('inactive');
    } else { 
      this.element.find('.first').removeClass('inactive');
      this.element.find('.prev').removeClass('inactive');
    } 

    // update position-indicator 
    this.element.find('.offset').html(parseInt(this.options.offset/this.options.size, 10)+1);
    this.element.find('.pagecount').html(parseInt(this.totalAmount/this.options.size, 10));
  },
  getHTML: function(){ 
    var html = '<div class="paginator">'; 
    html += '<a class="first" href="#"><img src="images/first.png" alt="First Page"></a>'; 
    html += '<a class="prev" href="#"><img src="images/prev.png" alt="Previous Page"></a>'; 
    html += '<div class="pagenofm"><span class="offset"></span>/<span class="pagecount"></span></div>';
    html += '<a class="next" href="#"><img src="images/next.png" alt="Next Page"></a>'; 
    html += '<a class="last" href="#"><img src="images/last.png" alt="Last Page"></a>'; 
    html += '</div>';
    return html; 
  } 
}); 


// logTable 

$.widget('ui.logTable', { 
  // displays log messages in a paginated table. 
  // uses the paginator widget for the ui
  options: { 
    url: '', // urls is mandatory
    dateFormat: '%A, %d %B, %H:%M:%S', 
    levels: ['Fatal', 'Error', 'Warning', 'Info', 'Debug', 'Trace'], 
    upto: 'trace',
    size: 10, 
    fields: ['ID', 'Level', 'Time', 'Text'], 
    search: '',
    pageSizes: [5,10,20,40]
  }, 
  _create: function(){
    // Builds the element(s) for the widgets, add them to the DOM 
    // and connects event-listeners.  also requests initial data. 
    this.element.addClass('logTable'); 
    if(!this.options.url){ 
      throw "logTable needs to be passed the URL to the logs."; 
    } 

    $('<table><thead><tr><th>'+this.options.fields.join('</th><th>')+'</th></tr></thead><tbody></tbody></table>').appendTo(this.element[0]); 
    this.controlsBar();
    this.loadData(); 
  }, 
  controlsBar: function(){ 
    // adds a toolbar for controls 
    // adds the html for the different elements and connects events
    // TODO:  consider moving this out of the logtable-widget?
    var barhtml = '<div class="controlsbar">';
    barhtml += '<div class="cb_pager"></div>'; 
    barhtml += '<div class="cb_options"></div>'; 
    barhtml += '<div class="cb_search"></div>'; 
    barhtml += '</div><div class="clearboth"></div>';
    $(barhtml).prependTo(this.element[0]);

    this.searchBoxElement().prependTo(this.element.find('.cb_search')); 
    // enable searching with enter-key
    this.element.find('.searchBox').keyup(function(e){ 
      if(e.keyCode === 13){ // enter key
        var elm  = $(this).parents('.logTable');
        var searchterm = elm.find('.searchBox').val();
        elm.logTable('option', 'search', searchterm); 
        elm.logTable('loadData'); 
        e.stopPropagation();
        return 0;
      } 
    });


    this.element.find('.searchButton').bind('click.logTable',function(){
      var elm  = $(this).parents('.logTable');
      var searchterm = elm.find('.searchBox').val();
      elm.logTable('option', 'search', searchterm); 
      elm.logTable('loadData'); 
    });
    makeSelect('pageSize', this.options.pageSizes, function(opt){ 
      return opt == this.options.size;
    },this).prependTo(this.element.find('.cb_options')).change(function(){
      // jquery widgets have the worst outside interface ever.
      var elm  = $(this).parents('.logTable');
      elm.find('.paginator').paginator('setSize', $(this).val()); 
      elm.logTable('loadData'); 
    });
    makeSelect('levelSelector', this.options.levels, function(opt){ 
      return opt.toLowerCase() == this.options.upto;
    },this).prependTo(this.element.find('.cb_options')).change(function(){
      var elm  = $(this).parents('.logTable');
      elm.logTable('option', 'upto', $(this).val()); 
      elm.logTable('loadData'); 
    }); 
    this.element.find('.cb_pager').paginator({
      size: this.options.size,
      change: function(){ 
        $(this).parents('.logTable').logTable('loadData');
      } 
    }); 
  },   
/*
  tail: function(){ // not in use
    // show the tail view
    this.element.find('tbody').empty(); 
    if(this.totalAmount > 0){
      var start = this.totalAmount - this.options.size; 
      this.updateTail(start);
    } else {
      this.requestData({ upto: this.options.upto  }, function(data){ 
        this.totalAmount = data.totalAmount; 
        var start = this.totalAmount - this.options.size; 
        this.updateTail(start);
      }); 
    }   
  }, 
  updateTail: function(start){  // not in use
    // update the tail
    start = start || this.lastLid || 0; 
    this.requestData({ start:start, upto: this.options.upto  }, function(data){ 
      var len = data.text.length; 
      var tbody = this.element.find('tbody');
      for(var i=0; i<len; i++){
        this.lastLid = data.lid[i];
        $(this.renderRow({ 
          lid: data.lid[i], 
          level: this.options.levels[data.level[i]], 
          timestamp: data.timestamp[i],
          text: data.text[i],  
          //hidden: true
        })).appendTo(tbody)//.fadeIn('slow'); 
        // tbody.scrollTop = tbody.scrollHeight; 
      } 
    }); 
    var that = this;
    setTimeout(function(){ that.updateTail(); }, 1500); 
  }, 
*/
  formatText: function(text){ 
    // takes a body of text and alters it for html output.
    text = text.replace(/\n/g, '<br />'); 
    // insert soft hyphens every 20 non-whitespace characters 
    text = text.replace(new RegExp("(\\w{20})(?=\\w)", "g"), "$1$shy;");
    return text;
  }, 
  renderRow: function(data){ 
    // builds html for a single table row and returns it
    // expects an object with the members: lid, level, timestamp, text 
    var dateObj = new Date(data.timestamp * 1000); 
    var datestr = '';
    if(dateObj.toLocaleFormat){
      datestr = dateObj.toLocaleFormat(this.options.dateFormat); 
    } else { 
      datestr = dateObj.toLocaleDateString() + ' ' + dateObj.toLocaleTimeString();
    }  
    var rowdata = [data.lid, data.level, datestr, this.formatText(data.text)];  
    return '<tr class="level'+data.level+(data.hidden ? ' hidden' : '')+'"><td>'+rowdata.join('</td><td>')+'</td></tr>'; 
  }, 
  requestData: function(params, cb){ 
    // get log-items from the server. 
    $.ajax({ 
      url: this.options.url,
      data: params, 
      context: this, 
      dataType: 'json', 
      success: function(){ cb.apply(this, arguments); }, 
      error: function(xhr){ 
        var err = { message: xhr.responseText };
        var errTxt = 'Error on fetching log-data. ' + err.message;
        $('.notifier').notifier('notify', 'error', errTxt);

        // in case of error, remove the fragments of the failed 
        // table.
        this.element.find('table').remove(); 
        this.element.find('.controlsbar').remove(); 

      } 
    }); 
  }, 
  loadData: function(cb){ 
    // gets log-items according to the settings of the pager
    // tries to call method this[cb] afterwards
    var params = {
      upto: this.options.upto, 
      search: this.options.search
    }; 
    $.extend(params, this.element.find('.paginator').paginator('getParams'));
    this.requestData(params, function(data){
      var len = data.text.length; 
      this.totalAmount = data.totalAmount; 
      this.element.find('tbody').empty(); 
      var html = ''; 
      for(var i=0; i<len; i++){
        html += this.renderRow({ 
          lid: data.lid[i], 
          level: this.options.levels[data.level[i]], 
          timestamp: data.timestamp[i],
          text: data.text[i] 
        }); 
      } 
      $(html).appendTo(this.element.find('tbody')); 
      if(cb){ this[cb](); } 
      this.element.find('.paginator').paginator('setTotalAmount', this.totalAmount);

    });   
  }, 
  searchBoxElement: function(){ 
    var html = '<input type="text" value="'+this.options.search+'" name="search" class="searchBox" /><input type="button" value="go" name="searchButton" class="searchButton" />'; 
    return $(html);
  }
}); 

/* ------------------------------------------------------------------ */

// configEditor
$.widget('ui.configEditor', { 
  options: {
    url: '',
    defaultfields: [ 
      // the array of field objects defines the form for the configuration 
      //
      // NOTE: the attribute 'datatype' is mainly used to display right of 
      // input in the form, but also abused to determine if the value has to be
      // PUT back to the server as a float or not. this might break in future
      // see method makeConfigObjectString 
      {name: 'logLevel', label: 'Log Level', type: 'select',
       options: ['fatal', 'error', 'info', 'warning', 'debug', 'trace']},
      {name: 'limitKeys', label: 'Max. Key Number', type: 'text' }, 
      {name: 'limitKeySize', label: 'Max. Single Key Size', type: 'text', datatype: 'Byte' }, 
      {name: 'limitTotalKeySize', label: 'Max. Total Key Size', type: 'text', datatype: 'Byte' }, 
      {name: 'limitDataSize', label: 'Max. Data Size per Key', type: 'text', datatype: 'Byte' }, 
      {name: 'limitTotalDataSize', label: 'Max. Total Data Size', type: 'text', datatype: 'Byte' }, 
      {name: 'limitTotalSize', label: 'Max. Total Storage Size', type: 'text', datatype: 'Byte' }
    ] 
  }, 
  _create: function(){      
    this.options.fields = this.options.fields || this.options.defaultfields;
    if(!this.options.url){ 
      throw "configEditor needs an URL.";
    } 
    this.element.addClass('configEditor'); 
    this.getData('insertForm'); // fetch data and create the form afterwards
  },  
  getData: function(cb){ 
    // retrieves data via RESTful url. stores it in this.values and 
    // calls the method of this called cb
    $.ajax({ 
      url: this.options.url,
      context: this, 
      dataType: 'json', 
      success: function(data){
        this.values = data; 
        if(cb){ this[cb](); } 
      } 
    }); 
  }, 
  formToJson: function(){ 
    // returns a json object with the values from the form
    // numbers are parsed to floats
    var formObjects = this.element.find('form[name="configForm"]').serializeArray();
    var json = {};
    for(var i=0; i<formObjects.length; i++){
      // all formfield values come as strings. 
      // convert the ones that seem to be numbers to floats. 
      // FIXME: this will fail for strings starting with numbers.
      json[formObjects[i].name] = isNaN(parseInt(formObjects[i].value,10)) ? formObjects[i].value : parseFloat(formObjects[i].value,10);
    } 
    return json;
  }, 
  makeConfigObjectString: function(data){
    //  (TODO: maybe lib this out?) 
    // return a string that can be PUT back to the server 
    //
    // if data is not provided it will be taken from this.formToJson()
    //
    // the json required by the server looks like this: 
    //  keys are quoted, string-values are quoted, float values always
    //  need a decimal point. 
    //  example: 
    //  '{"logLevel" : "trace","limitKeys" : 12,"limitKeySize" : 0.0 [...] }'; 
    data = data || this.formToJson();
    var str = '';
    var pairs = [];
    for(var i=0; i<this.options.fields.length; i++){
      var field = this.options.fields[i];
      str = '"'+field.name+'" : '; 
      if( typeof data[field.name] === 'string' ){ 
        str += '"'+data[field.name]+'"';
      } else { 
        var val = String(data[field.name]); 
        if( field.datatype && field.datatype === 'Byte'){ 
          // Attention shitty conditional:
          // for now field with the datatype 'Byte' is always a float. 
          // this correlation might not always be true. 
          if(val.indexOf('.') === -1){
            // floats without a digits right of the decimal-point
            // don't get a decimal point when casted to string.
            val += '.0';
          } 
        } 
        str += val; 
      }  
      pairs.push(str);
    } 
    return '{'+pairs.join(',')+'}';
  },
  validate: function(data){ 
    data = data || this.formToJson();
    var fail = false; 
    for(var i=0; i<this.options.fields.length; i++){
      var field = this.options.fields[i]; 
      if(field.datatype === 'Byte'){ 
        if(isNaN(parseInt(data[field.name]))){ 
          var errTxt = 'Please enter a number (or 0) for '+field.label; 
          $('.notifier').notifier('notify', 'error', errTxt);
          fail = true;
        } 
      } else { 
        if(!field.optional){ 
          if(data[field.name]===''){ 
            var errTxt = 'Please enter a value for '+field.label; 
            $('.notifier').notifier('notify', 'error', errTxt);
            fail = true;
          } 
        } 
      } 
    } 
    return !fail;
  }, 
  putData: function(cb){ 
    // PUTs the data back to the url where it came from
    var data = this.formToJson(); 
    if(!this.validate(data)){ 
      return 0;
    } 
    var confobjstr = this.makeConfigObjectString(data);
    $.ajax({ 
      url: this.options.url,
      data: confobjstr,
      type: 'PUT',
      context: this, 
      dataType: 'json', 
      success: function(data){
        this.values = data;
        if(cb && this[cb]){ this[cb](); } 
        $('.notifier').notifier('notify', 'info', 'Saved changes to config.');
      }, 
      error: function(err){
        var errTxt = 'An Error happened while writing config data '; 
        $('.notifier').notifier('notify', 'error', errTxt);
        console.error.apply(console, arguments);
      } 
    }); 
  }, 
  updateForm: function(){ 
    this.element.find('.confed_refreshbutton').css('opacity', 0.5); 
    this.getData('_updateForm'); 
  }, 
  _updateForm: function(){ 
    var name; 
    for(var i=0; i<this.options.fields.length; i++){ 
      name = this.options.fields[i].name;
      $('input[name='+name+']').val( this.values[name] ); 
    } 
    this.element.find('.confed_refreshbutton').css('opacity', 1); 
  }, 
  insertForm: function(){
    // puts the form in the DOM and connects event-handlers
    $('<a href="#" class="confed_refreshbutton"><img src="images/refresh.png" alt="refresh"/></a>').appendTo(this.element[0]);
    this.element.find('.confed_refreshbutton').bind('click.configEditor', function(){ 
      $(this).parents('.configEditor').configEditor('updateForm'); 
    }); 

    this.createForm().appendTo(this.element[0]);
    this.element.find('.confed_submit').bind('click.configEditor', function(){
      $(this).parents('.configEditor').configEditor('putData'); 
    }); 
  }, 
  createForm: function(){ 
    // returns a form-element 
    // fields are defined by this.options.fields
    var form = $('<form />', {name:'configForm' });
    for(var i=0; i<this.options.fields.length; i++){ 
      form.append(this._createField(
        this.options.fields[i],
        this.values[ this.options.fields[i].name ]));
    } 
    var btns = $('<div />',{className: 'button'}).appendTo(form);
    $('<input />', {
      type: 'button', 
      className:'confed_submit', 
      value: 'Submit'}).appendTo(btns); 
    return form;
     
  }, 
  _createField: function(field, value){ 
    var lbl = $('<label />'); 
    $('<span />', {className:'label', text:field.label}).appendTo(lbl); 
    var fc=$('<div />',{className:'input'}).appendTo(lbl); 
    if($.isFunction(this['_create' + ucfirst(field.type) + 'Field'])){
      fc.append(this['_create'+ucfirst(field.type)+'Field'](field,value)); 
    } else { 
      fc.append(this._createTxtField(field,value));
      if(field.datatype){ 
        $('<span />', {text:' '+field.datatype}).appendTo(fc); 
      }
    } 
    return lbl;
  }, 
  _createTxtField: function(field, value){ 
    return $('<input />', {type:"text", value:value, name:field.name});
  },
  _createSelectField: function(field, value){ 
    // special method to create fields with type == 'select' 
    // fields are expected to have an array 'options' to populate the 
    // select-options.  
    // option values and labels are the same. labels are capitalized.
    var sel = $('<select />', {name: field.name}); 
    $.each(field.options, function(i,opt){
      $('<option />', { 
        value: opt, 
        selected: (opt===value), 
        text: ucfirst(opt) 
      }).appendTo(sel); 
    });  
    return sel; 
  }
}); 

// ---------------------------------------------------------------------


// note: the chartTypes are now in controllers.js. 
// not sure if they should stay there or not. 
var __chartTypes = [
  { 
    name: 'rest', 
    label: 'RESTful Requests', 
    available: ["rest", "restValueDelete", "restValuePut", "restValuePost", "restValueGet", "restInvalid", "restFlush", "restPrefix", "restVersion", "restStats"], 
    'default':  ['rest','restValuePost', 'restValuePut', 'restValueGet']
  }, 
  {
    name: 'memcache', 
    label: 'Memcache Requests',
    available: ["memcache", "memcacheInvalid", "memcacheAdd", "memcacheAppend", "memcacheCas", "memcacheDecr", "memcacheDelete", "memcacheFlushAll", "memcacheGet", "memcacheGets", "memcacheIncr", "memcachePGet", "memcachePGets", "memcachePrepend", "memcacheReplace", "memcacheSet", "memcacheStat", "memcacheVersion"], 
    'default': ['memcacheInvalid', 'memcacheAdd', 'memcacheStat']
  }, 
  { 
    name: 'postfix', 
    label: 'Postfix Requests',
    available: ["postfix", "postfixInvalid", "postfixGet", "postfixPut"],
    'default': ['postfixGet', 'postfixPut']
  },
  { 
    name: 'connection', 
    label: 'Number of Connections',
    available: ["connections", "httpConnections", "lineConnections"],
    'default':  ["httpConnections", "lineConnections"],
    type: 'connection', 
    _getValueFromDatasource: function(valuename, datasource){
      // this function works for the listView 
      // but breaks the chart view (kind-of)
      var period = 'minute'; 
      var start = datasource['runtime'][period].start;
      var items = datasource['connection'][period][valuename]['count']; 
      return [start[items.length-1], items[items.length-1]]; 
    } 
  }, 
  { 
    name: 'memory', 
    label: 'Memory Usage',
    available: 'numberKeys,limitKeys,totalKeySize,limitKeySize,limitTotalKeySize,totalDataSize,limitDataSize,limitTotalDataSize,limitTotalSize'.split(','), 
    'default':  ['numberKeys', 'totalKeySize'],
    getValueFromDatasource: function(valuename,datasource){ 
      var val = datasource.memory[valuename]; 
      return [ new Date().getTime(), val ];
    }, 
    maxValuesOnXAxis : 10 // never used again?
  }
]; 

// mapping of value-names to labels. 
// use labels.get(valname) to recieve label. 
// if no label is found for a string the same string will be capitalized
var labels = { "memcache": 'All',
               "memcacheInvalid": 'Invalid',
               "memcacheAdd": 'Add', 
               "memcacheAppend": 'Append', 
               "memcacheCas": "Cas", 
               "memcacheDecr": 'Decr', 
               "memcacheDelete": 'Delete', 
               "memcacheFlushAll": 'Flush All',
               "memcacheGet": 'Get',
               "memcacheGets": 'Gets',
               "memcacheIncr": 'Incr',
               "memcachePGet": 'PGet',
               "memcachePGets": 'PGets',
               "memcachePrepend": 'Prepend',
               "memcacheReplace": 'Replace',
               "memcacheSet": 'Set',
               "memcacheStat": 'Stat',
               "memcacheVersion": 'Version', 

               "rest": 'All',
               "restValueDelete": 'Delete',
               "restValuePut": 'Put', 
               "restValuePost": 'Post',
               "restValueGet": 'Get',
               "restInvalid": 'Invalid',
               "restFlush": 'Flush',
               "restPrefix": 'Prefix',
               "restVersion": 'Version',
               "restStats": 'Stats', 

               "connections": 'All Connections', 
               "httpConnections": 'HTTP Connections',
               "lineConnections": 'Line Connections',

               "postfix": 'All', 
               "postfixInvalid": 'Invalid',
               "postfixGet": 'Get', 
               "postfixPut": 'Put' }; 

labels.get = function(key){ 
  // returns label for a value-name. 
  // returns capitalized parameter if label not found.
  if(!key){ 
    var txt = 'No key passed to labels.get'; 
    $('.notifier').notifier('notify', 'error', txt);
    return ''; 
  } 

  if(key in this){ 
    return this[key];
  } 
  return ucfirst(key);
}   

/* --------------------------------------------------------- */

// widget to change the options on a chartType 
// used by flotChart-widget
$.widget('ui.chartTypeEditor', { 
  options: {
    period: 'minute',
    periodOptions: ['second', 'minute', 'hour', 'day'], 
    //chartTypes: chartTypes,
    // number of selectable, can be null or 0 to have a unlim. vals
    maxValues: 6, 
    removeButton: true
  }, 
  _create: function(){ 
    if(!this.options.chartTypes){ 
      throw "Chart types array must be passed to chartTypeEditor as chartTypes";
    } 
    this.element.addClass('chartTypesEditor'); 
    this.options.values = this.options.values || this.options.chartType.values;   
    this.showOptions();
    // add a remove-button. 
    if(this.options.removeButton){ 
      this.element.append($('<input />', { 
        type:'button',
        value: 'Remove Chart',
        click: function(e){ 
          $(this).parents('li').remove();
        } 
      })); 
    } 
  }, 
  showOptions: function(chartType){ 
    // creates html, puts it the dom and binds events. 
    // 
    // TODO: should maybe be changed to "new-style" DOM-node creation.
    chartType = chartType || this.options.chartType;
    // TODO doc
    // notInDatasource means the Data for this chart is not returned by
    // voc-server directly
    if(chartType.notInDatasource){ return 0; } 
    var types = $.map(this.options.chartTypes, function(ct){ return ct.name; }); 
    var html = '<form class="optionForm">';
    html += '<div class="avails"></div>';
    html += '<input type="button" class="applyButton" value="OK"></form>'; 
    
    this.element.html(html); 
    var form = this.element.find('.optionForm'); 

    var that = this;
    // selectbox for chartype
    makeSelect('chartType', types, function(ct){ 
      return ct === chartType.name; 
    }, this).prependTo(form);

    // event for charttype change updates selectable values.
    form.find('.chartType').change(function(){
      var ctname = $(this).val(); 
      for(var i=0;i<that.options.chartTypes.length; i++){
        if(ctname === that.options.chartTypes[i].name){
          ct = that.options.chartTypes[i]; 
        } 
      } 
      form.find('.avails').html(that.valuesMenu(ct.available, ct.values)); 
    }); 

    makeSelect('period', this.options.periodOptions, function(per){ 
      return per === this.options.period;
    }, this).prependTo(form); 
  
    // checkboxes for values
    form.find('.avails').html( 
      this.valuesMenu(chartType.available, this.options.values)
    ); 

    form.find('.applyButton').bind('click.chartTypeEditor',function(){
      that._trigger('change', null, [that.getPeriod(), that.getChartType(), that.getValues()]); 
    });

    // if there is a max for the number of selectable values set 
    // check now and bind re-check on selection 
    if(this.options.maxValues){ 
      this.checkNumberOfCheckedBoxes();
      // TODO: this is not bound namespacy. see if that's possibru
      this.element.find('.valbox').live('click', function(e){ 
        that.checkNumberOfCheckedBoxes();
      });
    } 
  }, 
  checkNumberOfCheckedBoxes: function(){
    // checks if the number of selected values exceeds the allowed max
    // (this.options.maxValues) and disables the other checkboxes if neccessary
    if(this.element.find('.valbox:checked').length >= this.options.maxValues){
      this.element.find('.valbox').not(':checked').attr('disabled','disabled');
    } else { 
      this.element.find('.valbox').removeAttr('disabled');
    }
  }, 
  getPeriod: function(){ 
    // returns a string  second, minute, etc
    return this.element.find('.period').val();
  }, 
  getChartType: function(){
    // returns the chart-type-object for the selected charttype
    var ctname = this.element.find('.chartType').val(); 
    for(var i=0;i<this.options.chartTypes.length; i++){
      if(ctname === this.options.chartTypes[i].name){
        return this.options.chartTypes[i]; 
      } 
    } 
    return null;
  }, 
  getValues: function(){
    // returns an array of strings of the selected values 
    var vals = [];
    this.element.find('input:checked').each(function(i){
      vals.push($(this).val());
    });
    return vals;
  }, 
  valuesMenu: function(avails, selecteds){
    // returns html string with checkboxes to select the values 
    // that should be displayed by the chart.
    // the checkboxes are displayed in collumns, the number of boxes
    // in a col depends on the total number of items.
    var html = '', avail = '', colLen, colwidth; 
    if(avails.length <= 5){ 
      colLen = 2;
    } else if(avails.length <= 9){ 
      colLen = 3;
    } else if(avails.length <= 18){ 
      colLen = 6;
    } else { 
      colLen = 8;
    }  
    colwidth = 95/(Math.ceil(avails.length/colLen)); 
    html += '<div class="valcol" style="width:'+colwidth+'%">'; 
    for(var i=0; i<avails.length; i++){ 
      avail = avails[i]; 
      if(i % colLen === 0){ // start new col if indicated. 
        html += '</div><div class="valcol" style="width:'+colwidth+'%">';
      } 
      html += '<label><input class="valbox" type="checkbox" value="'+avail+'" name="'+avail+'"';
      // see if this box has to be checked
      for(var ii=0; ii<selecteds.length; ii++){
        if(avail === selecteds[ii]){
          html += ' checked '; 
        }  
      } 
      html += '>'+labels.get(avail)+'</label>';
    } 
    html += '</div>';
    html += '<div class="clearboth"></div>';
    return html; 
  }, 
});
 
/* --------------------------------------------------------- */

function getDatasetsFromDataSource(data, period, values, type){ 
  // extracts the raw data for one chart from the big datastructure 
  // returned by /admin/statistics/overview
  // returns an array of arrays of two-element-arrays  
  type = type || 'runtime'; 
  var sets = []; 
  dglob = data;
  var start = data['runtime'][period].start;
  for(var iV=0; iV<values.length; iV++){
    var valueName = values[iV];
    var items = data[type][period][valueName]['count']; 
    var set = []; 
    for(var i=0; i<items.length; i++){ 
      if(start[i] > 0){  // sometimes there are timestamps 0
        set.push([ start[i] * 1000, items[i]]); 
      } 
    } 
    sets.push(set); 
  }  
  return sets;
};

/* --------------------------------------------------------- */

// rest, memchached, postfix, num connections
//
$.widget('ui.flotChart', { 
  options: {
    period: 'minute',
    periodOptions: ['second', 'minute', 'hour', 'day'],
    dateformat: '%H:%M:%S', 
    numberValues: 60,
    numberTicks: 4,
    seriesColors: ['#dabf28','#d66d1c', '#d83517', '#8b118d', '#b4b2ac' ], //.reverse(), 
    width: 358,
    height: 220,
    removable: true,
    showLines: true, 
    showBars: false,
    lineSteps: null, 
    // when creating datasets these defaults will be mixed in to 
    // reduce redundancy. 
    datasetDefaults: {
      stack: null,
      label: 'other',
      lines: { 
        show: true, 
        lineWidth: 2, 
        fill: 0.5
      } ,
      bars: { 
        show: false, 
        barWidth: 0.8, 
        fill: 0.5
      } 
    },
    linefillFun: function(num, count){ 
      return 0.8 - ((count - num)*0.1); 
    } 
  },
  _create: function(){ 
    this.element.addClass('flotChart'); 
    if(!this.options.chartType){ 
      var errTxt = 'No ChartType set!'; 
      $('.notifier').notifier('notify', 'error', errTxt);
      return 0; 
    } 

    this.applyType(this.options.chartType); 

    // override defaults with chartType 
    // function to get info from data-object
    this.getDatasetsFromDataSource = this.options.getDatasetsFromDataSource || getDatasetsFromDataSource; 


    // TODO: understand, clear, comment, refactor this shit:
    if(this.options.datasource){ 
      if(this.options.getValueFromDatasource){
        this.options.rawDatasets = []; 
        for(var i=0; i<this.options.values.length; i++){
          this.options.rawDatasets.push(
            [this.options.getValueFromDatasource(this.options.values[i],this.options.datasource)]
          ); 
        } 
      } else { 
        this.options.rawDatasets = this.getDatasetsFromDataSource(this.options.datasource, this.options.period, this.options.values, this.options.type); 
      } 
    } 
    $(this.getHTML()).appendTo(this.element); 
    this.element.find('.headline').html(this.options.label);
    this.element.find('.chartContainer, .listViewContainer, .optionsContainer').css({ 
      'width': this.options.width + 'px', 
      'height': this.options.height + 'px'
    });
    this.initChart(this.options.rawDatasets);

    this.element.find('.listViewButton').click(function(){
      $(this).parents('.flotChart').flotChart('showListView');
    });

    this.element.find('.editOptionsButton').click(function(){
      $(this).parents('.flotChart').flotChart('showOptions');
   });

    this.element.find('.chartButton').click(function(){
      $(this).parents('.flotChart').flotChart('showChart');
    });

    if(this.options.removable){ 
      this.element.find('.removeButton').click(function(){
         // doing
        //$(this).parents('.flotChart').flotChart('showChart');
        $(this).parents('li').remove();
      });
    } 
    this.showChart();
  }, 
  applyType: function(chartType){ 
    chartType = chartType || this.options.chartType; 
    // merge chartType into options.  
    //

    // the default-values should be overridable by constructor-options
    // of the widget. 
    if('values' in this.options){ 
      // is this ever reached? 
      delete chartType.values; 
    } 

    // chart type can contain a function to initialize itself.
    if(chartType.initType){ chartType.initType(this); } 

    $.extend(this.options, chartType); 
  }, 
  showListView: function(){
    // display the tabular view of the latest dataset. 
    this.element.find('.container>div').hide(); 
    this.element.flotChart('makeListView');
    this.element.find('.listViewContainer').show(); 
    this.element.find('.headline').html(this.options.label);
  }, 
  showChart: function(){ 
    // display the chart of data over time
    this.element.find('.container>div').hide(); 
    this.element.find('.chartContainer').show(); 
    this.element.flotChart('updateFromDatasource');
    this.element.find('.headline').html(this.options.label); //+ ' &40;'+this.options.period.charAt(0)+'&41;');
  }, 
  showOptions: function(){ 
    // display the options menu.
    // this is an instance of chartTypesEditor 
    this.element.find('.container>div').hide(); 
    this.element.flotChart('makeOptionsMenu');
    this.element.find('.optionsContainer').show(); 
    this.element.find('.headline').html('Options');
  }, 
  updateFromDatasource: function(datasource){
    // takes the response of /admin/statistics/overview
    // and tries to fiddle the data for this chart (defined by
    // this.options.chartType) out. 
    if(this.options.notInDatasource){ return 0; } 

    datasource = datasource || this.options.datasource;  
    this.options.datasource = datasource;  

    if(this.options.getValueFromDatasource){
      for(var i=0; i<this.options.values.length; i++){
        this.options.rawDatasets[i] = this.options.rawDatasets[i] || [];
        this.options.rawDatasets[i].push( this.options.getValueFromDatasource(this.options.values[i], datasource) ); 
        if(this.options.rawDatasets[i].length > 15){
          this.options.rawDatasets[i] = this.options.rawDatasets[i].slice(this.options.rawDatasets[i].length-15);
        } 
      } 
      this.update(this.options.rawDatasets); 
    } else {  
      var ds = this.getDatasetsFromDataSource(datasource, this.options.period, this.options.values, this.options.type); 
      this.update(ds); 
    } 
  },
  drawPlot: function(){ 
    // wraps .draw() so it can be called via weird function names as parameters thing
    this.plot.draw(); 
  }, 
  update: function(data){ 
    data = data || this.options.datasource;
    var datasets = this.buildDatasets(data); 
    this.plot.setData(datasets);  
    this.plot.setupGrid(); 
    this.plot.draw(); 
  }, 
  buildDatasets: function(data){ 
    // iterates over datasets in data and augments with default values 
    // and calculates calculated values 
    if(this.options.seriesColors.length < data.length){
      var errTxt = 'Not enough colors in widget flotChart default options for ' + data.length + ' series';
      $('.notifier').notifier('notify', 'error', errTxt);
    }  
    var sets = [];
    for(var i=0; i<data.length; i++){
      var tmpSet = $.extend({}, this.options.datasetDefaults); 
      tmpSet.data = data[i]; 
      // do any calculations on the individual sets here.
      tmpSet.color = this.options.seriesColors[i];
      tmpSet.label = labels.get(this.options.values[i]);
      tmpSet.bars.fill = tmpSet.lines.fill = this.options.linefillFun(i, data.length); 
      tmpSet.lines.show = this.options.showLines;
      tmpSet.bars.show = this.options.showBars;
      tmpSet.lines.steps = this.options.lineSteps;
      if('stack' in this.options){ 
        tmpSet.stack = this.options.stack; 
      } 

      sets.push(tmpSet);
    }  
    return sets;
  }, 
  initChart: function(data){ 
    // TODO: initChart gets called only in _create not when the chartType 
    // is changed by the user.  some of the differences between the 
    // charttypes will not be displayed if changed after _create... FIXME 
    var datasets = this.buildDatasets(data); 
    this.element.find('.chartContainer').empty();
    this.plot = $.plot(this.element.find('.chartContainer'), 
      datasets,
      {
        series: {
          lines:  { show: true, lineWidth: 1 },   
          bars:   { show: false, lineWidth: 10 }, 
          points: { show: false }, 
          shadowSize: 0,
          yaxis: 2
        }, 
        legend: { 
          show: true, 
          noColumns: 5,
          position: "ne", 
          margin: 1, 
          labelBoxBorderColor:null,
          backgroundColor:    null,
          backgroundOpacity:  0.2,
          container:          null
        }, 
        xaxis: {
          mode: "time",
          ticks: this.options.numberTicks,
          timeformat: this.options.timeformat, 
          autoscaleMargin: this.options.xaxis_autoscale || null
        },  
        yaxis: { 
          //min: 0, 
          ticks: this.options.numberTicks
        }, 
        grid: { 
          show: true,
          color: '#999',
          rickColor: null,
          backgroundColor: { colors: ['#e8e7e5', '#fff'] }, 
          borderWidth: 1,
          hoverable: false,
          autoHighlight: false
        } 
      }
    ); 
  }, 
  makeOptionsMenu: function(){ 
    var that = this;
    this.element.find('.optionsContainer').chartTypeEditor({
      period: this.options.period,
      chartType: this.options.chartType, 
      chartTypes: this.options.chartTypes, 
      values: this.options.values, 
      removeButton: this.options.removable, 
      change: function(e,args){
        var period = args[0],
            chartType = args[1], 
            values = args[2];
        that.options.period = period; 
        that.options.chartType = chartType;
        that.options.values = values; 
        that.options.rawDatasets = [];
        that.applyType();
        that.showChart();
        that.updateFromDatasource(); // FIXME this gets old data 
      } 
    });
  }, 
  getHTML: function(){
    var html = '<div class="flotChart_inner">';
    html += '<div class="fc_header">';
    html += '<h4 class="headline"></h4>';
    html += '<div class="chartButton"><img src="images/chart_icon.gif" alt="Chart"></div>';
    html += '<div class="listViewButton"><img src="images/List-Icon.gif" alt="List"></div>';
    html += '<div class="editOptionsButton"><img src="images/edit.png" alt="Edit"></div>';
    if(this.options.removable){ 
      html += '<div class="removeButton"><img src="images/close.png" alt="Remove"></div>';
    }
    html += '</div>';
    html += '<div class="container">';
    html += '<div class="chartContainer"></div>';
    html += '<div class="optionsContainer"></div>';
    html += '<div class="listViewContainer"></div>';
    html += '</div>';
    html += '</div>';
    return html; 
  }, 
  makeListView: function(){ 
    if(this.options.notInDatasource){ return 0; } 
    var html = '<table><thead><tr><th>Name</th><th>Val</th></tr></thead><tbody>';
    if(this.options.getValueFromDatasource){
      for(var i=0; i<this.options.available.length; i++){
        html += '<tr><td>';
        html += labels.get(this.options.available[i]); 
        html += '</td><td>';
        html += this.options.getValueFromDatasource(this.options.available[i], this.options.datasource)[1 /* not i */]; 
        html +='</td></tr>'; 
      } 
    } else { 
      var datasets = this.getDatasetsFromDataSource(this.options.datasource, this.options.period, this.options.available, this.options.type); 

      for(var i=0; i<this.options.available.length; i++){
        html += '<tr><td>';
        html += labels.get(this.options.available[i]); 
        html += '</td><td>';
        html +=  datasets[i] && datasets[i][datasets[i].length-1][1/*not i*/]; 
        html +='</td></tr>'; 

      } 

    }  
    html += '</tbody></table>';
    this.element.find('.listViewContainer').html(html); 
  }, 
  getOptions: function(){ 
    return this.options;
  } 
}); 

$.widget('ui.restForm', { 
  // this is a bit of a super-widget.
  // TODO: maybe break this into smaller chunks that are combined in
  // the controllers? 
  options: { 
    url: '',
    method: 'post', 
    newkey: true
  }, 
  _create: function(){

    if(!this.options.url){ 
      throw 'restForm needs an URL.';
    } 

    if(this.options.keylist){  
      // restForm can be passed a jquery selector to a keylist widget.
      // the keylist will then be updated automatically.
      this.keylist = this.options.keylist; 
    } 

    this.key = ''; 
    this.val = ''; 
    this.newkey = this.options.newkey;
    this.element.addClass('restForm'); 
    $(this.html()).appendTo(this.element); 
    this.updateUI();
    this.element.find('.postputbutton').click(function(){ 
      $(this).parents('.restForm').restForm('httpPostPut'); 
    });
    this.element.find('.getbutton').click(function(){ 
      $(this).parents('.restForm').restForm('httpGet'); 
    });
    this.element.find('.deletebutton').click(function(){ 
      $(this).parents('.restForm').restForm('httpDelete'); 
    });
    this.element.find('.keyInput').keyup(function(e){ 
      if(e.keyCode === 13){  // enter key
        $(this).parents('.restForm').restForm('httpGet'); 
        e.stopPropagation();
        return 0;
      } 
      $(this).parents('.restForm').restForm('keyChanged'); 
    });
  }, 
  getKey: function(){ 
    return this.element.find('.keyInput').val();
  }, 
  getVal: function(){ 
    return this.element.find('.valTextarea').val();
  }, 
  updateUI: function(){
    var newkey = this.newkey;  
    if(newkey){ 
      this.element.find('.postputbutton').attr('value', 'POST'); 
      this.element.find('.deletebutton').attr('disabled', true); 
    } else { 
      this.element.find('.postputbutton').attr('value', 'PUT'); 
      this.element.find('.deletebutton').attr('disabled', false); 
    }  
  }, 
  keyChanged: function(){ 
    if(this.keylist.keyList('hasKey', this.getKey())){ 
      this.newkey = false; 
    } else { 
      this.newkey = true;
    } 
    this.updateUI();
  }, 
  httpGet: function(key){ 
    if(key){
      this.element.find('.keyInput').val(key);
    } 
    this.fire('GET'); 
  }, 
  httpPostPut: function(){ 
    this.fire(this.newkey ? 'POST' : 'PUT', this.getVal()); 
  }, 
  httpDelete: function(){ 
    this.fire('DELETE'); 
  }, 
  fire: function(method, val){ 
    val = val || '';
    method = method.toUpperCase();
    var key = this.getKey(); 
    if(!key){ return 0; } 
    var url =  this.options.url + key + '/'; 
    var ajaxOptions = {
      url: url,
      type:  method,
      data: val, 
      context:this,
      //dataType: 'json', 
      success: function(data, code, xhr){
        switch(method){ 
          case 'GET':
            this.newkey = false; 
            this.element.find('.valTextarea').val(data); 
            this._trigger('onKey', null, [key]); 
          break;
          case 'DELETE':
            this.newkey = true; 
            this.element.find('.valTextarea').val(''); 
            this.element.find('.keyInput').val(''); 
            this.keylist.keyList('remove', key); 
          break;
          case 'POST':
          case 'PUT':
            this.newkey = false; 
            this._trigger('onKey', null, [key]); 
          break;
        } 
        this.updateUI(); 
        this._trigger('result', null, [data, code, xhr, method, url]); 
      }, 
      error: function(xhr, code, err) {
        this._trigger('result', null, [err, code, xhr, method, url]); 
      }
    }
    if((method==='POST' || method==='PUT') && this.options.extendedvals){ 
      var xkv = this.options.extendedvals; 
      if(xkv.validate()){
        ajaxOptions.beforeSend = function(xhr, settings){
          xkv.setHeaders(xhr); 
        } 
      } else {
        $('.notifier').notifier('add', 'error','Extended Key Values not valid.');
        return 0;
      } 
    } 
    $.ajax(ajaxOptions);
  }, 
  html: function(){ 
    var html = '';
    html += '<div>';
    html += '<div class="">';
    html += '<input type="text" name="key" value="'+this.key+'" class="keyInput" />'; 
    html += '<input type="button" class="httpbutton postputbutton" value="POST">';
    html += '<input type="button" class="httpbutton getbutton" value="GET">';
    html += '<input type="button" class="httpbutton deletebutton" value="DELETE">';
    html += '<hr>'; 
    html += '<textarea name="val" class="valTextarea">'; 
    html += this.val; 
    html += '</textarea>';
    html += '</div>';
    //html += '<h3>Keys</h3>';
    //html += '<div class="keys"></div>';
    html += '</div>';
    return html; 
  } 
});  // end of restForm

$.widget('ui.keyList', { 
  // maintains an array of keynames (this.keys) and displays am alphabetically
  // ordered and devided ui of keys. 
  // of a key is clicked the 'keyklick'-event is triggered.
  // 
  // if a single alphabetic list has more keys than this.options.pagination
  // it will be paginated using the widget 'slideList' 
  options: { 
    allowDuplicates: false, 
    pagination: 50
  }, 
  _create: function(){
    this.keys = []; 
    this.element.addClass('keyList'); 
    this.element.append($('<h3 />',{
      text:'Keys',
      click: function(e){ 
        $(this).parent().find('.outer').toggle('fast'); 
      } 
    }));
    this.element.append(this.html());
    var that = this;  

    this.element.find('ul').live('click',function(e){ 
      var target = $(e.target); 
      if(target.hasClass('keyDelButton')){ 
        that.remove(target.parents('li').text());
        e.stopPropagation();
      } else {
        that._trigger('keyclick', null, $(e.target).text()); 
      }
    });
  }, 
  hasKey: function(key){ 
    // returns true if the key is in the keylist
    for(var i=0; i<this.keys.length; i++){ 
      if(key === this.keys[i]){ 
        return true;
      } 
    }
    return false; 
  }, 
  add: function(key){ 
    // add a key to the list.
    // the key is added to the list corresponding to it's first letter.
    // if there is no list for the first letter, a new list is created and
    // inserted to the other lists at the right position
    if(this.options.allowDuplicates || (!this.hasKey(key))){
      var alpha = key.charAt(0).toLowerCase(); 
      this.keys.push(key);  
      this.keys.sort();
      var list =this.element.find('.keys_with_'+alpha); 
      if(list.length > 0){
        list.empty(); 
      } else { 
        // find proper attachpoint 
        var that = this; 
        this.element.find('.keylists [class^="keys_with_"]').each(function(i){ 
          if($(this).attr('class') > 'keys_with_'+alpha){ 
            $(this).parent().before(that.newlist(alpha));  
            return false; 
          } 
        }); 
        list =this.element.find('.keys_with_'+alpha); 
        if(list.length < 1){ 
          this.element.find('.keylists').append(this.newlist(alpha)); 
          list =this.element.find('.keys_with_'+alpha); 
        }
      }  
      list.append(this.keyshtml(alpha)); 
      return true;
    }  
    return false;
  }, 
  remove: function(key){ 
    // removes a key from the list
    var keys = this.keys;
    this.element.find('li').each(function(i){
      if($(this).text() === key){
        $(this).remove();
        keys.splice(keys.indexOf(key),1); 
      } 
    });
  },
  keyshtml: function(alpha){ 
    var keys = this.getKeysForAlpha(alpha); 
    if(keys.length <= this.options.pagination){ 
      var list = $('<ul />', { className: 'keyblock' }); 
      for(var i=0; i<keys.length; i++){ 
        list.append(this.itemHTML(keys[i])); 
      } 
      return list; 
    } else { 
      var list = $('<div />', {className:'oversize keyblock'}); 
      var ul = $('<ul />'); 
      for(var i=0; i<keys.length; i++){ 
        ul.append(this.itemHTML(keys[i])); 
        if(ul.children().length >= this.options.pagination){
          ul.appendTo(list); 
          ul = $('<ul />'); 
        } 
      } 
      ul.appendTo(list); 
      //this.element.find('h4').css('float', 'left'); 
      list.slidelist({});
      return list; 
    }
  }, 
  newlist: function(alpha){ 
    var list = $('<div />', {className: 'clearboth' });
    $('<h4 />', {
      id: 'headline_keylist_' + alpha,
      text: alpha.toUpperCase(),
      click: function(e){
        $(this).parent().find('.keyblock').toggle('fast');
      } 
    }).appendTo(list); 
    $('<div />', {className: 'keys_with_'+alpha }).appendTo(list); 
    return list; 
  }, 
  listshtml: function(){ 
   var lists = []; 
   if(this.keys.length < 1){ 
     return lists; 
   } 
   this.keys.sort();  
   var alpha = this.keys[0].charAt(0).toLowerCase(); 
   lists.push(this.newlist(alpha)); 
   for(var i=0; i<this.keys.length; i++){ 
     if( alpha !== this.keys[i].charAt(0).toLowerCase()){ 
       alpha = this.keys[i].charAt(0).toLowerCase(); 
       lists.push(this.newlist(alpha)); 
     } 
     lists[lists.length-1].find('ul').append(this.itemHTML(this.keys[i])); 
   } 
   return lists;
  }, 
  getKeysForAlpha: function(alpha){ 
    // returns an array of keys starting with the character <alpha>
    alpha = alpha.toLowerCase();
    return $.grep(this.keys, function(key){ 
      return alpha === key.charAt(0).toLowerCase();
    });
  }, 
  html: function(){ 
    var outer = $('<div />', {className:'outer'});
    var that = this;
    $('<input />', {
      className: 'keySearchField', 
      type:'text', 
      keyup: function(e){
        if(e.keyCode === 13){  // enter key 
          that.prefixSearch(that.element.find('.keySearchField').val()); 
        }
      } 
    }).appendTo(outer);
    $('<input />', { 
      className: 'keySearchButton',
      type:'button', 
      value:'Key Search',
      click: function(){  
         that.prefixSearch(that.element.find('.keySearchField').val()); 
      } 
    }).appendTo(outer);
    $('<input />', { 
      className: 'allKeySearchButton',
      type:'button', 
      value:'All',
      click: function(){  
         // iterate over the alphabet and perform a prefix search for every 
         // letter/number.
         // it's little brute-force. not sure how this scales to lots of keys. 
         var abc = 'abcdefghijklmnopqrstuvwxyz1234567890';
         for(var i=0; i<abc.length; i++){
           that.prefixSearch(abc.charAt(i)); 
         } 
      } 
    }).appendTo(outer);

    var listsDiv = $('<div />', {className:'keylists'}).appendTo(outer);
    $('<div />', {className: 'clearboth'}).appendTo(outer);
    var lists = this.listshtml(); 
    for(var i=0; i<lists.length; i++){ 
      lists[i].appendTo(listsDiv); 
    } 

    return outer; 
  }, 
  item: function(key){ 
    return $(this.itemHMTL(key)); 
  }, 
  itemHTML: function(key){ 
    return '<li>' + key + '<span class="keyDelButtonCont"><img class="keyDelButton" src="images/deleteIcon.gif" /></span></li>'; 
  }, 
  prefixSearch: function(prefix){ 
    $.ajax({
      url: this.options.keysearchurl + prefix, 
      type:  'GET',
      context:this, 
      dataType: 'json', 
      success: function(data, code, xhr) {
        var msg = 'Key search "'+prefix+'": ' + data.length + ' Keys found.';
        $('.notifier').notifier('add', 'info', msg);
        for(var i=0; i<data.length; i++){
          this.add(data[i]);
        } 
      }, 
      error: function(xhr){
         var err = JSON.parse(xhr.responseText);
         $('.notifier').notifier('notify', 'error', 'Key search for "'+prefix+'": ' + err.message);
      } 
    }); 
  } 
}); // end of keyList
 
// -------------------------------------------------------------------------

$.widget('ui.httpRequestList', { 
  // Displays the data of http request and responses as stacked divs
  // only used in the rest interface, acutally i don't see much reusability. 
  // this is only a widget for encapsulation issues. 
  options: { 
    maxLen: 5 
  }, 
  _create: function(){
    this.element.addClass('httpRequestList'); 
    this.element.append($(this.html()));
  }, 
  html: function(){ 
    return '<ul></ul>';
  },
  addCycle: function(method, url, xhr, body){ 
    // add both request and response as one <li>. 
    //
    // this is called in the restEditor controller, making a connection to the 
    // restForm-widget
    var html = '';
    html += '<li>';
    html += this.requestHTML(method, url, body);
    html += this.responseHTML(xhr, body);
    html += '</li>';
    $(html).prependTo(this.element.find('ul')); 
    // if a maximum number of items is configured, cut at the bottom.
    if(this.options.maxLen){ 
      var max = this.options.maxLen; 
      this.element.find('li').each(function(i){  
        if(i>=max){ 
          $(this).remove(); 
        }  
      }); 
    } 
  }, 
  requestHTML: function(method, url, body){ 
    // returns a html-string representing one http request 
    var html = ''; 
    html += '<div class="request">'; 
    html += method.toUpperCase() + ' ' +  url + ' HTTP/1.1<br />';
    if(method === 'PUT' || method === 'POST' ){ 
      html += this.bodyHTML(body);
    } 
    html += '</div>';
    return html;
  },
  responseHTML: function(xhr, body){ 
    // returns a html-string representing one http response 
    var html = ''; 
    html += '<div class="response">'; 
    html += 'HTTP/1.1 ' + xhr.status + ' ' + xhr.statusText + '<br />';

    html += '<pre>' + xhr.getAllResponseHeaders() + '</pre>';
    html += this.bodyHTML(body);
    html += '</div>';
    return html;
  }, 
  bodyHTML: function(body){ 
    // returns a html string for a http body
    var html = '';
    html += '<div class="http_body">';
    html += body;
    html += '</div>';
    return html;
  } 
}); // end of httpRequestList

// -------------------------------------------------------------------------

$.widget('ui.slidelist', { 
  // if used on an element containing multiple <ul> it all of the <ul> but
  // one and adds a pagination to step through the lists
  // 
  // so far, this is only used from the widget keyList. this means it's hard get to the options from 'outside' (meaning from somewhere where the voc-object is available).   this might be worth a refactor. 
  options: { 
    activeSlide: 0, 
    elem: 'ul',  // it's possible to use another element that <ul> 
    prevLink:  '<< prev ',
    nextLink: ' next >>'
  }, 
  _create: function(){
    this.element.addClass('slidelist');
    this.slideCount = this.element.find(this.options.elem).length; 
    $('<div />', { className: 'clearboth' }).prependTo(this.element);  

    // create the paginator 
    var navi = $('<div />', { className: 'slidelist_navi' }).prependTo(this.element); 
    $('<a />', { 
      href: '#',
      className: 'prev', 
      text: this.options.prevLink, 
      click: function(e){ 
        $(this).parents('.slidelist').slidelist('prev'); 
        e.preventDefault();
      }  
    }).appendTo(navi);  
    $('<span />', { 
      className: 'positionIndicator', 
      text: (this.options.activeSlide+1)+' / '+this.slideCount
    }).appendTo(navi);  
    $('<a />', { 
      href: '#',
      className: 'next', 
      text: this.options.nextLink, 
      click: function(e){ 
        $(this).parents('.slidelist').slidelist('next'); 
        e.preventDefault();
      }  
    }).appendTo(navi);  

    // initialise the lists 
    this.element.find(this.options.elem).hide(); 
    this.element.find(this.options.elem).eq(this.options.activeSlide).show(); 
  }, 
  updatePositionIndicator: function(){ 
    // redraws the "page n of m"-display 
    this.element.find('.positionIndicator').text((this.options.activeSlide+1)+' / '+this.slideCount); 
  },
  next: function(){ 
    // displays the next element 
    if(this.options.activeSlide < this.slideCount-1){ 
      this.options.activeSlide++;
      this.element.find(this.options.elem).hide(); 
      this.element.find(this.options.elem).eq(this.options.activeSlide).show(); 
      this.updatePositionIndicator();
    } 
  }, 
  prev: function(){ 
    // displays the previous element 
    if(this.options.activeSlide > 0){ 
      this.options.activeSlide--;
      this.element.find(this.options.elem).hide(); 
      this.element.find(this.options.elem).eq(this.options.activeSlide).show(); 
      this.updatePositionIndicator();
    } 
  } 
});// end of slidelist
 

// -------------------------------------------------------------------------

$.widget('ui.notifier', { 
  // widget that displays messages. 
  // there should be only one instance of this.
  options: { 
    maxLen: 7, 
    notifyDuration : 2000
  }, 
  _create: function(){
    this.element.addClass('notifier');
    this.element.hide();
    var box = $('<div />', {className:'notifications hidden'} ).appendTo(this.element);
    $('<img />', {
      className: 'closebutton',
      src:"images/close.png",
      click: function(){
        $(this).parents('.notifier').notifier('hide');
      }
    }).appendTo(box); 
    $('<ul />').appendTo(box); 
  }, 
  notify: function(type, text){ 
    // opens the notifier to show a new notification and automatially closes
    // notifier after a timeout. 
    this.show(); 
    this.add(type, text); 
    var that = this;

    // the handle of the timeout is stored to prevent multiple timouts 
    // concurrently trying to close the same notifier.
    // the last timeout 'wins' to ensure the display time per item.
    if(this.closetimeout){ 
      clearTimeout(this.closetimeout); 
    } 
    this.closetimeout = setTimeout(function(){ 
      that.hide();
      that.closetimeout = null;   
    }, this.options.notifyDuration); 
  }, 
  show: function(){ 
    this.element.find('.notifications').addClass('hidden');
    this.element.css('display', 'block');
    this.element.find('.notifications').removeClass('hidden');
  }, 
  add: function(type, text){ 
    var note = $('<li>', {className: 'notification '+type, text: text });
    note.addClass('hidden');
    note.prependTo(this.element.find('ul')); 
    note.removeClass('hidden');

    // enforce the maximum itemcount if there is any. 
    if(this.options.maxLen){ 
      var max = this.options.maxLen; 
      this.element.find('li').each(function(i){  
        if(i>=max){ 
          $(this).remove(); 
        }  
      }); 
    } 
  }, 
  hide: function(){ 
    this.element.find('.notifications').addClass('hidden');
    var that = this; 
    setTimeout(function(){ that.element.hide(); }, 500 ); 
  } 
});

// --------------------------------------------------------

$.widget('ui.extendedKeyVals', { 
  // widget that costructs a form from the servers extended key-vals
  // definition. the values can be editied and attached to a xhr object as 
  // the appropriate http-header. 
  // its used in combination with restform. restform takes an 
  // extendedKeyVals-Widget-object as an option and uses the setHeaders-method
  options: { 
    httpHeaderName: 'x-voc-extended', 
    // TODO: types-object should be provided by the voc-server 
    // and fetched on initialisation of extendedKeyVals-object
    /* 
    types: { 'piff': 'string', 
             'paff':'integer',
             'wann':'datetime', 
             'isNaN': 'boolean'
    }, 
    */
  }, 
  _create: function(){

    this.map = this.options.map || {}; 
    this.element.addClass('extendedKeyVals'); 

    // if there is not 'types'-array defining the aavailable extended-vals
    // the widget shall display nothing.
    if(!'types' in this.options || !this.options.types){ 
      this.getTypes(); 
    } else { 
      this.display();
    } 
  },
  getTypes: function(){ 
    // load the types definition via xhr 
    // TODO!
    if(!this.options.typesUrl){
      throw "extendedKeyVals-Widgets need typesUrl to load type-definition.";
      return 0; 
    } 
    $.ajax({ 
      url: this.options.typesUrl,
      context: this, 
      dataType: 'json', 
      success: function(types){ 
        this.options.types = types; 
        this.display();
      }, 
      error: function(err){ 
        var errTxt = 'Error on fetching recent-data. ' + err.message;
        $('.notifier').notifier('notify', 'error', errTxt);
      } 
    }); 

  }, 
  loadMap: function(xhr){ 
    // takes a xhr object, extracts the data from the header and
    // (re-)displays the form.
    var xkvstr = xhr.getResponseHeader(this.options.httpHeaderName); 
    if(xkvstr){ 
      this.map = JSON.parse(xkvstr); 
      this.display();
    } 
  }, 
  updateMap: function(){ 
    // read the xkv-form and fill the key-val object in this.map 
    // with the proper datatypes. 
    var map = this.map = {}; 
    var types = this.options.types;
    this.element.find('.xkv_keyval').each(function(i){ 
      var key = $(this).attr('name');
      var val = $(this).val();
      // integers need to be real integers in order to be
      // posted/put back to the voc
      if(types[key] === 'integer') { 
        val = parseInt(val, 10) || 0;
      } else if (types[key] === 'boolean') {
        val = !!$(this).is(':checked');  // get checked status as bool
      } else if (types[key] === 'datetime') {
        if(val){ 
          val = new Date(val);
        } else { 
          val = null;
        } 
      } else if (types[key] === 'double') {
        val = parseFloat(val,10);
      } 
      if(key){ 
        map[key] = val; 
      } 
    });  
  }, 
  setHeaders: function(xhr){ 
    // takes a xhr object and sets the header to write the xkv back to the 
    // server. 
    this.updateMap(); 
    for( var name in this.options.types){ 
      if(this.options.types.hasOwnProperty(name) && this.options.types[name]==='datetime' && this.map[name]){ 
        // convert timestamp to milliseconds from epoch
        this.map[name] = parseFloat(Math.round(this.map[name].getTime() / 1000) + '.0', 10);
        //this.map[name] = ISODateString(this.map[name]); 
      }
    }
    xhr.setRequestHeader(this.options.httpHeaderName, JSON.stringify(this.map));
  }, 
  validate: function(){ 
    // checks of values this.map are corresponding to the datatypes in
    // this.options.types 
    // if value for any key doesn't validate, there will be a notification
    // and false is returned. 
    // if all vals are ok true is returned.
    this.updateMap(); 
    var fail = false; 
    for( var name in this.options.types){ 
      if(this.options.types.hasOwnProperty(name)){ 
        switch( this.options.types[name] ) { 
          case 'integer': 
            if(isNaN(parseInt(this.map[name],10))){ 
              var errtxt = ucfirst(name) + ' must be an Integer.';
              $('.notifier').notifier('notify', 'error', errtxt );
              fail = true; 
            } 
          break;
          case 'double': 
            if(isNaN(parseFloat(this.map[name],10))){ 
              var errtxt = ucfirst(name) + ' must be a Double.';
              $('.notifier').notifier('notify', 'error', errtxt );
              fail = true; 
            } 
          break;
          case 'datetime': 
            if(this.map[name] && this.map[name].toString() === 'Invalid Date'){ 
              var errtxt = ucfirst(name) + ' must be a Datetime.';
              $('.notifier').notifier('notify', 'error', errtxt );
              fail = true; 
            } 
          break;
        } 
      }
    }
    return !fail;
  }, 
  createForm: function(){ 
    // returns a form-element (as jQuery-obj)
    var form = $('<form />', {name:'xkvForm' });
    for( var name in this.options.types){ 
      if(this.options.types.hasOwnProperty(name)){ 
        var lbl = $('<label />',{text:ucfirst(name)}).appendTo(form); 
        var type = this.options.types[name]; 
        var funName = '_create'+ucfirst(type)+'Field'; 
        lbl.append(this[funName](name, this.map[name])); 
      }
    }
    return form;
  }, 
  // functions that return jquery-objects for input elements 
  // for the different datatypes
  _createStringField: function(name, value){ 
    return $('<input />', {type:"text", value:value, name:name, className:'xkv_keyval'});
  },
  _createIntegerField: function(name, value){ 
    return $('<input />', {type:"text", value:value, name:name, className:'xkv_keyval'});
  },
  _createDatetimeField: function(name, value){
    // this uses the jquery-ui datepicker 
    // plus a timepicker-plugin by w3visions.com 
    // http://blog.w3visions.com/2009/04/date-time-picker-with-jquery-ui-datepicker/
    return $('<input />', {
      type:"text", value:value,
      name:name,
      className:'xkv_keyval', 
      ready: function(e){
      } 
    }).datetimepicker({});
  }, 
  _createDoubleField: function(name, value){ 
    return $('<input />', {type:"text", value:value, name:name, className:'xkv_keyval'});
  }, 
  _createBooleanField: function(name, value){ 
    var o = {type:"checkbox", value:'1', name:name, className:'xkv_keyval'};
    if(value){ 
      o['checked'] = true; 
    } 
    return $('<input />',o );
  }, 
  display: function(){ 
    this.element.empty();
    $('<h3 />', {
      text: 'Extendend Key-Values',
      click: function(){ 
        $(this).parent().find('form').toggle('fast'); 
      } 
    }).appendTo(this.element); 
    this.element.append(this.createForm()); 
  }
});

// ---------------------------------------------- 

/*
 *  type, request (http-str), result, runtime (ms), response
 *
 *  type, request, resultCode, runtime, resultSize
 *
 */ 

$.widget('ui.recentRequests', { 
  options: { 
    types: [1,2,3], 
    typesName: ['', 'HTTP', 'Postfix', 'Memcache', 'Admin'],
    size: 10 
  }, 
  _create: function(){
    if(!this.options.url){ 
      throw "recent requests widget needs to be passed the URL to the logs."; 
    } 
    this.data = {}; 
    /*
    "type" : [ 1, 1, 2, 3 ],
    "request" : [ "GET /value/emil", "GET /value/hugo", "GET emil", "GET hugo" ],
    "resultCode" : [ 200, 500, 200, 1 ],
    "runtime" : [ 10, 20, 10, 10 ],
    "resultSize" : [ 200, 300, 100, 1 ]
     */
    this.element.addClass('recentRequests'); 
    this.display();
  },  
  resultDistribution: function(){ 
  }, 
  averageRuntime: function(){ 
  }, 
  runtimeTimeseries: function(){ 
  }, 
  table: function(){ 
    html = '<table>'; 
    html += '<tr><th>Type</th><th>Request</th><th>Code</th><th>Runtime</th><th>Size</th></tr>'; 
    for(var i=0; i<this.data.type.length; i++){ 
      html += this.tableRow(
        this.data[i].type, 
        this.data[i].request, 
        this.data[i].resultCode, 
        this.data[i].runtime, 
        this.data[i].resultSize 
      ); 
    } 
    html += '</table>';
    return html; 
  }, 
  tableRow: function(type, request, code, runtime, size){ 
    var html = '<tr>'; 
    html += '<td>'+type+'</td>'; 
    html += '<td>'+request+'</td>'; 
    html += '<td>'+code+'</td>'; 
    html += '<td>'+runtime+' ms</td>'; 
    html += '<td>'+size+' Bytes</td>'; 
    html += '</tr>'; 
    return html;
  }, 
  replace: function(){ 
    // fetches requests from voc and replaces the old data.
  }, 
  append: function(data){ 
    // appends more requests to the allready displayed 
    // older requests  
    this.fetchRequests(function(data){ 
      this.data.type = this.data.type.concat(data.type); 
      this.data.request = this.data.request.concat(data.request); 
      this.data.resultCode = this.data.resultCode.concat(data.resultCode); 
      this.data.runtime = this.data.runtime.concat(data.runtime); 
      this.data.resultSize = this.data.resultSize.concat(data.resultSize); 
    }); 

  }, 
  fetchRequests: function(cb){ 
    // get the recent requests from the server  
    $.ajax({ 
      url: this.options.url,
      data: {
        types: this.options.types.join(','), 
        size: this.options.size
      }, 
      context: this, 
      dataType: 'json', 
      success: function(){ cb.apply(this, arguments); }, 
      error: function(err){ 
        var errTxt = 'Error on fetching recent-data. ' + err.message;
        $('.notifier').notifier('notify', 'error', errTxt);
      } 
    }); 
  } 
});

// ------------------------------------------------------- 

$.widget('ui.loginForm', { 
  options: { 
  }, 
  _create: function(){
    if(!this.options.url){ 
      throw "loginForm needs to be passed the URL to login."; 
    } 
    this.element.addClass('loginForm'); 
    this.display();
  }, 
  display: function(){ 
    var cont = $('<div />').appendTo(this.element); 
    $('<h3 />', { text: 'Login' }).appendTo(cont);  
    var lblUsr = $('<label />', { text: 'Username:' }).appendTo(cont);  
    $('<input />', { type: 'text', name:'username' }).appendTo(lblUsr);  

    var lblPwd = $('<label />', { text: 'Password:' }).appendTo(cont);  
    $('<input />', { type: 'password', name:'password' }).appendTo(lblPwd);  

    $('<input />', { 
      type: 'button', 
      value:'Login',
      click: function(){ 
        $(this).parents('.loginForm').loginForm('login');
      } 
    }).appendTo(cont);  

    $('<input />', { 
      type: 'button', 
      value:'Cancel',
      click: function(){ 
        $(this).parents('.loginForm').loginForm('cancel');
      } 
    }).appendTo(cont);  

  },  
  cancel: function(){ 
    this._trigger('cancel', null); 
  }, 
  login: function(username, password){ 
    password = password || this.element.find('input[name="password"]').val();
    if(password !== ''){ 
      password = calcMD5(password); 
    } 
    var data =  {
      user: username || this.element.find('input[name="username"]').val(),
      password: password
    };
    data = JSON.stringify(data);
    $.ajax({ 
      url: this.options.url,
      type: 'PUT',
      data:data, 
      context: this, 
      dataType: 'json', 
      success: function(data){ 
        this._trigger('loggedin', null, data); 
      }, 
      error: function(xhr){ 
        var err = JSON.parse(xhr.responseText); 
        var errTxt = 'Could not log in. ' + err.message;
        $('.notifier').notifier('notify', 'error', errTxt);
      } 
    }); 
  }  
});

// ------------------------------------------------------ 

$.widget('ui.userList', { 
  options: { 
    users: [] 
  }, 
  _create: function(){
    if(!this.options.url){ 
      throw "userList needs to be passed the URL."; 
    } 
    this.element.addClass('userList'); 
    this.getUsers();
  }, 
  getUsers: function(){ 
    $.ajax({ 
      url: this.options.url,
      type: 'GET',
      context: this, 
      dataType: 'json', 
      success: function(data){ 
        this.options.users = data.users;
        this.display(); 
      }, 
      error: function(err){ 
        var errTxt = 'Could not log in. ' + err.message;
        $('.notifier').notifier('notify', 'error', errTxt);
      } 
    }); 

  }, 
  removeItem: function(uname){ 
    this.element.find('li[title="'+uname+'"]').hide('fast', function(){ 
      $(this).remove();
    });
  }, 
  addItem: function(item){ 
    // 
    var list = this.element.find('ul');
    var li = $('<li />', { 
      title: item.name,
      text: item.name,
      click: function(){
        $(this).parents('.userList').userList('userClick', $(this).text());
        $(this).parent().find('li.active').removeClass('active');
        $(this).addClass('active');
      } 
    }).appendTo(list);
  }, 
  display: function(){ 
    var list = $('<ul />').appendTo(this.element);
    for(var i=0; i<this.options.users.length; i++){ 
      this.addItem({name: this.options.users[i]});  
    } 
  }, 
  userClick: function(usr){ 
    // trigger custom event
    this._trigger('userclick', null, [usr]); 
  } 
});

// ----------------------------------------------------- 

$.widget('ui.userDetail', { 
  options: { 
    roles: ['user', 'manager'],
    rights: { 
      '1000': 'RIGHT_TO_MANAGE_ADMIN',
      '1001': 'RIGHT_TO_MANAGE_USER',
      '1003': 'RIGHT_TO_LOGIN'
    } 
  }, 
  _create: function(){
    if(!this.options.url){ 
      throw "userDetail needs to be passed the URL."; 
    } 
    this.element.addClass('userDetail'); 
    //this.display();
  }, 
  getUserData: function(username){ 
    $.ajax({ 
      //url: this.options.url + username,
      url: this.options.url.replace('{username}', username),
      type: 'GET',
      context: this, 
      dataType: 'json', 
      success: function(data){ 
        this.display(data);
      }, 
      error: function(xhr){ 
        var err = JSON.parse(xhr.responseText); 
        var errTxt = 'Could not get userdata for '+username+'. '+err.message;
        $('.notifier').notifier('notify', 'error', errTxt);
      } 
    }); 
  }, 
  deleteUser: function(username){ 
    $.ajax({ 
      url: this.options.newUserUrl.replace('{username}', username),
      type: 'DELETE',
      context: this, 
      dataType: 'json', 
      success: function(data){ 
        this._trigger('userdeleted', null, username, data); 
      }, 
      error: function(xhr){ 
        var err = JSON.parse(xhr.responseText); 
        var errTxt = 'Could not get userdata for '+username + '. '+err.message;
        $('.notifier').notifier('notify', 'error', errTxt);
      } 
    }); 
  }, 
  userHasRight: function(data, right){ 
    // takes a userdata-object and 
    for(var i=0; i<data.rights.length; i++){ 
      if(data.rights[i] == right){ 
        return true; 
      } 
      return false; 
    } 
  }, 
  empty: function(data){ 
    this.element.empty();
  }, 
  display: function(data){ 
    // username = username || this.options.username; 
    this.element.empty();
    $('<h2 />', { text: 'User Detail' }).appendTo(this.element);  

    // name
    // this.nameField(data).appendTo(this.element); 
    $('<h3 />', { text: data.name }).appendTo(this.element);  
    
    // role 
    this.roleField(data).appendTo(this.element); 

    // rights
    $('<div />', { className: 'clearboth' }).appendTo(this.element);  
    for(var right in this.options.rights){ 
      if(this.options.rights.hasOwnProperty(right)){ 
        var rightDiv = $('<div />', { 
          className: 'rightOption', 
          text: this.options.rights[right]
        }).appendTo(this.element); 
        $('<input />', { 
          type: 'checkbox',
          name: right, 
          checked: this.userHasRight(data, right),
          disabled: true
        }).prependTo(rightDiv); 
      } 
    }

    lbl('New Password', $('<input />', { 
      type: 'password',
      value: '', 
      name: 'Passwd'
    })).appendTo(this.element);  

    lbl('Repeat', $('<input />', { 
      type: 'password',
      value: '',
      name: 'Passwd2'
    })).appendTo(this.element);  



    $('<div />', { className: 'clearboth' }).appendTo(this.element);  


    var  ud = this; 

    $('<input />', { 
      type: 'button',
      value: 'Update User',
      click: function(e){ 
      } 
    }).appendTo(this.element);  

    $('<input />', { 
      type: 'button',
      value: 'Delete User',
      click: function(e){ 
        ud.element.userDetail('deleteUser', data.name);
      } 
    }).appendTo(this.element);  

  }, 
  nameField: function(data){ 
    // name
    return $('<input />', { 
      type: 'text',
      name: 'name',
      value: data.name 
    })
  }, 
  roleField: function(data){ 
    return makeSelect('role', this.options.roles, function(elm){ return elm===data.role });
  }, 
  newUserForm: function(){ 
    var nuf = $('<div />', {className: 'newUserForm'});
    $('<h3 />', { text: 'New User' }).appendTo(nuf); 
    lbl('Username',this.nameField({name:''})).appendTo(nuf); 
    lbl('Role',this.roleField({role:'user'})).appendTo(nuf); 
    lbl('Password',$('<input />', { type: 'password', name: 'passwd' })).appendTo(nuf);
    lbl('Password2',$('<input />', { type: 'password', name: 'passwd2' })).appendTo(nuf);

    var that = this; 
    $('<input />', { type: 'button', value: 'OK', 
      click: function(){
        that.element.userDetail('newUserFromForm', $(this).parents('.newUserForm')); 
    }}).appendTo(nuf);
    return nuf;
  },
  newUserFromForm: function(form){ 
    var username = form.find('[name="name"]').val(); 
    var password = form.find('[name="passwd"]').val(); 
    var password2 = form.find('[name="passwd2"]').val(); 
    var role = form.find('[name="role"]').val(); 

    if(password !== password2){ 
      $('.notifier').notifier('notify', 'error', 'Passwords don\'t match.');
      return false; 
    } else { 
      this.newUser( username, password, role); 
    }

      
  }, 
  newUser: function(username, password, role){ 
    var data = '{"password" : "'+calcMD5(password)+'", "role" : "'+role+'"}';
    $.ajax({ 
      url: this.options.newUserUrl.replace('{username}', username),
      data: data, 
      type: 'POST',
      context: this, 
      dataType: 'json', 
      success: function(data){ 
        this._trigger('newuser', null, data); 
      }, 
      error: function(xhr){ 
        var err = JSON.parse(xhr.responseText); 
        var errTxt = 'Could not create user '+username+'. ' + err.message;
        $('.notifier').notifier('notify', 'error', errTxt);
      } 
    }); 
  } 
});
 
})(jQuery);
