////////////////////////////////////////////////////////////////////////////////
/// @brief simple key-value model
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef SIMPLEVOC_SIMPLEVOC_SIMPLE_MODEL_H
#define SIMPLEVOC_SIMPLEVOC_SIMPLE_MODEL_H 1

#include <Basics/Common.h>

#include <Basics/Mutex.h>
#include <Hpdf/AttributeDescriptor.h>
#include <Hpdf/DynamicSort.h>

#include "simple.h"

namespace triagens {
  namespace rest {
    class Task;
    class Scheduler;
  }

  namespace hpdf {
    class HpdfFilterTree;
  }

  namespace simple {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief response types
    ////////////////////////////////////////////////////////////////////////////////

    enum ResponseType {
      ERROR,
      EXISTS,
      NOT_FOUND,
      NOT_STORED,
      OK,
      STORED,
      KEY_LIMIT_EXCEEDED,
      VALUE_LIMIT_EXCEEDED,
      NUM_KEYS_LIMIT_EXCEEDED,
      TOTAL_SIZE_EXCEEDED,
      TOTAL_KEY_SIZE_EXCEEDED,
      TOTAL_VALUE_SIZE_EXCEEDED,
      GENERIC_LIMIT_EXCEEDED
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief value structure
    ////////////////////////////////////////////////////////////////////////////////

    struct Value {
      Value ()
        : creationTime(0.0),
          deletionTime(0.0),
          expireTime(0.0),
          lastAccess(0.0),
          lastUpdate(0.0),
          deletionPending(false),
          flags(0),
          cas(0),
          flushNumber(0),
          incrDecr(0),
          flushDelay(0.0) {
      }

      double creationTime;
      double deletionTime;
      double expireTime;
      double lastAccess;
      double lastUpdate;
      bool   deletionPending;
      uint64_t flags;
      uint64_t cas;
      uint64_t flushNumber;
      uint64_t incrDecr;
      double flushDelay;
      string data;
      string identifier;
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief statistics structure
    ////////////////////////////////////////////////////////////////////////////////

    struct StatsValues {
      StatsValues ()
        : numKeyValue(0),
          totalDataSize(0),
          totalKeySize(0),
          serverStartup(0) {
      }

      uint64_t numKeyValue;
      uint64_t totalDataSize;
      uint64_t totalKeySize;
      datetime_t serverStartup;
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief limit structure
    ////////////////////////////////////////////////////////////////////////////////

    struct LimitValues {
      LimitValues ()
        : maxNumKeys(0),
          maxKeySize(0),
          maxValueSize(0),
          maxTotalKeySize(0),
          maxTotalSize(0),
          maxTotalValueSize(0),
          limitsInUse(false) {
      }

      uint64_t maxNumKeys;
      uint64_t maxKeySize;
      uint64_t maxValueSize;
      uint64_t maxTotalKeySize;
      uint64_t maxTotalSize;
      uint64_t maxTotalValueSize;
      bool limitsInUse;
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief simple key-value model
    ////////////////////////////////////////////////////////////////////////////////

    class SimpleModel {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief current time
        ////////////////////////////////////////////////////////////////////////////////

        static datetime_t now ();

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        SimpleModel ();

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructor
        ////////////////////////////////////////////////////////////////////////////////

        virtual ~SimpleModel ();

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief sets the scheduler
        ////////////////////////////////////////////////////////////////////////////////

        void setScheduler (rest::Scheduler* scheduler) {
          this->scheduler = scheduler;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a new extended key-value description
        ////////////////////////////////////////////////////////////////////////////////

        bool addExtendedKeyValueDescription (string const& description);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns the current extended key value description
        ////////////////////////////////////////////////////////////////////////////////

        vector<hpdf::AttributeDescriptor> const& getExtendedKeyValues () {
          return extendedKeyValues;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns current set of limits imposed
        ////////////////////////////////////////////////////////////////////////////////

        LimitValues const& getLimitValues () const {
          return limits;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief sets a new limit
        ////////////////////////////////////////////////////////////////////////////////

        void setLimitValue (ResponseType response, uint64_t limitValue);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief sets a set of limits
        ////////////////////////////////////////////////////////////////////////////////

        void setLimitValues (LimitValues& limitValues);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief finalize the model
        ////////////////////////////////////////////////////////////////////////////////

        void finalise ();

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief build a filter
        ////////////////////////////////////////////////////////////////////////////////

        hpdf::HpdfFilterTree* dynamicFilterTree (string const& parseString);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief build a sorting
        ////////////////////////////////////////////////////////////////////////////////

        hpdf::DynamicSort::DynamicSortDescription dynamicSortDescription (string const& parseString);

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief create key-value
        ////////////////////////////////////////////////////////////////////////////////

        hpdf::databases::simple::TR_SimpleTable const* create (Value const& value);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief read key-value
        ////////////////////////////////////////////////////////////////////////////////

        hpdf::databases::simple::TR_SimpleTable const* lookup (string const& identifier);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief finds all rows given a prefix key
        ////////////////////////////////////////////////////////////////////////////////

        vector<hpdf::databases::simple::TR_SimpleTable const*> lookupAll (string const& prefix);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief update key-value data
        ////////////////////////////////////////////////////////////////////////////////

        void updateData (hpdf::databases::simple::TR_SimpleTable const*, string const& data);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief update cas
        ////////////////////////////////////////////////////////////////////////////////

        void updateCas (hpdf::databases::simple::TR_SimpleTable const*);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief update last access
        ////////////////////////////////////////////////////////////////////////////////

        void updateLastAccess (hpdf::databases::simple::TR_SimpleTable const*);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief update information
        ////////////////////////////////////////////////////////////////////////////////

        void updateInformation (hpdf::databases::simple::TR_SimpleTable const*, Value const&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief update extended key-values
        ////////////////////////////////////////////////////////////////////////////////

        void updateExtended (hpdf::databases::simple::TR_SimpleTable const* row, basics::VariantArray* extended);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief delete key-value
        ////////////////////////////////////////////////////////////////////////////////

        void destroy (hpdf::databases::simple::TR_SimpleTable const*);

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds row to database
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType addValue (Value&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds row to database with extended key-values
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType addValue (Value&, basics::VariantArray* extended);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief appends data to a row
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType appendValue (Value&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks and sets a row
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType casValue (Value&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief decrements the data by the amount specified in value
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType decrValue (Value&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief marks a row as deleted
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType deleteValue (Value&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief database wide deletion with a possible delay
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType flushAllValue (Value& value);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns a value based upon a key
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType getValue (Value&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns a value based upon a key
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType getValue (Value&, basics::VariantObject*& extended);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief decrements the data by the amount specified in value
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType incrValue(Value&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns a list of values when a prefix key is used
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType prefixGetValue (Value&, vector<Value>&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns a list of values when a prefix key is used
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType prefixGetValue (Value&,
                                     hpdf::HpdfFilterTree*,
                                     hpdf::DynamicSort::DynamicSortDescription const&,
                                     vector<Value>&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief prepends data to a row
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType prependValue (Value&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief replaces a value in a row with another sent by client
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType replaceValue (Value&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief replaces a value in a row with another sent by client
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType replaceValue (Value&, basics::VariantArray* extended);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds/overwrites row with another sent by client
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType setValue (Value&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds/overwrites row with another sent by client
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType setValue (Value&, basics::VariantArray* extended);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds/overwrites row with another sent by client
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType statsValue (StatsValues&);

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns the current flush number
        ////////////////////////////////////////////////////////////////////////////////

        uint64_t getFlushNumber () const {
          return flushNumber;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief increments the current flush number
        ////////////////////////////////////////////////////////////////////////////////

        void incrementFlushNumber () {
          flushNumber++;
        }

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks if a row is not there, deleted, or flushed
        ////////////////////////////////////////////////////////////////////////////////

        bool checkDeleted (hpdf::databases::simple::TR_SimpleTable const*);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief flushs all
        ////////////////////////////////////////////////////////////////////////////////

        void flushAll (datetime_t delay);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks limits for a new row
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType checkAllLimits (size_t keyLength, size_t dataLength);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks limits for an existing row
        ////////////////////////////////////////////////////////////////////////////////

        ResponseType checkDataLimits (size_t dataLength, size_t oldDataLength);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destroys all extended data
        ////////////////////////////////////////////////////////////////////////////////

        void destroyExtended (hpdf::databases::simple::TR_SimpleTable const*);

      private:
        uint64_t flushNumber;           // current flushnumber
        rest::Scheduler* scheduler;     // server scheduler
        rest::Task* flushAllTask;
        StatsValues stats;
        LimitValues limits;

        set<string> extendedKeys;
        vector<hpdf::AttributeDescriptor> extendedKeyValues;
        size_t extendedSize;

        hpdf::AttributeDescriptor* extendedDescription;
    };
  }
}

#endif
