////////////////////////////////////////////////////////////////////////////////
/// @brief simple key-value model
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "SimpleModel.h"

#include <sys/time.h>

#include <Rest/Scheduler.h>
#include <Rest/TimerTask.h>

using namespace triagens::basics;
using namespace triagens::rest;
using namespace triagens::simple;
using namespace triagens::hpdf;
using namespace triagens::hpdf::databases::simple;

// -----------------------------------------------------------------------------
// helper
// -----------------------------------------------------------------------------

namespace {

  // -----------------------------------------------------------------------------
  // global database lock
  // -----------------------------------------------------------------------------

  static ReadWriteLock simpleLock;

  // -----------------------------------------------------------------------------
  // visitor
  // -----------------------------------------------------------------------------

  class ForAllVisitor : public triagens::hpdf::TableVisitor {
    public:
      void visit (Row const* r , size_t) {
        TR_SimpleTable** row = reinterpret_cast<TR_SimpleTable**>(const_cast<Row*>(r));
        result.push_back(*row);
      }

    public:
      vector<TR_SimpleTable const*> result;
  };

  // -----------------------------------------------------------------------------
  // flush task
  // -----------------------------------------------------------------------------

  class FlushTask : public TimerTask {
    public:
      FlushTask (triagens::simple::SimpleModel* model, double timeout)
        : Task("FlushTask"), TimerTask(timeout), model(model) {
      }

    public:
      bool handleTimeout () {
        LOGGER_DEBUG << "timer reached " << seconds;
        model->incrementFlushNumber();
        return true;
      }

    private:
      triagens::simple::SimpleModel* model;
  };

  // -----------------------------------------------------------------------------
  // check, if a deletion is pending
  // -----------------------------------------------------------------------------

  inline bool isDeletionPending (TR_SimpleTable const* buffer) {
    return buffer->A_deletionPending;
  }

  // -----------------------------------------------------------------------------
  // if a deletion is pending, determine if we delete the item
  // -----------------------------------------------------------------------------

  inline bool isDeleted (TR_SimpleTable const* buffer) {
    if (! buffer->A_deletionPending) {
      return false;
    }

    // value is relative
    else if (buffer->A_deletionTime < 2600000.0) {
      if (buffer->A_lastUpdate + buffer->A_deletionTime <= SimpleModel::now())  {
        return true;
      }
    }

    // value is absolute
    else if (buffer->A_deletionTime < SimpleModel::now()) {
      return true;
    }

    return false;
  }

  // -----------------------------------------------------------------------------
  // checks if item expired
  // -----------------------------------------------------------------------------

  inline bool isExpired (TR_SimpleTable const* buffer) {
    if (buffer->A_expireTime <= 0.001) {
      return false;
    }

    // value is relative
    else if (buffer->A_expireTime < 2600000.001) {
      if (buffer->A_created + buffer->A_expireTime < SimpleModel::now()) {
        return true;
      }
    }

    // value is absolute
    else if (buffer->A_expireTime < SimpleModel::now()) {
      return true;
    }

    return false;
  }

  // -----------------------------------------------------------------------------
  // checks if item has been flushed
  // -----------------------------------------------------------------------------

  inline bool isFlushed (TR_SimpleTable const* buffer, uint64_t flushNumber) {
    if (buffer->A_flush_number < flushNumber) {
      return true;
    }

    return false;
  }

  // -----------------------------------------------------------------------------
  // creates a unique number
  // -----------------------------------------------------------------------------

  uint64_t casUniqueValue = 0;
  Mutex accessCasUnique;

  inline uint64_t casUnique () {
    MutexLocker guard(&accessCasUnique);

    return ++casUniqueValue;
  }

  // -----------------------------------------------------------------------------
  // sorts a value list
  // -----------------------------------------------------------------------------

  struct RowSort {
      RowSort (DynamicSort::DynamicSortDescription const& desc) : desc(desc) {
    }

    bool operator() (TR_SimpleTable const* const& left, TR_SimpleTable const* const& right) {
      return DynamicSort::dynamicLess(desc, (void const*) left, (void const*) right);
    }

    DynamicSort::DynamicSortDescription const& desc;
  };
}

namespace triagens {
  namespace simple {

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    SimpleModel::SimpleModel()
      :  flushNumber(0), scheduler(0), flushAllTask(0), extendedSize(0), extendedDescription(0) {

      // initialisation of simple stats
      stats.numKeyValue   = 0;
      stats.totalDataSize = 0;
      stats.totalKeySize  = 0;
      stats.serverStartup = SimpleModel::now();

      // initialisation of any limits in use
      limits.maxNumKeys        = 0;
      limits.maxKeySize        = 0;
      limits.maxValueSize      = 0;
      limits.maxTotalKeySize   = 0;
      limits.maxTotalSize      = 0;
      limits.maxTotalValueSize = 0;
      limits.limitsInUse       = false;
    }



    SimpleModel::~SimpleModel() {
      if (flushAllTask != 0) {
        scheduler->destroyTask(flushAllTask);
      }

      if (extendedDescription != 0) {
        for (AttributeDescriptor* d = extendedDescription;  d->name != 0;  ++d) {
          delete[] d->name;
        }

        delete[] extendedDescription;
      }
    }

    // -----------------------------------------------------------------------------
    // static public methods
    // -----------------------------------------------------------------------------

    datetime_t SimpleModel::now () {
      struct timeval tv;
      gettimeofday(&tv, 0);

      double sec = tv.tv_sec;  // seconds
      double usc = tv.tv_usec; // microseconds

      return sec + usc / 1000000.0;
    }

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    void SimpleModel::setLimitValue(ResponseType response, uint64_t limitValue) {
      WRITE_LOCKER(guard, simpleLock);

      limits.maxKeySize        = (response == KEY_LIMIT_EXCEEDED ? limitValue : limits.maxKeySize);
      limits.maxValueSize      = (response == VALUE_LIMIT_EXCEEDED ? limitValue : limits.maxValueSize);
      limits.maxNumKeys        = (response == NUM_KEYS_LIMIT_EXCEEDED ? limitValue : limits.maxNumKeys);
      limits.maxTotalSize      = (response == TOTAL_SIZE_EXCEEDED ? limitValue : limits.maxTotalSize);
      limits.maxTotalKeySize   = (response == TOTAL_KEY_SIZE_EXCEEDED ? limitValue : limits.maxTotalKeySize);
      limits.maxTotalValueSize = (response == TOTAL_VALUE_SIZE_EXCEEDED ? limitValue : limits.maxTotalValueSize);
      limits.limitsInUse       = (response == GENERIC_LIMIT_EXCEEDED ? (limitValue == 0 ? false : true) : limits.limitsInUse);
    }



    void SimpleModel::setLimitValues(LimitValues& limitValues) {
      WRITE_LOCKER(guard, simpleLock);

      limits.maxNumKeys        = limitValues.maxNumKeys;
      limits.maxKeySize        = limitValues.maxKeySize;
      limits.maxValueSize      = limitValues.maxValueSize;
      limits.maxTotalKeySize   = limitValues.maxTotalKeySize;
      limits.maxTotalSize      = limitValues.maxTotalSize;
      limits.maxTotalValueSize = limitValues.maxTotalValueSize;
      limits.limitsInUse       = limitValues.limitsInUse;
    }



    bool SimpleModel::addExtendedKeyValueDescription (string const& description) {
      vector<string> s = StringUtils::split(description, ':');

      if (s.size() != 2) {
        LOGGER_ERROR << "cannot parse '" << description << "', expecting <name>:<type>";
        return false;
      }

      string const& name = s[0];
      string const& tnam = s[1];

      TypeAttribute type = AT_UNKNOWN;
      size_t rsize = 0;
      size_t align = 0;
      ssize_t size = -1;

      if (tnam == "string") {
        type = AT_STRING;
        rsize = sizeof(char*);
        size = -1;
        align = rsize;
      }
      else if (tnam == "datetime") {
        type = AT_DATETIME;
        rsize = sizeof(datetime_t);
        size = rsize;
        align = rsize;
      }
      else if (tnam == "double") {
        type = AT_DOUBLE;
        rsize = sizeof(double);
        size = rsize;
        align = rsize;
      }
      else if (tnam == "integer") {
        type = AT_INTEGER;
        rsize = sizeof(int32_t);
        size = rsize;
        align = rsize;
      }
      else if (tnam == "boolean") {
        type = AT_BOOLEAN;
        rsize = sizeof(bool);
        size = rsize;
        align = rsize;
      }
      else {
        LOGGER_ERROR << "unknown extended key-value type '" << tnam << "'";
        return false;
      }

      if (extendedKeys.find(name) != extendedKeys.end()) {
        LOGGER_ERROR << "duplicate extended key-value name '" << name << "'";
        return false;
      }

      size_t pos = name.find_first_not_of("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_");

      if (pos != string::npos) {
        LOGGER_ERROR << "name contains illegal character '" << name[pos] << "'";
        return false;
      }

      extendedKeys.insert(name);

      size_t offset = ((extendedSize + align - 1) / align) * align;
      extendedSize = offset + rsize;

      LOGGER_DEBUG << "using extended key-value '" << name
                   << "', type = " << type
                   << ", offset = " << offset
                   << ", size = " << rsize;

      AttributeDescriptor desc;
      desc.name = StringUtils::duplicate(name);
      desc.type = type;
      desc.subtype = 0;
      desc.offset = offset;
      desc.size = size;

      extendedKeyValues.push_back(desc);

      return true;
    }



    void SimpleModel::finalise () {
      if (! extendedKeyValues.empty()) {
        extendedDescription = new AttributeDescriptor[extendedKeyValues.size() + 1];

        size_t p = 0;

        for (vector<AttributeDescriptor>::iterator i = extendedKeyValues.begin();  i != extendedKeyValues.end();  ++i, ++p) {
          AttributeDescriptor& desc = *i;

          extendedDescription[p] = desc;
        }

        extendedDescription[p].name = 0;
      }
    }



    HpdfFilterTree* SimpleModel::dynamicFilterTree (string const& parseString) {
      if (extendedDescription == 0) {
        return 0;
      }

      LOGGER_DEBUG << "got parse string '" << parseString << "' for dynamic filter";

      HpdfFilterTree* filter = new HpdfFilterTree(parseString, extendedDescription);

      if (! filter->validFilterTree) {
        delete filter;
        return 0;
      }

      return filter;
    }



    DynamicSort::DynamicSortDescription SimpleModel::dynamicSortDescription (string const& parseString) {
      if (extendedDescription == 0) {
        return DynamicSort::DynamicSortDescription();
      }

      LOGGER_DEBUG << "got parse string '" << parseString << "' for dynamic sorting";

      try {
        return DynamicSort::lessDescriptor(parseString, extendedDescription);
      }
      catch (TriagensError const& ex) {
        LOGGER_DEBUG << "failed to parse dynamic sorting: " << DIAGNOSTIC_INFORMATION(ex);
      }

      return DynamicSort::DynamicSortDescription();
    }

    // -----------------------------------------------------------------------------
    // public CRUD operations
    // -----------------------------------------------------------------------------

    TR_SimpleTable const* SimpleModel::create (Value const& value) {

      // create a new, empty table row on the stack
      TR_SimpleTable row;
      memset((void*) &row, 0, sizeof(row));

      // fill in the key for the row -- this is what the client sent us
      row.A_identifier = StringUtils::duplicate(value.identifier);

      // fill in the blob of data, including the length -- this is what the client sent
      row.A_data.length = value.data.size();
      row.A_data.data   = StringUtils::duplicate(value.data.c_str(), row.A_data.length);

      // fill in creation date time -- this is now as "unix time"
      row.A_created = now();

      // fill in the last access -- which is 0 since this is a new record
      row.A_lastAccess = 0.0;

      // fill in the last update -- which is now as "unix time"
      row.A_lastUpdate = row.A_created;

      // fill in the expiration date of the item. An expTime of 0 implies that the item WILL NOT auto expire.
      // Non-zero value implies that the item will auto expire when that time in seconds has elapsed
      row.A_expireTime = value.expireTime;

      // fill in the deletion time. The deletion time is the delay before the item is "deleted".
      row.A_deletionTime = 0.0;

      // fill in whether or not a deletion is pending -- obviously false when we add a new item
      row.A_deletionPending = false;

      // flags is a client integer. We just store this.
      row.A_flags = value.flags;

      // cas_unique is a server unique number issued to the item. Used for quasi-transaction handling.
      row.A_cas_unique = casUnique();

      // set the flush number for the case that a flush_all command has been issued
      row.A_flush_number = getFlushNumber();

      // add the row to the table (page)
      TR_SimpleTable* buffer = PT_SimpleTable.add(&row);

      // update the statistics
      stats.numKeyValue   += 1; // increase the number of rows we have in the database
      stats.totalDataSize += value.data.size();
      stats.totalKeySize  += value.identifier.size();

      // and return the newly created row
      return buffer;
    }



    TR_SimpleTable const* SimpleModel::lookup (string const& identifier) {
      QT_HashQuery query;
      PA_HashQuery param;
      PA_HashQuery* paramPtr = &param;

      param.P_identifier_param = const_cast<char*>(identifier.c_str());
      Parameter* ptr = reinterpret_cast<Parameter*>(paramPtr);
      Table* table = query.execute(ptr);
      Row const* row = table->any();

      if (row != 0) {
        TR_HashQuery const* queryRow = reinterpret_cast<TR_HashQuery const*>(row);
        TR_SimpleTable const* result = reinterpret_cast<TR_SimpleTable const*>(queryRow->A_row);

        delete table;
        return result;
      }

      delete table;
      return 0;
    }



    vector<TR_SimpleTable const*> SimpleModel::lookupAll (string const& identifier) {
      ForAllVisitor visitor;

      QT_PrefixQuery query;
      PA_PrefixQuery param;
      PA_PrefixQuery* paramPtr = &param;

      param.P_identifier_param = const_cast<char*>(identifier.c_str());
      Parameter* ptr = reinterpret_cast<Parameter*>(paramPtr);

      Table* table = query.execute(ptr);
      table->forall(&visitor);

      delete table;
      return visitor.result;
    }



    void SimpleModel::updateData (TR_SimpleTable const* row, string const& data) {
      TR_SimpleTable* buffer = const_cast<TR_SimpleTable*>(row);

      size_t oldSize = buffer->A_data.length;
      size_t newSize = data.size();

      // optimize if the blob length hasn't changed
      if (oldSize == newSize) {
        memcpy(buffer->A_data.data, data.c_str(), newSize);
      }

      // general case, buffer is smaller or larger
      else {

        // first delete the existing blob
        StringUtils::destroy(buffer->A_data);

        // fill in the blob of data, including the length
        buffer->A_data.length = newSize;
        buffer->A_data.data   = StringUtils::duplicate(data);
      }

      stats.totalDataSize += newSize - oldSize;

      // fill in the last access -- which is 0 since this is a new record
      buffer->A_lastAccess = now();

      // fill in the last update -- which is now as "unix time"
      buffer->A_lastUpdate = buffer->A_created;

      // fill in whether or not a deletion is pending
      buffer->A_deletionPending = false;
    }



    void SimpleModel::updateCas (TR_SimpleTable const* row) {
      const_cast<TR_SimpleTable*>(row)->A_cas_unique = casUnique();
    }



    void SimpleModel::updateLastAccess (TR_SimpleTable const* row) {
      const_cast<TR_SimpleTable*>(row)->A_lastAccess = now();
    }



    void SimpleModel::updateInformation (TR_SimpleTable const* row, Value const& value) {
      TR_SimpleTable* buffer = const_cast<TR_SimpleTable*>(row);

      // fill in the expiration date of the item. An expireTime of 0 implies that the item WILL NOT auto expire.
      // Non-zero value implies that the item will auto expire when that time in seconds has elapsed
      buffer->A_expireTime = value.expireTime;

      // fill in the deletion time. The deletion time is the delay before the item is "deleted".
      buffer->A_deletionTime = value.deletionTime;

      // flags is a client integer. We just store this.
      buffer->A_flags = value.flags;

      // set the flush number for the case that a flush_all command has been issued
      buffer->A_flush_number = getFlushNumber();
    }



    void SimpleModel::updateExtended (TR_SimpleTable const* row, VariantArray* extended) {
      if (extendedDescription == 0) {
        return;
      }

      if (extended == 0) {
        destroyExtended(row);
        return;
      }

      if (row->A_extended.length == 0) {
        const_cast<TR_SimpleTable*>(row)->A_extended.length = extendedSize;
        const_cast<TR_SimpleTable*>(row)->A_extended.data = new char[extendedSize];
        memset(const_cast<TR_SimpleTable*>(row)->A_extended.data, 0, extendedSize);
      }

      char* ext = row->A_extended.data;

      for (AttributeDescriptor* d = extendedDescription;  d->name != 0;  ++d) {
        VariantObject* value;

        switch (d->type) {
          case AT_BOOLEAN: value = extended->lookupBoolean(d->name);  break;
          case AT_INTEGER: value = extended->lookupInt32(d->name);  break;
          case AT_DOUBLE: value = extended->lookupDouble(d->name);  break;
          case AT_DATETIME: value = extended->lookupDatetime(d->name);  break;

          case AT_STRING: {
            value = extended->lookup(d->name);

            if (value != 0 && ! (value->is<VariantString>() || value->is<VariantNull>())) {
              value = 0;
            }

            break;
          }

          default:
            THROW_INTERNAL_ERROR("missing type");
        }

        if (value == 0) {
          continue;
        }

        size_t off = d->offset;

        switch (d->type) {
          case AT_BOOLEAN:
            (* (uint16_t*) (((char*) ext) + off)) = static_cast<VariantBoolean*>(value)->getValue() ? 1 : 0;
            break;

          case AT_INTEGER:
            * (int32_t*) (((char*) ext) + off) = static_cast<VariantInt32*>(value)->getValue();
            break;

          case AT_DOUBLE:
            * (double*) (((char*) ext) + off) = static_cast<VariantDouble*>(value)->getValue();

          case AT_DATETIME:
            * (datetime_t*) (((char*) ext) + off) = static_cast<VariantDatetime*>(value)->getValue();

          case AT_STRING: {
            char** ptr = (char**) (((char*) ext) + off);

            if (*ptr != 0) {
              StringUtils::destroy(*ptr);
            }

            if (value->is<VariantString>()) {
              * (char**) ptr = StringUtils::duplicate(static_cast<VariantString*>(value)->getValue());
            }

            break;
          }

          default:
            THROW_INTERNAL_ERROR("missing type");
        }

      }
    }



    void SimpleModel::destroyExtended (TR_SimpleTable const* row) {
      if (extendedDescription == 0) {
        return;
      }

      if (row->A_extended.length == 0) {
        return;
      }

      char* ext = row->A_extended.data;

      for (AttributeDescriptor* d = extendedDescription;  d->name != 0;  ++d) {
        size_t off = d->offset;

        if (d->type == AT_STRING) {
          StringUtils::destroy(* (char**) (((char*) ext) + off));
        }
      }
    }



    void SimpleModel::destroy (TR_SimpleTable const* row) {
      size_t oldDataLength = row->A_data.length;
      size_t oldKeyLength  = strlen(row->A_identifier);

      destroyExtended(row);

      if (! PT_SimpleTable.remove(const_cast<TR_SimpleTable*>(row), true)) {
        LOGGER_ERROR << "item could not be deleted";
      }
      else {
        stats.numKeyValue   -= 1;
        stats.totalDataSize -= oldDataLength;
        stats.totalKeySize  -= oldKeyLength;

        LOGGER_TRACE << "item deleted";
      }
    }

    // -----------------------------------------------------------------------------
    // api operations
    // -----------------------------------------------------------------------------

    // -----------------------------------------------------------------------------
    // addValue without extended key-values
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::addValue (Value& value) {
      return addValue(value, 0);
    }

    // -----------------------------------------------------------------------------
    // addValue with extended key-values
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::addValue (Value& value, VariantArray* extended) {
      WRITE_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // A row exists in the database. If item is "deleted" or "flushed", then
      // we can STOMP this item. Notice that an "expired" item will not be
      // STOMPED on. Only a SET command will achieve this.

      if (! checkDeleted(buffer)) {
        return NOT_STORED;
      }

      // check for limits -- we are adding a new key/value pair
      ResponseType rt = checkAllLimits(value.identifier.size(), value.data.size());

      if (rt != OK) {
        return rt;
      }

      // no such row exists in the database, add a new row
      buffer = create(value);

      if (extended != 0) {
        updateExtended(buffer, extended);
      }

      // and return success
      return STORED;
    }

    // -----------------------------------------------------------------------------
    // appendValue
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::appendValue (Value& value) {
      WRITE_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // Check row exists in the database and has either been marked as deleted
      // or flushed. Cannot append anything to an expired row.

      if (checkDeleted(buffer) || isExpired(buffer) || isDeletionPending(buffer)) {
        return NOT_STORED;
      }

      // check for limits
      ResponseType rt = checkDataLimits(value.data.size() + buffer->A_data.length, buffer->A_data.length);

      if (rt != OK) {
        return rt;
      }

      // add a blob to the end of an existing blob
      size_t newLength = buffer->A_data.length + value.data.size();

      string newData;
      newData.reserve(newLength);
      newData.append(buffer->A_data.data, buffer->A_data.length);
      newData.append(value.data.c_str(), value.data.size());

      // update value
      value.data = newData;

      // update data
      updateData(buffer, newData);
      updateCas(buffer);

      return STORED;
    }

    // -----------------------------------------------------------------------------
    // casValue
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::casValue (Value& value) {
      WRITE_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // Row exists in the database and has either been marked as deleted or flushed.
      // Cannot check and set (CAS) to a deleted or flushed  row. Obviously it has
      // changed!

      if (checkDeleted(buffer)) {
        return NOT_STORED;
      }

      // If Row exists in the database and has expired, then we DO ALLOW access here if
      // the cas_unique number matches. Same for depPending.

      // has there been a change since last access?
      if (buffer->A_cas_unique != value.cas) {
        return EXISTS;
      }

      // check the limits
      ResponseType rt = checkDataLimits(value.data.size(), buffer->A_data.length);

      if (rt != OK) {
        return rt;
      }

      // update data, cas, and information
      updateData(buffer, value.data);
      updateCas(buffer);
      updateInformation(buffer, value);

      return STORED;
    }

    // -----------------------------------------------------------------------------
    // decrValue
    //
    // Assumes that the data blob is an integer and attempts to reduce its value by
    // the amount passed on the command line. We do not check limits here. Nor do we
    // check whether or not the data blob represents a valid 64 bit integer.
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::decrValue (Value& value) {
      WRITE_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // Row exists in the database and has either been marked as deleted or flushed.
      // Cannot decrement anything to a deleted row.

      if (checkDeleted(buffer)) {
        return NOT_STORED;
      }

      // Row exists in the database and has expired. Do nothing. Same for deletionPending.
      if (isExpired(buffer) || isDeletionPending(buffer)) {
        return NOT_FOUND;
      }

      // treat the blob as a decimal representation of an integer
      uint64_t intValue = StringUtils::uint64(buffer->A_data.data,buffer->A_data.length);

      if (intValue < value.incrDecr) {
        intValue = 0;
      }
      else {
        intValue -= value.incrDecr;
      }

      string valueStr = StringUtils::itoa(intValue);

      ResponseType rt = checkDataLimits(valueStr.size(), buffer->A_data.length);

      if (rt != OK) {
        return rt;
      }

      // update the data
      updateData(buffer, valueStr);
      updateCas(buffer);

      // return the new value so that this can be passed back to the client
      value.data = valueStr;
      value.incrDecr = intValue;

      // everything ok
      return OK;
    }

    // -----------------------------------------------------------------------------
    // deleteValue
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::deleteValue (Value& value) {
      WRITE_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // Row exists in the database and has either been marked as deleted or flushed.
      // Cannot deleted if it already is marked for deletion.

      if (checkDeleted(buffer)) {
        return NOT_FOUND;
      }

      // Row exists in the database. Could be expired - no problem we still mark
      // as deleted.  Could be marked as deleted - no problem re-mark as deleted
      // and possibly extend deletion time.

      // perhaps we are required to delete record immediately
      if  (value.deletionTime == 0)  {
        destroy(buffer);
      }

      // deleting is pending
      else {
        updateInformation(buffer, value);
        updateCas(buffer);
        const_cast<TR_SimpleTable*>(buffer)->A_deletionPending = true;
      }

      return OK;
    }

    // -----------------------------------------------------------------------------
    // deleteValue
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::flushAllValue (Value& value) {
      flushAll(value.flushDelay);
      return OK;
    }

    // -----------------------------------------------------------------------------
    // getValue
    //
    // Gets the value without the extended key-value information
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::getValue (Value& value) {
      READ_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // if a deletion is pending, we do NOT delete the item here - too slow
      // either later for garbage collection, or if some modification is made to the item.
      // At the moment we simply indicate item does not exist

      if (buffer == 0 || isDeletionPending(buffer) || isExpired(buffer) || isFlushed(buffer, flushNumber)) {
        return NOT_FOUND;
      }

      // update the last time the record was accessed
      updateLastAccess(buffer);

      // fill in the value
      value.creationTime = buffer->A_created;
      value.expireTime   = buffer->A_expireTime;
      value.flags        = buffer->A_flags;
      value.cas          = buffer->A_cas_unique;
      value.data         = string(buffer->A_data.data, buffer->A_data.length);

      return OK;
    }

    // -----------------------------------------------------------------------------
    // getValue
    //
    // Gets the value with the extended key-value information
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::getValue (Value& value, VariantObject*& extended) {
      READ_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // if a deletion is pending, we do NOT delete the item here - too slow
      // either later for garbage collection, or if some modification is made to the item.
      // At the moment we simply indicate item does not exist

      if (buffer == 0 || isDeletionPending(buffer) || isExpired(buffer) || isFlushed(buffer, flushNumber)) {
        return NOT_FOUND;
      }

      // update the last time the record was accessed
      updateLastAccess(buffer);

      // fill in the value
      value.creationTime = buffer->A_created;
      value.expireTime   = buffer->A_expireTime;
      value.flags        = buffer->A_flags;
      value.cas          = buffer->A_cas_unique;
      value.data         = string(buffer->A_data.data,buffer->A_data.length);

      // and the extended key/values
      if (buffer->A_extended.length == 0) {
        extended = 0;
      }
      else {
        extended = new VariantRow(buffer->A_extended.data, extendedDescription);
      }

      return OK;
    }

    // -----------------------------------------------------------------------------
    // incrValue
    //
    // Assumes that the data blob is an integer and attempts to increase its value
    // by the amount passed on the command line. We do not check limits here.
    // -----------------------------------------------------------------------------

    ResponseType  SimpleModel::incrValue (Value& value) {
      WRITE_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // Row exists in the database and has either been marked as deleted or flushed.
      // Cannot increment anything to a deleted row.

      if (checkDeleted(buffer)) {
        return NOT_STORED;
      }

      // Row exists in the database and has expired. Do nothing. Same for deletionPending
      if (isExpired(buffer) || isDeletionPending(buffer)) {
        return NOT_FOUND;
      }

      // treat the blob as a decimal representation of an integer
      uint64_t intValue = StringUtils::uint64(buffer->A_data.data,buffer->A_data.length) + value.incrDecr;
      string valueStr = StringUtils::itoa(intValue);

      ResponseType rt = checkDataLimits(valueStr.size(), buffer->A_data.length);

      if (rt != OK) {
        return rt;
      }

      // update the data
      updateData(buffer, valueStr);
      updateCas(buffer);

      // return the new value so that this can be passed back to the client
      value.data = valueStr;
      value.incrDecr = intValue;

      return OK;
    }

    // -----------------------------------------------------------------------------
    // prefixGetValue without filter
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::prefixGetValue (Value& value, vector<Value>& values) {
      READ_LOCKER(guard, simpleLock);

      vector<TR_SimpleTable const*> all = lookupAll(value.identifier);

      if (all.empty()) {
        return NOT_FOUND;
      }

      for (vector<TR_SimpleTable const*>::const_iterator k = all.begin();  k != all.end();  ++k) {
        TR_SimpleTable const* buffer = *k;

        if (isDeletionPending(buffer) || isExpired(buffer) || isFlushed(buffer, flushNumber)) {
          continue;
        }

        Value value;

        value.identifier   = string(buffer->A_identifier);
        value.creationTime = buffer->A_created;
        value.expireTime   = buffer->A_expireTime;
        value.flags        = buffer->A_flags;
        value.cas          = buffer->A_cas_unique;
        value.data         = string(buffer->A_data.data, buffer->A_data.length);

        values.push_back(value);
      }

      return OK;
    }

    // -----------------------------------------------------------------------------
    // prefixGetValue with filter and sorting
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::prefixGetValue (Value& value,
                                              HpdfFilterTree* filter,
                                              DynamicSort::DynamicSortDescription const& sorting,
                                              vector<Value>& values) {
      READ_LOCKER(guard, simpleLock);

      vector<TR_SimpleTable const*> all = lookupAll(value.identifier);

      if (all.empty()) {
        return NOT_FOUND;
      }

      // filter returned rows
      vector<TR_SimpleTable const*> filtered;

      for (vector<TR_SimpleTable const*>::const_iterator k = all.begin();  k != all.end();  ++k) {
        TR_SimpleTable const* buffer = *k;

        if (isDeletionPending(buffer) || isExpired(buffer) || isFlushed(buffer, flushNumber)) {
          continue;
        }

        if (filter != 0) {
          if (buffer->A_extended.length == 0) {
            continue;
          }

          bool f = DynamicCondition::dynamicFilter(*filter, (void const*) buffer->A_extended.data);

          if (! f) {
            continue;
          }
        }

        filtered.push_back(buffer);
      }

      // sort rows
      if (! sorting.empty() && extendedDescription != 0) {
        sort(filtered.begin(), filtered.end(), RowSort(sorting));
      }

      // and convert to values
      for (vector<TR_SimpleTable const*>::const_iterator k = filtered.begin();  k != filtered.end();  ++k) {
        TR_SimpleTable const* buffer = *k;

        Value value;

        value.identifier   = string(buffer->A_identifier);
        value.creationTime = buffer->A_created;
        value.expireTime   = buffer->A_expireTime;
        value.flags        = buffer->A_flags;
        value.cas          = buffer->A_cas_unique;
        value.data         = string(buffer->A_data.data, buffer->A_data.length);

        values.push_back(value);
      }

      return OK;
    }

    // -----------------------------------------------------------------------------
    // prependValue
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::prependValue (Value& value) {
      WRITE_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // Row exists in the database and has either been marked as deleted or flushed.
      // Cannot prepend anything to an deleted row.

      if (checkDeleted(buffer) || isExpired(buffer) || isDeletionPending(buffer)) {
        return NOT_STORED;
      }

      // check for limits
      ResponseType rt = checkDataLimits(value.data.size() + buffer->A_data.length, buffer->A_data.length);

      if (rt != OK) {
        return rt;
      }

      // add a blob to the beginning of an existing blob
      size_t newLength = buffer->A_data.length + value.data.size();

      string newData;
      newData.reserve(newLength);
      newData.append(value.data.c_str(), value.data.size());
      newData.append(buffer->A_data.data, buffer->A_data.length);

      // update value
      value.data = newData;

      // update data
      updateData(buffer, newData);
      updateCas(buffer);

      return STORED;
    }

    // -----------------------------------------------------------------------------
    // preplaceValue without extended key-values
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::replaceValue (Value& value) {
      return replaceValue(value, 0);
    }

    // -----------------------------------------------------------------------------
    // preplaceValue with extended key-values
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::replaceValue (Value& value, VariantArray* extended) {
      WRITE_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // Row exists in the database and has either been marked as deleted or flushed.
      // Cannot replace anything in an deleted row.

      if (checkDeleted(buffer)) {
        return NOT_STORED;
      }

      // Row exists in the database. If deletionPending do not replace
      if (isDeletionPending(buffer)) {
        return NOT_STORED;
      }

      // check for limits
      ResponseType rt = checkDataLimits(value.data.size(), buffer->A_data.length);

      if (rt != OK) {
        return rt;
      }

      // Row exists in the database, stomp on it. Even if it has expired!
      updateData(buffer, value.data);
      updateCas(buffer);
      updateInformation(buffer, value);

      if (extended != 0) {
        updateExtended(buffer, extended);
      }

      return STORED;
    }

    // -----------------------------------------------------------------------------
    // setValue without extended key-values
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::setValue (Value& value) {
      return setValue(value, 0);
    }

    // -----------------------------------------------------------------------------
    // setValue with extended key-values
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::setValue (Value& value, VariantArray* extended) {
      WRITE_LOCKER(guard, simpleLock);

      TR_SimpleTable const* buffer = lookup(value.identifier);

      // row exists in the database
      if (buffer != 0) {

        // a set command always stomps on the item irrespective of whether or not a deletion is pending.
        ResponseType rt = checkDataLimits(value.data.size(), buffer->A_data.length);

        if (rt != OK) {
          return rt;
        }

        updateData(buffer, value.data);
        updateCas(buffer);
        updateInformation(buffer, value);

        if (extended != 0) {
          updateExtended(buffer, extended);
        }
      }

      // row doesn't exists in the database
      else {

        // check for limits -- we are adding a new key/value pair
        ResponseType rt = checkAllLimits(value.identifier.size(), value.data.size());

        if (rt != OK) {
          return rt;
        }

        // no such row exists in the database, add a new row
        buffer = create(value);

        if (extended != 0) {
          updateExtended(buffer, extended);
        }
      }

      return STORED;
    }

    // -----------------------------------------------------------------------------
    // statsValue
    // -----------------------------------------------------------------------------

    ResponseType SimpleModel::statsValue(StatsValues& value) {
      value.numKeyValue   = this->stats.numKeyValue;
      value.serverStartup = this->stats.serverStartup;
      value.totalDataSize = this->stats.totalDataSize;
      value.totalKeySize  = this->stats.totalKeySize;

      return OK;
    }

    // -----------------------------------------------------------------------------
    // protected methods
    // -----------------------------------------------------------------------------

    bool SimpleModel::checkDeleted (TR_SimpleTable const* row) {
      if (row == 0) {
        return true;
      }

      if (isDeleted(row) || isFlushed(row, flushNumber)) {
        destroy(row);
        return true;
      }

      return false;
    }



    void SimpleModel::flushAll (datetime_t delay) {
      WRITE_LOCKER(guard, simpleLock);

      // if a delayed flush all command is in operation, remove it
      if (flushAllTask != 0) {
        scheduler->destroyTask(flushAllTask);
      }

      // if the delay sent is 0 seconds, then increment the flush counter and return
      if (delay <= 0.0) {
        flushNumber++;
        return;
      }

      // delay sent
      flushAllTask = new FlushTask(this, delay);
      scheduler->registerTask(flushAllTask);

    }



    ResponseType SimpleModel::checkAllLimits (size_t keyLength, size_t dataLength) {
      if (limits.limitsInUse) {

        // check atomic limits
        if (limits.maxNumKeys != 0 && limits.maxNumKeys <= stats.numKeyValue) {
          return NUM_KEYS_LIMIT_EXCEEDED;
        }

        if (limits.maxValueSize != 0 && limits.maxValueSize < dataLength) {
          return VALUE_LIMIT_EXCEEDED;
        }

        if (limits.maxKeySize != 0 && limits.maxKeySize < keyLength) {
          return KEY_LIMIT_EXCEEDED;
        }

        // check aggregated limits
        if (limits.maxTotalKeySize != 0 && limits.maxTotalKeySize < stats.totalKeySize + keyLength) {
          return TOTAL_KEY_SIZE_EXCEEDED;
        }

        if (limits.maxTotalValueSize != 0 && limits.maxTotalValueSize < stats.totalDataSize + dataLength) {
          return TOTAL_VALUE_SIZE_EXCEEDED;
        }

        if ( (limits.maxTotalSize != 0) && (limits.maxTotalSize < stats.totalKeySize + keyLength
                                                                  + stats.totalDataSize + dataLength) ) {
          return TOTAL_SIZE_EXCEEDED;
        }
      }

      return OK;
    }



    ResponseType SimpleModel::checkDataLimits (size_t dataLength, size_t oldDataLength) {
      if (limits.limitsInUse) {

        // check atomic limits
        if (limits.maxValueSize != 0 && limits.maxValueSize < dataLength) {
          return VALUE_LIMIT_EXCEEDED;
        }

        // check aggregated limits
        size_t newTotalDataSize = stats.totalDataSize - oldDataLength + dataLength;

        if (limits.maxTotalValueSize != 0 && limits.maxTotalValueSize < newTotalDataSize) {
          return TOTAL_VALUE_SIZE_EXCEEDED;
        }

        if (limits.maxTotalSize != 0 && limits.maxTotalSize < stats.totalKeySize + newTotalDataSize) {
          return TOTAL_SIZE_EXCEEDED;
        }
      }

      return OK;
    }
  }
}
