////////////////////////////////////////////////////////////////////////////////
/// @brief scheduler thread
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Martin Schoenert
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "SchedulerThread.h"

#include <Basics/Logger.h>
#include <Rest/Task.h>

using namespace triagens::basics;

namespace triagens {
  namespace rest {

    // -----------------------------------------------------------------------------
    // private helper functions and classes
    // -----------------------------------------------------------------------------

    namespace {
      void wakerCallback (struct ev_loop* loop, ev_async*, int) {
#if defined(DEBUG_SCHEDULER_THREAD)
        LOGGER_TRACE << "caught scheduler thread notification";
#endif

        ev_unloop((struct ev_loop*) loop, EVUNLOOP_ALL);
      }
    }

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    SchedulerThread::SchedulerThread (Scheduler*, bool defaultLoop, int backend)
      : Thread("SchedulerThread"),
        defaultLoop(defaultLoop),
        backend(backend),
        stopping(0),
        loop(0),
        hasWork(0) {

      // construct the loop
      if (defaultLoop) {
        loop = (event_loop_t*) ev_default_loop(backend);
      }
      else {
        loop = (event_loop_t*) ev_loop_new(backend);
      }

      // create a waker
      ev_async* w = &waker;
      ev_async_init(w, wakerCallback);
      ev_async_start((struct ev_loop*) loop, w);

      // allow cancelation
      allowAsynchronousCancelation();
    }

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    void SchedulerThread::beginShutdown () {

      // same thread, in this case it does not matter if we are inside the loop
      if (threadId() == currentThreadId()) {

        if (stopping == 0) {
          LOGGER_DEBUG << "beginning shutdown sequence (self, " << threadId() << ")";

          stopping = 1;
          ev_unloop((struct ev_loop*) loop, EVUNLOOP_ALL);
        }
      }

      // different thread, be careful and send a signal
      else {
        if (stopping == 0) {
          LOGGER_DEBUG << "beginning shutdown sequence (" << threadId() << ")";

          stopping = 1;
          ev_async_send((struct ev_loop*) loop, &waker);
        }
      }
    }



    void SchedulerThread::registerTask (Scheduler* scheduler, Task* task) {

      // same thread, in this case it does not matter if we are inside the loop
      if (threadId() == currentThreadId()) {
        setupTask(task, scheduler, loop);
        ev_async_send((struct ev_loop*) loop, &waker);
      }

      // different thread, be careful - we have to stop the event loop
      else {

        // put the register request unto the queue
        if (! queueLock.lock()) {
          LOGGER_FATAL << "cannot lock in " << __FILE__ << "@" << __LINE__;
          exit(EXIT_FAILURE);
        }

        Work w(SETUP, scheduler, task);
        queue.push_back(w);
        hasWork = 1;

        ev_async_send((struct ev_loop*) loop, &waker);

        if (! queueLock.unlock()) {
          LOGGER_FATAL << "cannot unlock in " << __FILE__ << "@" << __LINE__;
          exit(EXIT_FAILURE);
        }
      }
    }



    void SchedulerThread::unregisterTask (Task* task) {
      deactivateTask(task);

      // same thread, in this case it does not matter if we are inside the loop
      if (threadId() == currentThreadId()) {
        cleanupTask(task);
        ev_async_send((struct ev_loop*) loop, &waker);
      }

      // different thread, be careful - we have to stop the event loop
      else {

        // put the unregister request unto the queue
        if (! queueLock.lock()) {
          LOGGER_FATAL << "cannot lock in " << __FILE__ << "@" << __LINE__;
          exit(EXIT_FAILURE);
        }

        Work w(CLEANUP, 0, task);
        queue.push_back(w);
        hasWork = 1;

        ev_async_send((struct ev_loop*) loop, &waker);

        if (! queueLock.unlock()) {
          LOGGER_FATAL << "cannot unlock in " << __FILE__ << "@" << __LINE__;
          exit(EXIT_FAILURE);
        }
      }
    }



    void SchedulerThread::destroyTask (Task* task) {
      deactivateTask(task);

      // same thread, in this case it does not matter if we are inside the loop
      if (threadId() == currentThreadId()) {
        cleanupTask(task);
        deleteTask(task);
        ev_async_send((struct ev_loop*) loop, &waker);
      }

      // different thread, be careful - we have to stop the event loop
      else {

        // put the unregister request unto the queue
        if (! queueLock.lock()) {
          LOGGER_FATAL << "cannot lock in " << __FILE__ << "@" << __LINE__;
          exit(EXIT_FAILURE);
        }

        Work w(DESTROY, 0, task);
        queue.push_back(w);
        hasWork = 1;

        ev_async_send((struct ev_loop*) loop, &waker);

        if (! queueLock.unlock()) {
          LOGGER_FATAL << "cannot unlock in " << __FILE__ << "@" << __LINE__;
          exit(EXIT_FAILURE);
        }
      }
    }

    // -----------------------------------------------------------------------------
    // Thread methods
    // -----------------------------------------------------------------------------

    void SchedulerThread::run () {
      LOGGER_TRACE << "scheduler thread started (" << threadId() << ")";

      if (defaultLoop) {
        sigset_t all;
        sigemptyset(&all);

        pthread_sigmask(SIG_SETMASK, &all, 0);
      }

      while (stopping == 0) {
        try {
          ev_loop((struct ev_loop*) loop, 0);
        }
        catch (...) {
#ifdef HAVE_POSIX_THREADS
          if (stopping != 0) {
            LOGGER_WARNING << "caught cancellation exception during work";
            throw;
          }
#endif

          LOGGER_WARNING << "caught exception from ev_loop";
        }

#if defined(DEBUG_SCHEDULER_THREAD)
        LOGGER_TRACE << "left scheduler loop " << threadId();
#endif

        if (hasWork != 0) {
          if (! queueLock.lock()) {
            LOGGER_FATAL << "cannot lock in " << __FILE__ << "@" << __LINE__;
            exit(EXIT_FAILURE);
          }

          while (! queue.empty()) {
            Work w = queue.front();
            queue.pop_front();

            if (! queueLock.unlock()) {
              LOGGER_FATAL << "cannot unlock in " << __FILE__ << "@" << __LINE__;
              exit(EXIT_FAILURE);
            }

            switch (w.work) {
              case CLEANUP:
                cleanupTask(w.task);
                break;

              case SETUP:
                setupTask(w.task, w.scheduler, loop);
                break;

              case DESTROY:
                cleanupTask(w.task);
                deleteTask(w.task);
                break;
            }

            if (! queueLock.lock()) {
              LOGGER_FATAL << "cannot lock in " << __FILE__ << "@" << __LINE__;
              exit(EXIT_FAILURE);
            }
          }

          hasWork = 0;

          if (! queueLock.unlock()) {
            LOGGER_FATAL << "cannot unlock in " << __FILE__ << "@" << __LINE__;
            exit(EXIT_FAILURE);
          }
        }
      }

      LOGGER_TRACE << "scheduler thread stopped (" << threadId() << ")";

      if (! defaultLoop) {
        ev_loop_destroy((struct ev_loop*) loop);
      }
    }
  }
}
