////////////////////////////////////////////////////////////////////////////////
/// @brief value request handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Achim Brandt
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "RestValueHandler.h"

#include <Basics/StringBuffer.h>

using namespace triagens::basics;
using namespace triagens::rest;

namespace triagens {
  namespace simple {

    // -----------------------------------------------------------------------------
    // Handler methods
    // -----------------------------------------------------------------------------

    HttpHandler::status_e RestValueHandler::execute () {

      // check for illegal request
      if (request->requestType() == rest::HttpRequest::HTTP_REQUEST_ILLEGAL) {
        generateError("method not supported");
        return HANDLER_DONE;
      }

      computeKey();

      // check if empty was sent
      if (key.empty()) {
        generateError("empty key");
        return HANDLER_DONE;
      }

      switch (request->requestType()) {
        case rest::HttpRequest::HTTP_REQUEST_GET: return executeGet();
        case rest::HttpRequest::HTTP_REQUEST_POST: return executePost();
        case rest::HttpRequest::HTTP_REQUEST_PUT: return executePut();
        case rest::HttpRequest::HTTP_REQUEST_DELETE: return executeDelete();

        default:
          generateError("method not supported");
          return HANDLER_DONE;
      }
    }

    // -----------------------------------------------------------------------------
    // private methods
    // -----------------------------------------------------------------------------

    void RestValueHandler::computeKey () {
      basics::StringBuffer sb;
      sb.initialise();

      bool first = true;

      for (vector<string>::const_iterator i = request->suffix().begin();
           i != request->suffix().end();
           ++i) {

        if (first) {
          first = false;
        }
        else {
          sb.appendChar('/');
        }
        sb.appendText(*i);
      }

      if (sb.length() == 0) {
        return;
      }

      key = string(sb.c_str(), sb.length());

      sb.free();
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // this request is sent ONE key and attempts to extract one row and return it
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    Handler::status_e RestValueHandler::executeGet () {
      statistics = STAT_REST_VALUE_GET;

      Value value;
      value.identifier = key;

      VariantObject* extended = 0;
      ResponseType result = model->getValue(value, extended);

      if (result == OK) {
        response = new rest::HttpResponse(rest::HttpResponse::OK);

        // add the body
        response->body().appendText(value.data.c_str(), value.data.size());

        // add some header fields
        response->setHeader(VOC_HEADER_FIELD_CREATED,
                            StringUtils::formatDatetime(value.creationTime) + "Z");

        response->setHeader(VOC_HEADER_FIELD_FLAGS,
                            StringUtils::itoa(value.flags));

        if (value.expireTime > 0.0) {
          response->setHeader(VOC_HEADER_FIELD_EXPIRES,
                              StringUtils::formatDatetime(value.expireTime)+ "Z");
        }

        // add the extended key-values
        if (extended != 0) {
          StringBuffer sb;
          sb.initialise();

          string contentType;
          bool ok = OutputGenerator::output("json", sb, extended, contentType);
          delete extended;

          if (! ok ) {
            sb.free();
            generateError("cannot generate output for extended key-values");
            return HANDLER_DONE;
          }

          response->setHeader(VOC_HEADER_FIELD_EXTENDED, sb.c_str());
          sb.free();
        }

        // add the content type
        response->setContentType("application/octet-stream");
      }
      else {
        generateError("key not found");
      }

      return HANDLER_DONE;
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // this request is sent ONE key and adds a row if it does not already exist
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    Handler::status_e RestValueHandler::executePost () {
      statistics = STAT_REST_VALUE_POST;

      Value value;

      value.identifier = key;
      value.data  = string(request->body().c_str(), request->body().size());
      value.flags = uint64HeaderField(VOC_HEADER_FIELD_FLAGS);

      RestDefaultHandler::ParseErrorType parseResult = datetimeHeaderField(VOC_HEADER_FIELD_EXPIRES, value.expireTime);

      if (parseResult == PARSE_ERROR) {
        generateError("invalid expiration date time");
        return HANDLER_DONE;
      }

      VariantArray* extended;
      parseResult = extendedHeaderField(VOC_HEADER_FIELD_EXTENDED, extended);

      if (parseResult == PARSE_ERROR) {
        generateError("invalid extended key-value");
        return HANDLER_DONE;
      }

      ResponseType result = model->addValue(value, extended);
      delete extended;

      if (result == STORED) {
        generateStatus("stored");
      }
      else if (result == NOT_STORED) {
        generateError("use 'PUT /value/<key>' to overwrite value");
      }
      else {
        generateError("not stored");
      }

      return HANDLER_DONE;
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // this request is sent ONE key and adds a new row or replaces an existing row
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    Handler::status_e RestValueHandler::executePut () {
      statistics = STAT_REST_VALUE_PUT;

      Value value;

      value.identifier = key;
      value.data  = string(request->body().c_str(),request->body().size());
      value.flags = uint64HeaderField(VOC_HEADER_FIELD_FLAGS);

      RestDefaultHandler::ParseErrorType parseResult = datetimeHeaderField(VOC_HEADER_FIELD_EXPIRES,value.expireTime);

      if (parseResult == PARSE_ERROR) {
        generateError("invalid expiration date time");
        return HANDLER_DONE;
      }

      VariantArray* extended;
      parseResult = extendedHeaderField(VOC_HEADER_FIELD_EXTENDED, extended);

      if (parseResult == PARSE_ERROR) {
        generateError("invalid extended key-value");
        return HANDLER_DONE;
      }

      // create flag sent indicating only replace
      ResponseType result;

      if (request->value("create") == "1") {
        result = model->setValue(value, extended);
      }
      else {
        result = model->replaceValue(value, extended);
      }

      delete extended;

      if (result == STORED) {
        generateStatus("stored");
      }
      else {
        generateError("not stored");
      }

      return HANDLER_DONE;
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // this request is sent ONE key and attempts to mark a record as DELETED
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    Handler::status_e RestValueHandler::executeDelete () {
      statistics = STAT_REST_VALUE_DELETE;

      Value value;
      value.identifier = key;

      RestDefaultHandler::ParseErrorType parseResult = datetimeHeaderField(VOC_HEADER_FIELD_DELETE_TIME, value.deletionTime);

      LOGGER_TRACE << "deletion time " << value.deletionTime;

      if (parseResult == PARSE_ERROR) {
        generateError("invalid deletion date time");
        return HANDLER_DONE;
      }

      ResponseType result = model->deleteValue(value);

      if (result == OK) {
        generateStatus("deleted");
      }
      else {
        generateError("key not found");
      }

      return HANDLER_DONE;
    }
  }
}
