////////////////////////////////////////////////////////////////////////////////
/// @brief default request handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Achim Brandt
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef SIMPLEVOC_RESTHANDLER_REST_DEFAULT_HANDLER_H
#define SIMPLEVOC_RESTHANDLER_REST_DEFAULT_HANDLER_H 1

#include <Basics/Common.h>

#include <Basics/Timing.h>
#include <Basics/VariantArray.h>
#include <Rest/HttpResponse.h>
#include <Rest/OutputGenerator.h>

#include "Basics/Exceptions.h"
#include "Basics/StringUtils.h"
#include "Rest/HttpHandler.h"
#include "Rest/HttpRequest.h"
#include "Rest/HttpResponse.h"
#include "SimpleVoc/SimpleModel.h"
#include "SimpleVoc/SimpleVocFigures.h"

namespace triagens {
  namespace simple {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief default request handler
    ////////////////////////////////////////////////////////////////////////////////

    class RestDefaultHandler : public rest::HttpHandler {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief called failed with an unexpected error
        ////////////////////////////////////////////////////////////////////////////////

        static basics::LoggerData::Extra LOG_FAIL;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief called failed with an expected error
        ////////////////////////////////////////////////////////////////////////////////

        static basics::LoggerData::Extra LOG_ERROR;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief called succeeded
        ////////////////////////////////////////////////////////////////////////////////

        static basics::LoggerData::Extra LOG_PASS;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        RestDefaultHandler (rest::HttpRequest* request, void* model, size_t statistics)
          : rest::HttpHandler(request),
            model((SimpleModel*) model),
            statistics(statistics),
            timing(basics::Timing::TI_WALLCLOCK) {
          info << basics::LoggerData::Task("REST " + request->requestPath());
          LOGGER_REQUEST_IN_START_I(info);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructor
        ////////////////////////////////////////////////////////////////////////////////

        ~RestDefaultHandler () {
          double time = timing.time() / 1000000.0;
          addFigure<SimpleVocStatistics::runtimesAccessor>(statistics, time);

          LOGGER_REQUEST_IN_END_I(info) << infoResult;
        }

      public:

        enum ParseErrorType {
            PARSE_OK,
            PARSE_NO_FIELD,
            PARSE_ERROR
        };

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        bool isDirect () {
          return true;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        void handleError (basics::TriagensError const&);

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generates an result
        ////////////////////////////////////////////////////////////////////////////////

        void generateResult (basics::VariantArray* result, bool html);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generates an error
        ////////////////////////////////////////////////////////////////////////////////

        void generateError (string const& message);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief generates a status
        ////////////////////////////////////////////////////////////////////////////////

        void generateStatus (string const& message);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns a uint64 header field
        ////////////////////////////////////////////////////////////////////////////////

        uint64_t uint64HeaderField (string const& name);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns a datetime header field
        ////////////////////////////////////////////////////////////////////////////////

        ParseErrorType datetimeHeaderField (string const& name, datetime_t& dt);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns a json field
        ////////////////////////////////////////////////////////////////////////////////

        ParseErrorType extendedHeaderField (string const& name, basics::VariantArray*& extended);

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the underlying rest interface
        ////////////////////////////////////////////////////////////////////////////////

        SimpleModel* model;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief type
        ////////////////////////////////////////////////////////////////////////////////

        size_t statistics;

      private:
        basics::Timing timing;
        basics::LoggerTiming info;
        basics::LoggerData::Extra infoResult;
    };
  }
}

#endif
