////////////////////////////////////////////////////////////////////////////////
/// @brief default request handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Achim Brandt
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "RestDefaultHandler.h"

#include <Rest/InputParser.h>

using namespace triagens::basics;
using namespace triagens::rest;

namespace triagens {
  namespace simple {

    // -----------------------------------------------------------------------------
    // constants
    // -----------------------------------------------------------------------------

    basics::LoggerData::Extra RestDefaultHandler::LOG_FAIL(0, "RES-FAIL");
    basics::LoggerData::Extra RestDefaultHandler::LOG_ERROR(0, "RES-ERR");
    basics::LoggerData::Extra RestDefaultHandler::LOG_PASS(0, "RES-OK");

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    void RestDefaultHandler::handleError (basics::TriagensError const&) {
      if (response != 0) {
        delete response;
      }

      response = new HttpResponse(HttpResponse::BAD);
      infoResult = LOG_FAIL;
    }

    // -----------------------------------------------------------------------------
    // protected methods
    // -----------------------------------------------------------------------------

    void RestDefaultHandler::generateResult (VariantArray* result, bool html) {
      response = new HttpResponse(HttpResponse::OK);

      string contentType;
      bool ok = OutputGenerator::output(html ? "html" : "json",
                                              response->body(),
                                              result,
                                              contentType);

      if (ok) {
        response->setContentType(contentType);
        infoResult = LOG_PASS;
      }
      else {
        delete response;

        generateError("cannot generate output");
      }

      delete result;
    }



    void RestDefaultHandler::generateError (string const& message) {
      response = new HttpResponse(HttpResponse::BAD);

      response->body().appendText("{ \"error\" : true, \"message\" : \"" );
      response->body().appendText(StringUtils::escapeUnicode(message));
      response->body().appendText("\" }");

      response->setContentType("application/json");

      infoResult = LOG_ERROR;
    }



    void RestDefaultHandler::generateStatus (string const& message) {
      response = new HttpResponse(HttpResponse::OK);

      response->body().appendText("{ \"error\" : false, \"message\" : \"" );
      response->body().appendText(StringUtils::escapeUnicode(message));
      response->body().appendText("\" }");

      response->setContentType("application/json");

      infoResult = LOG_PASS;
    }



    uint64_t RestDefaultHandler::uint64HeaderField (string const& name) {
      bool found = false;
      string value = request->header(name, found);

      if (!found) {
        return 0;
      }

      return basics::StringUtils::uint64(value);
    }



    RestDefaultHandler::ParseErrorType RestDefaultHandler::datetimeHeaderField (string const& name, datetime_t& dt) {
      bool found = false;
      string value = request->header(name, found);
      dt = 0.0;

      if (! found) {
        return PARSE_NO_FIELD;
      }

      try {
        dt = StringUtils::datetime("YYYY-MM-DDTHH:MI:SSZ", value);
        return PARSE_OK;
      }
      catch (const triagens::basics::ParseError& pe) {
        return PARSE_ERROR;
      }
    }



    RestDefaultHandler::ParseErrorType RestDefaultHandler::extendedHeaderField (string const& name, VariantArray*& extended) {
      extended = 0;

      bool found = false;
      string value = request->header(name, found);

      if (! found) {
        return PARSE_NO_FIELD;
      }

      extended = InputParser::jsonArray(value);

      if (extended == 0) {
        return PARSE_ERROR;
      }

      return PARSE_OK;
    }
  }
}
