////////////////////////////////////////////////////////////////////////////////
/// @brief back-end configuration request handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Achim Brandt
/// @author Copyright 2010, triagens GmbH, Coconfigne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "RestAdminBeConfigurationHandler.h"

#include <boost/scoped_ptr.hpp>

#include <Basics/VariantArray.h>
#include <Basics/VariantVector.h>
#include <Rest/HttpResponse.h>
#include <Rest/OutputGenerator.h>
#include <Rest/InputParser.h>

using namespace triagens::basics;
using namespace triagens::rest;

namespace triagens {
  namespace simple {

    // -----------------------------------------------------------------------------
    // Handler methods
    // -----------------------------------------------------------------------------

    HttpHandler::status_e RestAdminBeConfigurationHandler::execute () {
      switch (request->requestType()) {
        case HttpRequest::HTTP_REQUEST_GET:
          return executeRead();

        case HttpRequest::HTTP_REQUEST_PUT:
          return executeWrite();

        default:
          generateError("expecting GET or PUT /configuration");
          return HANDLER_DONE;
      }
    }

    // -----------------------------------------------------------------------------
    // private methods
    // -----------------------------------------------------------------------------

    HttpHandler::status_e RestAdminBeConfigurationHandler::executeRead () {
      VariantArray* result = new VariantArray();
      LimitValues limit = model->getLimitValues();

      result->add("logLevel", new VariantString(Logger::logLevel()));
      result->add("limitKeys", new VariantInt32(limit.maxNumKeys));
      result->add("limitKeySize", new VariantDouble(limit.maxKeySize));
      result->add("limitTotalKeySize", new VariantDouble(limit.maxTotalKeySize));
      result->add("limitDataSize", new VariantDouble(limit.maxValueSize));
      result->add("limitTotalDataSize", new VariantDouble(limit.maxTotalValueSize));
      result->add("limitTotalSize", new VariantDouble(limit.maxTotalSize));

      generateResult(result, htmlContent);
      return HANDLER_DONE;
    }



    HttpHandler::status_e RestAdminBeConfigurationHandler::executeWrite () {
      boost::scoped_ptr<VariantArray> json(InputParser::jsonArray(request));

      struct Configuration : public InputParser::ObjectDescription {
          Configuration () {
            optional("logLevel", logLevel, hasLogLevel);
            optional("limitKeys", limitKeys, hasLimitKeys);
            optional("limitKeySize", limitKeySize, hasLimitKeySize);
            optional("limitTotalKeySize", limitTotalKeySize, hasLimitTotalKeySize);
            optional("limitDataSize", limitDataSize, hasLimitDataSize);
            optional("limitTotalDataSize", limitTotalDataSize, hasLimitTotalDataSize);
            optional("limitTotalSize", limitTotalSize, hasLimitTotalSize);
          }

          string logLevel;
          bool hasLogLevel;

          int32_t limitKeys;
          bool hasLimitKeys;

          double limitKeySize;
          bool hasLimitKeySize;

          double limitTotalKeySize;
          bool hasLimitTotalKeySize;

          double limitDataSize;
          bool hasLimitDataSize;

          double limitTotalDataSize;
          bool hasLimitTotalDataSize;

          double limitTotalSize;
          bool hasLimitTotalSize;
      } desc;

      bool ok = desc.parse(json.get());

      if (! ok) {
        generateError("illegal configuration: " + desc.lastError());
        return HANDLER_DONE;
      }

      if (desc.hasLogLevel && desc.logLevel != Logger::logLevel()) {
        LOGGER_DEBUG << "switching to log level '" << desc.logLevel << "'";
        Logger::setLogLevel(desc.logLevel);
      }

      if (desc.hasLimitKeys) {
        LOGGER_DEBUG << "setting new limitKeys '" << desc.limitKeys << "'";
        model->setLimitValue(NUM_KEYS_LIMIT_EXCEEDED, uint64_t(desc.limitKeys <= 0 ? 0 : desc.limitKeys));
      }

      if (desc.hasLimitKeySize) {
        LOGGER_DEBUG << "setting new limitKeySize '" << desc.limitKeySize << "'";
        model->setLimitValue(KEY_LIMIT_EXCEEDED, uint64_t(desc.limitKeySize <= 0.0 ? 0.0 : desc.limitKeySize));
      }

      if (desc.hasLimitTotalKeySize) {
        LOGGER_DEBUG << "setting new limitTotalKeySize '" << desc.limitTotalKeySize << "'";
        model->setLimitValue(TOTAL_KEY_SIZE_EXCEEDED, uint64_t(desc.limitTotalKeySize <= 0.0 ? 0.0 : desc.limitTotalKeySize));
      }

      if (desc.hasLimitDataSize) {
        LOGGER_DEBUG << "setting new limitDataSize '" << desc.limitDataSize << "'";
        model->setLimitValue(VALUE_LIMIT_EXCEEDED, uint64_t(desc.limitDataSize <= 0.0 ? 0.0 : desc.limitDataSize));
      }

      if (desc.hasLimitTotalDataSize) {
        LOGGER_DEBUG << "setting new limitTotalDataSize '" << desc.limitTotalDataSize << "'";
        model->setLimitValue(TOTAL_VALUE_SIZE_EXCEEDED, uint64_t(desc.limitTotalDataSize <= 0.0 ? 0.0 : desc.limitTotalDataSize));
      }

      if (desc.hasLimitTotalSize) {
        LOGGER_DEBUG << "setting new limitTotalSize '" << desc.limitTotalSize << "'";
        model->setLimitValue(TOTAL_SIZE_EXCEEDED, uint64_t(desc.limitTotalSize <= 0.0 ? 0.0 : desc.limitTotalSize));
      }

      return executeRead();
    }
  }
}
