////////////////////////////////////////////////////////////////////////////////
/// @brief base class for input-output tasks from sockets
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_REST_SOCKET_TASK_H
#define TRIAGENS_REST_SOCKET_TASK_H 1

#include <Rest/IoTask.h>

#include <Basics/Mutex.h>

namespace triagens {
  namespace basics {
    class StringBuffer;
  }

  namespace rest {

    ////////////////////////////////////////////////////////////////////////////////
    /// @ingroup Scheduler
    /// @brief base class for input-output tasks from sockets
    ////////////////////////////////////////////////////////////////////////////////

    class SocketTask : public IoTask {
      private:
        static size_t const READ_BLOCK_SIZE = 10000;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructs a new task with a given socket
        ////////////////////////////////////////////////////////////////////////////////

        explicit
        SocketTask (socket_t fd);

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief fills the read buffer
        ///
        /// @param closed
        ///     will be set to true, if the system receives a close on the socket.
        ///
        /// The function should be called by input task if the scheduler has
        /// indicated that new data is available. It will return true, if data could
        /// be read and false if the connection has been closed.
        ////////////////////////////////////////////////////////////////////////////////

        virtual bool fillReadBuffer (bool& closed);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief handles a read
        ///
        /// @param closed
        ///     will be set to true, if the system receives a close on the socket.
        ////////////////////////////////////////////////////////////////////////////////

        virtual bool handleRead (bool& closed) = 0;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief handles a write
        ///
        /// @param closed
        ///     will be set to true, if the system receives a close on the socket.
        ///
        /// @param noWrite
        ///     is true if no writeBuffer exists.
        ////////////////////////////////////////////////////////////////////////////////

        virtual bool handleWrite (bool& closed, bool noWrite);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief called if write buffer has been sent
        ///
        /// This called is called if the current write buffer has been sent
        /// completly to the client.
        ////////////////////////////////////////////////////////////////////////////////

        virtual void completedWriteBuffer (bool& closed) = 0;

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief deletes a socket task
        ///
        /// This method will close the underlying socket.
        ////////////////////////////////////////////////////////////////////////////////

        ~SocketTask ();

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        void setup (Scheduler*, event_loop_t*);

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        void cleanup ();

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        bool handleEvent (void* token, int event);

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief callback structure
        ////////////////////////////////////////////////////////////////////////////////

        struct watcher_t;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief callback data
        ////////////////////////////////////////////////////////////////////////////////

        watcher_t* watcher;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief callback function
        ////////////////////////////////////////////////////////////////////////////////

        static void callback (event_loop_t*, watcher_t*, int revents);

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief sets an active write buffer
        ////////////////////////////////////////////////////////////////////////////////

        void setWriteBuffer (basics::StringBuffer*, bool ownBuffer = true);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for presence of an active write buffer
        ////////////////////////////////////////////////////////////////////////////////

        bool hasWriteBuffer () const;

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief communication socket
        ////////////////////////////////////////////////////////////////////////////////

        socket_t commSocket;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the current write buffer
        ////////////////////////////////////////////////////////////////////////////////

        basics::StringBuffer* writeBuffer;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief if true, the resource writeBuffer is owned by the write task
        ///
        /// If true, the writeBuffer is deleted as soon as it has been sent to the
        /// client. If false, the writeBuffer is keep alive.
        ////////////////////////////////////////////////////////////////////////////////

        bool ownBuffer;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief number of bytes already written
        ////////////////////////////////////////////////////////////////////////////////

        size_t writeLength;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief read buffer
        ///
        /// The function fillReadBuffer stores the data in this buffer.
        ////////////////////////////////////////////////////////////////////////////////

        basics::StringBuffer* readBuffer;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief lock on the write buffer
        ////////////////////////////////////////////////////////////////////////////////

        mutable basics::Mutex writeBufferLock;

      private:
        char * tmpReadBuffer;
    };
  }
}

#endif
