////////////////////////////////////////////////////////////////////////////////
/// @brief postfix request
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "PostfixHandler/PostfixRequest.h"

#include "Basics/StringUtils.h"

using namespace triagens::basics;

namespace triagens {
  namespace simple {

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    PostfixRequest::PostfixRequest ()
      : type(POSTFIX_REQUEST_INVALID) {
      input = 0;
      key = 0;
      value = 0;
    }



    PostfixRequest::PostfixRequest (string const& line)
      : type(POSTFIX_REQUEST_INVALID)  {

      input = StringUtils::duplicate(line);

      parseLine(line.size());
    }



    PostfixRequest::PostfixRequest (char const* line, size_t length)
      : type(POSTFIX_REQUEST_INVALID) {

      input = StringUtils::duplicate(line, length);

      parseLine(length);
    }



    PostfixRequest::~PostfixRequest () {
      if (input != 0) {
        delete[] input;
      }
    }

    // -----------------------------------------------------------------------------
    // private methods
    // -----------------------------------------------------------------------------

    void PostfixRequest::parseLine (size_t length) {
      char* cmdStart = input;
      char* end = input + length;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // recall that the \n or \r\n have already been removed
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      // skip any spaces until we hit something else
      for (;   cmdStart < end;  ++cmdStart) {
        if (*cmdStart != ' ') {
          break;
        }
      }

      // read the command we received
      char* cmdEnd = cmdStart;

      for (;  cmdEnd < end;  ++cmdEnd) {
        *cmdEnd = ::tolower(*cmdEnd);

        if (*cmdEnd == ' ') {
          *cmdEnd++ = '\0';
          break;
        }
      }

      // parse the key
      char* keyStart = cmdEnd;
      char* keyEnd;

      readHexcaped(keyStart, keyEnd, end);

      // parse the value
      char* valueStart = keyEnd;
      char* valueEnd;

      readHexcaped(valueStart, valueEnd, end);

      // store parsed key and values
      key = keyStart;
      value = valueStart;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // PUT syntax: PUT <space> KEY <space> VALUE
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      if (strcmp(cmdStart, "put") == 0) {
        type = POSTFIX_REQUEST_PUT;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // GET syntax: GET <space> KEY
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(cmdStart, "get") == 0) {
        type = POSTFIX_REQUEST_GET;
      }
    }



    void PostfixRequest::readHexcaped (char*& start, char*& stop, char* end) {

      // skip any spaces until we hit something else
      for (;   start < end;  ++start) {
        if (*start != ' ') {
          break;
        }
      }

      // read the value we received
      stop = start;
      char* ptr = start;

      for (;  stop < end;  ++stop) {
        if (*stop == '%') {
          if (stop + 2 < end) {
            *ptr++ = StringUtils::hex2int(stop[1]) << 4 | StringUtils::hex2int(stop[2]);
            stop += 2;
          }
          else if (stop + 1 < end) {
            *ptr++ = StringUtils::hex2int(stop[1]);
            stop += 1;
          }
          else {
            continue;
          }
        }
        else if (*stop == ' ') {
          ++stop;
          break;
        }
        else {
          *ptr++ = *stop;
        }
      }

      *ptr = '\0';
    }
  }
}
