////////////////////////////////////////////////////////////////////////////////
/// @brief postfix base request handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef SIMPLEVOC_POSTFIXHANDLER_POSTFIX_BASE_HANDLER_H
#define SIMPLEVOC_POSTFIXHANDLER_POSTFIX_BASE_HANDLER_H 1

#include <Basics/Timing.h>
#include <Rest/LineHandler.h>

#include "PostfixHandler/PostfixRequest.h"
#include "PostfixHandler/PostfixResponse.h"
#include "SimpleVoc/SimpleModel.h"
#include "SimpleVoc/SimpleVocFigures.h"

namespace triagens {
  namespace simple {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief postfix base request handler
    ////////////////////////////////////////////////////////////////////////////////

    class PostfixBaseHandler : public rest::LineHandler {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        PostfixBaseHandler (rest::LineRequest* request, SimpleModel* model, size_t statistics)
          : LineHandler(request),
            model(model),
            statistics(statistics),
            postfixRequest(dynamic_cast<PostfixRequest*>(request)),
            timing(basics::Timing::TI_WALLCLOCK) {
          info << basics::LoggerData::Task("POSTFIX " + postfixRequest->requestTypeString());
          LOGGER_REQUEST_IN_START_I(info);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructor
        ////////////////////////////////////////////////////////////////////////////////

        ~PostfixBaseHandler () {
          double time = timing.time() / 1000000.0;
          addFigure<SimpleVocStatistics::runtimesAccessor>(statistics, time);

          LOGGER_REQUEST_IN_END_I(info) << infoResult;
        }

      public:


        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        bool requiresBody () {
          return false;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        bool requiresBodyLines () {
          return false;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        size_t requiredBodyLength () {
          return 0;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        status_e execute () {
          response = new PostfixResponse();

          if (postfixRequest->requestType() != PostfixRequest::POSTFIX_REQUEST_INVALID) {
            work(postfixRequest, dynamic_cast<PostfixResponse*>(response));
          }
          else {
            response->body().appendText("400 cannot parse postfix request\n");
          }

          return HANDLER_DONE;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        void handleError (basics::TriagensError const& ex) {
          if (response != 0) {
            delete response;
          }

          response = new PostfixResponse();
          response->body().appendText("400 encountered error");
          response->body().appendText(DIAGNOSTIC_INFORMATION(ex));
          response->body().appendText("\r");
        }

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief method called to handle the request
        ////////////////////////////////////////////////////////////////////////////////

        virtual void work (PostfixRequest*, PostfixResponse*) = 0;

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the underlying SimpleModel interface
        ////////////////////////////////////////////////////////////////////////////////

        SimpleModel* model;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief type
        ////////////////////////////////////////////////////////////////////////////////

        size_t statistics;

      private:
        PostfixRequest* postfixRequest;
        basics::Timing timing;
        basics::LoggerTiming info;
        basics::LoggerData::Extra infoResult;
    };
  }
}

#endif
