////////////////////////////////////////////////////////////////////////////////
/// @brief memcached request
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "MCHandler/MCRequest.h"

#include "Basics/StringUtils.h"

using namespace triagens::basics;

namespace triagens {
  namespace simple {

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    MCRequest::MCRequest ()
      : type(MC_REQUEST_INVALID) {
      cmdLine = 0;
    }



    MCRequest::MCRequest (string const& commandLine)
      : type(MC_REQUEST_INVALID)  {

      cmdLine = StringUtils::duplicate(commandLine);

      parseHeader(commandLine.size());
    }



    MCRequest::MCRequest (char const* commandLine, size_t length)
      : type(MC_REQUEST_INVALID) {

      cmdLine = StringUtils::duplicate(commandLine, length);

      parseHeader(length);
    }



    MCRequest::~MCRequest () {
      if (cmdLine != 0) {
        delete[] cmdLine;
      }
    }

    // -----------------------------------------------------------------------------
    // private methods
    // -----------------------------------------------------------------------------

    void MCRequest::parseHeader (size_t length) {
      char* start = cmdLine;
      char* end = start + length;


      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // recall that the \n or \r\n have already been removed
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      // skip any spaces until we hit something else
      for (;   start < end;  ++start) {
        if (*start != ' ') {
          break;
        }
      }

      // read the command we received
      char* commandEnd = start;

      for (; commandEnd < end; ++commandEnd) {
        *commandEnd = ::tolower(*commandEnd);

        if (*commandEnd == ' ') {
          *commandEnd = '\0';
          ++commandEnd;
          break;
        }
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // ADD syntax is: ADD KEY FLAGS EXPTIME NUM_BYTES [NOREPLY]\r\n
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      if (strcmp(start, "add") == 0) {

        // read the key which tells us what was requested
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // read the flags
        if (! this->readInteger(commandEnd,end,cmdStructure.flags)) { return; }

        // read the time to live
        if (! this->readInteger(commandEnd,end,cmdStructure.expireTime)) { return; }

        // read the number of bytes in the data blob
        if (! this->readInteger(commandEnd,end,cmdStructure.bytes)) { return; }

        // check and see if client sent noreply
        if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }

        // check and see if they are any additional (no space) characters - if so this invalids the request
        if (this->readStringTest(commandEnd, end)) { return; }

        // the bytes remaining to read from socket initially is the number of bytes in the set command
        cmdStructure.bytesRemaining = cmdStructure.bytes;
        type = MC_REQUEST_ADD;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // APPEND syntax: APPEND key flags exptime num_bytes [noreply]\r\n
      // flags, exptime accepted but not used
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "append") == 0) {

        // read the key which tells us what was requested
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // the flags and time to live ARE ignored in the append statement
        // read the flags
        if (! this->readInteger(commandEnd,end,cmdStructure.flags)) { return; }

        // read the time to live
        if (! this->readInteger(commandEnd,end,cmdStructure.expireTime)) { return; }

        // read the number of bytes in the data blob
        if (! this->readInteger(commandEnd,end,cmdStructure.bytes)) { return; }

        // check and see if client sent noreply
        if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }

        // check and see if they are any additional (no space) characters - if so this invalids the request
        if (this->readStringTest(commandEnd, end)) { return; }

        // the bytes remaining to read from socket initially is the number of bytes in the set command
        cmdStructure.bytesRemaining = cmdStructure.bytes;
        type = MC_REQUEST_APPEND;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // CAS syntax is: CAS KEY FLAGS EXPTIME NUM_BYTES CAS_UNIQUE [NOREPLY]\r\n
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "cas") == 0) {

        // read the key which tells us what was requested
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // read the flags
        if (! this->readInteger(commandEnd,end,cmdStructure.flags)) { return; }

        // read the time to live
        if (! this->readInteger(commandEnd,end,cmdStructure.expireTime)) { return; }

        // read the number of bytes in the data blob
        if (! this->readInteger(commandEnd,end,cmdStructure.bytes)) { return; }

        // read the cas unique number
        if (! this->readInteger(commandEnd,end,cmdStructure.casUnique)) { return; }

        // check and see if client sent noreply
        if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }

        // check and see if they are any additional (no space) characters - if so this invalids the request
        if (this->readStringTest(commandEnd, end)) { return; }

        // the bytes remaining to read from socket initially is the number of bytes in the set command
        cmdStructure.bytesRemaining = cmdStructure.bytes;
        type = MC_REQUEST_CAS;

        return;
      }


      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // DATABASE syntax: database name_of_database_to_change_to
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "database") == 0) { // database command received
        type = MC_REQUEST_DATABASE;
        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // DECR syntax: DECR KEY VALUE [noreply]
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "decr") == 0) {

        // read the key which tells us what item was requested
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // read the value
        if (! this->readInteger(commandEnd,end,cmdStructure.increment)) { return; }

        // check and see if client sent noreply
        if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }

        // check and see if they are any additional (no space) characters - if so this invalids the request
        if (this->readStringTest(commandEnd, end)) { return; }

        type = MC_REQUEST_DECR;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // DELETE syntax: DELETE key [time] [noreply]\r\n
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      else if (strcmp(start, "delete") == 0) {

        // read the key which tells us what item was requested
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // we don't know if the next string is an integer, noreply or nothing
        char* oldCommandEnd = commandEnd;
        pair<char*, size_t> value = pair<char*, size_t>(0,0);

        if (! this->readString(commandEnd, end,value)) { // we have nothing else on the command line
          if (value.first != 0) {
            delete [] value.first;
          }

          type = MC_REQUEST_DELETE;

          return;
        }

        if (value.first[0] > 47 && value.first[0] < 58) { // we have a time value
          cmdStructure.deletionTime = StringUtils::uint64(value.first,value.second);

          if (value.first != 0) {
            delete [] value.first;
          }

          if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }
        }
        else {
          if (value.first != 0) {
            delete [] value.first;
          }

          commandEnd = oldCommandEnd;

          if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }
        }

        // check and see if they are any additional (no space) characters - if so this invalids the request
        if (this->readStringTest(commandEnd, end)) { return; }

        // the bytes remaining to read from socket initially is the number of bytes in the set command
        type = MC_REQUEST_DELETE;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // FLUSH_ALL: FLUSH_ALL delay [noreply]\r\n
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "flush_all") == 0) {

        // we don't know if the next string is an integer, noreply or nothing
        char* oldCommandEnd = commandEnd;
        pair<char*, size_t> value = pair<char*, size_t>(0,0);

        if (! this->readString(commandEnd, end,value)) { // we have nothing else on the command line

          if (value.first != 0) {
            delete [] value.first;
          }

          type = MC_REQUEST_FLUSH_ALL;

          return;
        }

        if (value.first[0] > 47 && value.first[0] < 58) { // we have a time value
          cmdStructure.flushAllDelay = StringUtils::uint64(value.first, value.second);

          if (value.first != 0) {
            delete [] value.first;
          }

          if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }
        }
        else { //
          if (value.first != 0) {
            delete [] value.first;
          }
          commandEnd = oldCommandEnd;
          if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }
        }

        // check and see if they are any additional (no space) characters - if so this invalids the request
        if (this->readStringTest(commandEnd, end)) { return; }

        type = MC_REQUEST_FLUSH_ALL;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // GET syntax:
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "get") == 0) { // get command received

        // read the key which tells us what was requested - there must be at least one key
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // keep readings the get keys
        int jj = 1;

        while (true) {
          cmdStructure.keys.push_back( pair<char*, size_t>(0,0) );

          if (! this->readString(commandEnd, end,cmdStructure.keys[jj]) ) {
            break;
          }

          ++jj;
        }

        type = MC_REQUEST_GET;
        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // GETS syntax:
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "gets") == 0) { // gets command received

        // read the key which tells us what was requested - there must be at least one key
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // keep readings the get keys
        int jj = 1;

        while (true) {
          cmdStructure.keys.push_back( pair<char*, size_t>(0,0) );

          if (! this->readString(commandEnd, end,cmdStructure.keys[jj]) ) {
            break;
          }

          ++jj;
        }

        type = MC_REQUEST_GETS;
        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // INCR syntax: INCR KEY VALUE [noreply]
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "incr") == 0) {

        // read the key which tells us what item was requested
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // read the value
        if (! this->readInteger(commandEnd,end,cmdStructure.increment)) { return; }

        // check and see if client sent noreply
        if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }

        // check and see if they are any additional (no space) characters - if so this invalids the request
        if (this->readStringTest(commandEnd, end)) { return; }

        type = MC_REQUEST_INCR;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // PGET syntax:
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "pget") == 0) { // pget command received

        // read the key which tells us what was requested - there must be at least one key
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // keep readings the pget keys
        int jj = 1;

        while (true) {
          cmdStructure.keys.push_back( pair<char*, size_t>(0,0) );

          if (! this->readString(commandEnd, end,cmdStructure.keys[jj]) ) {
            break;
          }

          ++jj;
        }

        type = MC_REQUEST_PGET;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // PGET syntax:
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "pgets") == 0) { // pget command received

        // read the key which tells us what was requested - there must be at least one key
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // keep readings the pget keys
        int jj = 1;

        while (true) {
          cmdStructure.keys.push_back( pair<char*, size_t>(0,0) );

          if (! this->readString(commandEnd, end,cmdStructure.keys[jj]) ) {
            break;
          }

          ++jj;
        }

        type = MC_REQUEST_PGETS;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // PREPEND syntax: PREPEND key flags exptime num_bytes [noreply]\r\n
      // flags, exptime accepted but not used
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "prepend") == 0) {

        // read the key which tells us what was requested
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // the flags and time to live ARE ignored in the prepend statement
        // read the flags
        if (! this->readInteger(commandEnd,end,cmdStructure.flags)) { return; }

        // read the time to live
        if (! this->readInteger(commandEnd,end,cmdStructure.expireTime)) { return; }

        // read the number of bytes in the data blob
        if (! this->readInteger(commandEnd,end,cmdStructure.bytes)) { return; }

        // check and see if client sent noreply
        if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }

        // check and see if they are any additional (no space) characters - if so this invalids the request
        if (this->readStringTest(commandEnd, end)) { return; }

        // the bytes remaining to read from socket initially is the number of bytes in the set command
        cmdStructure.bytesRemaining = cmdStructure.bytes;
        type = MC_REQUEST_PREPEND;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // QUERY syntax:
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "query") == 0) { // query command received
        type = MC_REQUEST_QUERY;
        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // QUIT syntax: quit\r\n
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "quit") == 0) { // query command received
        type = MC_REQUEST_QUIT;
        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // REPLACE syntax:
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "replace") == 0) { // replace command received

        // read the key which tells us what was requested
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // read the flags
        if (! this->readInteger(commandEnd,end,cmdStructure.flags)) { return; }

        // read the time to live
        if (! this->readInteger(commandEnd,end,cmdStructure.expireTime)) { return; }

        // read the number of bytes in the data blob
        if (! this->readInteger(commandEnd,end,cmdStructure.bytes)) { return; }

        // check and see if client sent noreply
        if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }

        // check and see if they are any additional (no space) characters - if so this invalids the request
        if (this->readStringTest(commandEnd, end)) { return; }

        // the bytes remaining to read from socket initially is the number of bytes in the set command
        cmdStructure.bytesRemaining = cmdStructure.bytes;
        type = MC_REQUEST_REPLACE;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // SET syntax:
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "set") == 0) {

        // read the key which tells us what was requested
        if (! this->readString(commandEnd, end,cmdStructure.keys[0])) { return; }

        // read the flags
        if (! this->readInteger(commandEnd,end,cmdStructure.flags)) { return; }

        // read the time to live
        if (! this->readInteger(commandEnd,end,cmdStructure.expireTime)) { return; }

        // read the number of bytes in the data blob
        if (! this->readInteger(commandEnd,end,cmdStructure.bytes)) { return; }

        // check and see if client sent noreply
        if (!this->readStringNoReply(commandEnd, end,cmdStructure.noreply)) { return; }

        // check and see if they are any additional (no space) characters - if so this invalids the request
        if (this->readStringTest(commandEnd, end)) { return; }

        // the bytes remaining to read from socket initially is the number of bytes in the set command
        cmdStructure.bytesRemaining = cmdStructure.bytes;
        type = MC_REQUEST_SET;

        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // STATS syntax:
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "stat") == 0 || strcmp(start, "stats") == 0) { // stats command received
        type = MC_REQUEST_STATS;
        return;
      }

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      // VERSION syntax:
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      else if (strcmp(start, "version") == 0) { // version command received
        type = MC_REQUEST_VERSION;
        return;
      }

    }



    bool MCRequest::readInteger (char*& start, char const* end, int64_t& value) {

      // first skip any spaces
      for (;  start < end;  ++start) {
        if (*start != ' ') {
          break;
        }
      }

      char* keyStart = start;
      size_t length  = 0;

      // next keep reading until we hit a space
      for (;  start < end;  ++start) {
        if (*start == ' ') {
          length = start - keyStart;
          *start = '\0';
          ++start;
          break;
        }
      }

      if (start == end) {
        length = start - keyStart;
      }

      if (length == 0) {
        return false;
      }

      value = StringUtils::int64(keyStart,length);
      return true;
    }



    bool MCRequest::readInteger (char*& start, char const* end, uint64_t& value) {

      // first skip any spaces
      for (; start < end; ++start) {
        if (*start != ' ') {
          break;
        }
      }

      char* keyStart = start;
      size_t length  = 0;

      // next keep reading until we hit a space
      for (; start < end; ++start) {
        if (*start == ' ') {
          length = start - keyStart;
          *start = '\0';
          ++start;
          break;
        }
      }

      if (start == end) {
        length = start - keyStart;
      }

      if (length == 0) {
        return false;
      }

      value = StringUtils::uint64(keyStart,length);
      return true;
    }



    bool MCRequest::readInteger (char*& start, char const* end, uint32_t& value) {

      // first skip any spaces
      for (; start < end; ++start) {
        if (*start != ' ') {
          break;
        }
      }

      char* keyStart = start;
      size_t length  = 0;

      // next keep reading until we hit a space
      for (; start < end; ++start) {
        if (*start == ' ') {
          length = start - keyStart;
          *start = '\0';
          ++start;
          break;
        }
      }

      if (start == end) {
        length = start - keyStart;
      }

      if (length == 0) {
        return false;
      }

      value = StringUtils::uint64(keyStart,length);
      return true;
    }



    bool MCRequest::readString (char*& start, char const* end, pair<char*, size_t>& value) {
      // first skip any spaces
      for (; start < end; ++start) {
        if (*start != ' ') {
          break;
        }
      }

      char* keyStart = start;
      size_t length  = 0;

      // next keep reading until we hit a space
      for (; start < end; ++start) {
        if (*start == ' ') {
          length = start - keyStart;
          *start = '\0';
          ++start;
          break;
        }
      }

      if (start == end) {
        length = start - keyStart;
      }

      if (length == 0) {
        return false;
      }

      value.first  = StringUtils::duplicate(keyStart, length);
      value.second = length;

      return true;
    }



    bool MCRequest::readStringLower (char*& start, char const* end, pair<char*, size_t>& value) {
      // first skip any spaces
      for (; start < end; ++start) {
        if (*start != ' ') {
          break;
        }
      }

      char* keyStart = start;
      size_t length  = 0;

      // next keep reading until we hit a space
      for (; start < end; ++start) {
        if (*start == ' ') {
          length = start - keyStart;
          *start = '\0';
          ++start;
          break;
        }
        *start = ::tolower(*start);
      }

      if (start == end) {
        length = start - keyStart;
      }

      if (length == 0) {
        return false;
      }

      value.first  = StringUtils::duplicate(keyStart, length);
      value.second = length;
      return true;
    }



    bool MCRequest::readStringNoReply (char*& start, char const* end, bool& value) {

      // first skip any spaces
      for (; start < end; ++start) {
        if (*start != ' ') {
          break;
        }
      }

      // default value for noreply = false
      value = false;

      // client didn't send anything further
      if (start == end) {
        return true;
      }

      char* keyStart = start;
      size_t length  = 0;

      // next keep reading until we hit a space
      int i = 0;
      char const* noreply = "noreply";

      for (; start < end; ++start) {
        if (*start == ' ') {
          length = start - keyStart;
          *start = '\0';
          ++start;
          break;
        }
        else {
          *start = ::tolower(*start);
          if (i < 7) {
            if (*start != noreply[i]) {
              return false;
            }
            i++;
          }
          else {
            return false;
          }
        }
      }

      value = true;
      return true;
    }



    bool MCRequest::readStringTest (char* start, char const* end) {

      // first skip any spaces
      for (; start < end; ++start) {
        if (*start != ' ') {
          break;
        }
      }

      if (start == end) {
        return false;
      }
      else {
        return true;
      }
    }
  }
}
