////////////////////////////////////////////////////////////////////////////////
/// @brief memcached base request handler
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef SIMPLEVOC_MCHANDLER_MCBASE_HANDLER_H
#define SIMPLEVOC_MCHANDLER_MCBASE_HANDLER_H 1

#include <Basics/Common.h>

#include <Basics/Timing.h>
#include <Rest/LineHandler.h>

#include "MCHandler/MCRequest.h"
#include "MCHandler/MCResponse.h"
#include "SimpleVoc/SimpleModel.h"
#include "SimpleVoc/SimpleVocFigures.h"

namespace triagens {
  namespace simple {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief memcached base request handler
    ////////////////////////////////////////////////////////////////////////////////

    class MCBaseHandler : public rest::LineHandler {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        MCBaseHandler (rest::LineRequest* request, SimpleModel* model, size_t statistics)
          : LineHandler(request),
            model(model),
            statistics(statistics),
            mcRequest(dynamic_cast<MCRequest*>(request)),
            timing(basics::Timing::TI_WALLCLOCK) {
          info << basics::LoggerData::Task("MC " + mcRequest->requestTypeString());
          LOGGER_REQUEST_IN_START_I(info);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructor
        ////////////////////////////////////////////////////////////////////////////////

        ~MCBaseHandler () {
          double time = timing.time() / 1000000.0;
          addFigure<SimpleVocStatistics::runtimesAccessor>(statistics, time);

          LOGGER_REQUEST_IN_END_I(info) << infoResult;
        }

      public:


        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        bool requiresBody () {
          return mcRequest->cmdStructure.bytesRemaining > 0;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        bool requiresBodyLines () {
          return false;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        size_t requiredBodyLength () {
          return mcRequest->cmdStructure.bytesRemaining;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        status_e execute () {
          response = new MCResponse();

          if (mcRequest->requestType() != MCRequest::MC_REQUEST_INVALID) {
            work(mcRequest, dynamic_cast<MCResponse*>(response));
          }
          else {
            response->body().appendText("ERROR\r\n");
          }

          return HANDLER_DONE;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// {@inheritDoc}
        ////////////////////////////////////////////////////////////////////////////////

        void handleError (basics::TriagensError const&) {
          if (response != 0) {
            delete response;
          }

          response = new MCResponse();
          response->body().appendText("ERROR\r\n");
        }

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief method called to handle the request
        ////////////////////////////////////////////////////////////////////////////////

        virtual void work (MCRequest*, MCResponse*) = 0;

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the underlying memcached interface
        ////////////////////////////////////////////////////////////////////////////////

        SimpleModel* model;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief type
        ////////////////////////////////////////////////////////////////////////////////

        size_t statistics;

      private:
        MCRequest* mcRequest;
        basics::Timing timing;
        basics::LoggerTiming info;
        basics::LoggerData::Extra infoResult;
    };
  }
}

#endif
