////////////////////////////////////////////////////////////////////////////////
/// @brief task for line communication
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "LineCommTask.h"

#include <Rest/LineHandler.h>
#include <Rest/LineHandlerFactory.h>
#include <Rest/LineRequest.h>
#include <Rest/LineResponse.h>
#include <Rest/Scheduler.h>

#include "GeneralServer/GeneralFigures.h"
#include "LineServer/LineServer.h"

using namespace triagens::basics;
using namespace triagens::rest::GeneralFigures;

namespace triagens {
  namespace rest {

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    LineCommTask::LineCommTask (LineServer* server, socket_t fd, ConnectionInfo const& info)
      : Task("LineCommTask"),
        GeneralCommTask<LineServer, LineHandlerFactory>(server, fd, info) {
      incCounter<GeneralServerStatistics::lineAccessor>();
    }



    LineCommTask::~LineCommTask () {
      decCounter<GeneralServerStatistics::lineAccessor>();
    }

    // -----------------------------------------------------------------------------
    // GeneralCommTask methods
    // -----------------------------------------------------------------------------

    bool LineCommTask::processRead () {

      //cout << __FILE__ << ":" << __LINE__ << ":" << requestPending << ":" << readBuffer->c_str() << ":" << readRequestBody << ":" << endl;

      if (requestPending || readBuffer->c_str() == 0) {
        return true;
      }

      bool handleRequest = false;

      if (! readRequestBody) {
        char const * ptr = readBuffer->c_str() + readPosition;
        char const * end = readBuffer->end();

        for (;  ptr < end;  ptr++) {
          if (ptr[0] == '\n') {
            break;
          }
        }

        size_t headerLength = ptr - readBuffer->c_str();

        if (headerLength > maximalHeaderSize) {
          LOGGER_WARNING << "maximal header size is " << maximalHeaderSize << ", request header size is "
                         << headerLength;
          return false;
        }

        if (ptr < end) {

          // the body starts after the "\n"
          bodyPosition = ptr - readBuffer->c_str() + 1;

          // skip any "\r" before the "\n"
          if (readBuffer->c_str() < ptr && ptr[-1] == '\r') {
            --ptr;
          }

          // correct the header length
          headerLength = ptr - readBuffer->c_str();

          LOGGER_TRACE << "LINE READ FOR " << static_cast<Task*>(this) << ":\n" << string(readBuffer->c_str(), headerLength);

          // create the request
          request = server->createRequest(readBuffer->c_str(), headerLength);

          if (request == 0) {
            LOGGER_WARNING << "cannot create the request, giving up";
            return false;
          }

          request->setConnectionInfo(connectionInfo);

          LOGGER_TRACE << "server port = " << connectionInfo.serverPort << ", client port = " << connectionInfo.clientPort;

          // create the handler
          handler = server->createHandler(request);

          if (handler == 0) {
            LOGGER_WARNING << "no handler is known for request, giving up";

            delete request;
            request = 0;

            return false;
          }

          // check if we need a body
          if (handler->requiresBody()) {
            readRequestBody = true;
            readPosition = bodyPosition;

            if (handler->requiresBodyLines()) {
              readBodyLines = true;
              bodyLength = 0;
            }
            else {
              readBodyLines = false;
              bodyLength = handler->requiredBodyLength();
            }
          }
          else {
            readBuffer->erase_front(bodyPosition);

            handleRequest = true;
          }
        }
        else {
          readPosition = end - readBuffer->c_str();
        }
      }

      //cout << __FILE__ << ":" << __LINE__ << ":" << readRequestBody << ":" <<bodyLength << ":" << endl;

      // readRequestBody might have changed, so cannot use else
      while (readRequestBody) {

        // we have to read one line
        if (readBodyLines) {
          char const * start = readBuffer->c_str() + readPosition;
          char const * ptr = start;
          char const * end = readBuffer->end();

          for (;  ptr < end;  ptr++) {
            if (ptr[0] == '\n') {
              break;
            }
          }

          if (ptr < end) {
            bodyLength = ptr - start;

            if (start < ptr && ptr[-1] == '\r') {
              --bodyLength;
            }

            request->addBodyLine(start, bodyLength);

            readBuffer->erase_front(ptr + 1 - readBuffer->c_str());

            readPosition = 0;
            bodyPosition = 0;
            bodyLength = 0;
          }
          else {
            readPosition = end - readBuffer->c_str();
            return true;
          }
        }

        // we have to read a blob of given size
        else {
          if (bodyLength > maximalBodySize) {
            LOGGER_WARNING << "maximal body size is " << maximalBodySize << ", request body size is " << bodyLength;
            return false;
          }

          if (readBuffer->length() - bodyPosition < bodyLength) {
            return true;
          }


          // read "bodyLength" from read buffer and add this body to the request
          request->addBody(readBuffer->c_str() + bodyPosition, bodyLength);

          LOGGER_TRACE << string(readBuffer->c_str() + bodyPosition, bodyLength);

          // remove body from read buffer and reset read position
          readBuffer->erase_front(bodyPosition + bodyLength);

          readPosition = 0;
          bodyPosition = 0;
          bodyLength = 0;
        }

        // check if we are done
        if (handler->requiresBody()) {
      //cout << __FILE__ << ":" << __LINE__ << ":" << readRequestBody << ":" <<bodyLength << ":" << endl;
          if (handler->requiresBodyLines()) {
            readBodyLines = true;
            bodyLength = 0;
          }
          else {
            readBodyLines = false;
            bodyLength = handler->requiredBodyLength();
          }
        }
        else {
          size_t buffLength = readBuffer->length();

          if (buffLength < 1) {
            readRequestBody = true;
            handleRequest = false;
            return true;
          }
          // finished with the blob
          else if (*(readBuffer->c_str()) == '\n') {
            readRequestBody = false;
            handleRequest = true;
            readBuffer->erase_front(1);
          }
          else if (buffLength > 1 && *(readBuffer->c_str()) == '\r' && *(readBuffer->c_str() + 1) == '\n') {
            readRequestBody = false;
            handleRequest = true;
            readBuffer->erase_front(2);
          }
          else { // malformed request
            readRequestBody = false;
            handleRequest = true;
            request->setLineRequestInvalid();
          }
        }
      }

      //cout << __FILE__ << ":" << __LINE__ << ":" << handleRequest  << ":" <<bodyLength << ":" << endl;

      if (handleRequest) {
        requestPending = true;

        request = 0;

        readPosition = 0;
        bodyPosition = 0;
        bodyLength = 0;

        bool ok = server->handleRequest(this, handler);

        if (! ok) {
          LOGGER_FATAL << "handleRequest failed";
          return false;
        }

        return processRead();
      }

      return true;
    }



    void LineCommTask::addResponse (LineResponse* response) {
      StringBuffer * buffer;

      // save header
      buffer = new StringBuffer();
      buffer->initialise();
      response->write(buffer);

      writeBuffers.push_back(buffer);

      LOGGER_TRACE << "LINE WRITE FOR " << static_cast<Task*>(this) << ":\n" << buffer->c_str();

      // start output
      fillWriteBuffer();
    }
  }
}
