////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_HPDF_HPDF_PREFIX_FILTERED_H
#define TRIAGENS_HPDF_HPDF_PREFIX_FILTERED_H 1

#include <Hpdf/Common.h>

#include <Hpdf/HpdfDirectTable.h>
#include <Hpdf/HpdfIndirectTable.h>


namespace triagens {
  namespace hpdf {
    class HpdfPrefixFilterTree;


    ////////////////////////////////////////////////////////////////////////////////
    /// @brief Indirect noble which is a prefix filtered of another direct table
    ////////////////////////////////////////////////////////////////////////////////


    template<typename R>
    class IndirectNoblePrefixFiltered : public IndirectNoble<R> {

      public:
        typedef typename IndirectNoble<R>::iterator iterator;
        typedef typename IndirectNoble<R>::row_t  row_t;

      // -----------------------------------------------------------------------------
      // constructors and destructors
      // -----------------------------------------------------------------------------

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor
        ///
        /// Constructs a new indirect noble from a given direct prefix table.
        ////////////////////////////////////////////////////////////////////////////////

        template<typename S, typename P>
        IndirectNoblePrefixFiltered (S& input, typename S::row_t* mark, P& find) {
          find.findIndex(input, *mark);
          //typename vector<typename S::row_t*>::const_iterator j;
          typename vector<typename S::row_t*>::const_iterator j;

          for (j = find.data.begin(); j != find.data.end(); ++j) {
            iterator result = this->add(reinterpret_cast<R*>(*j)); // type conversion required here
          }
        }

    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief Direct noble which is a prefix filtered of another direct table
    ////////////////////////////////////////////////////////////////////////////////


    template<typename R, typename N>
    class DirectNoblePrefixFiltered : public DirectNoble<R, N> {

      public:
        typedef typename DirectNoble<R,N>::iterator iterator;
        typedef typename DirectNoble<R,N>::row_t  row_t;

      // -----------------------------------------------------------------------------
      // constructors and destructors
      // -----------------------------------------------------------------------------

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor
        ///
        /// Constructs a new direct noble from a given direct prefix table.
        ////////////////////////////////////////////////////////////////////////////////

        template<typename S, typename P>
        DirectNoblePrefixFiltered (S& input, typename S::row_t* mark, P& find) {
          find.findIndex(input, *mark);
          typename vector<typename S::row_t*>::const_iterator j;
          for (j = find.data.begin(); j != find.data.end(); ++j) {
            Row const* result = this->add(reinterpret_cast<Row const *>(*j)); // type conversion required here
          }
        }

    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief Direct vassal which is a Prefix Filtered slice of another direct table
    ////////////////////////////////////////////////////////////////////////////////


    template<typename R, typename N>
    class DirectVassalPrefixFiltered : public DirectVassal<R, N> {

      public:
        typedef typename vector<R*>::iterator iterator;

      // -----------------------------------------------------------------------------
      // constructors and destructors
      // -----------------------------------------------------------------------------

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor
        ///
        /// Constructs a new Direct Vassal from a given direct table
        ////////////////////////////////////////////////////////////////////////////////

        template<typename S, typename P>
        DirectVassalPrefixFiltered (S& input, typename S::row_t* mark, P& find) {
          find.findIndex(input, *mark);
          typename vector<typename S::row_t*>::const_iterator j;
          for (j = find.data.begin(); j != find.data.end(); ++j) {
            rows.push_back(reinterpret_cast<R*>(*j)); // type conversion required here
          }
        }

        inline iterator begin () {
          return rows.begin();
        }

        inline iterator end() {
          return rows.end();
        }

        private:
          vector<R*> rows;
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief Indirect vassal which is a filtered slice of another indirect table
    ////////////////////////////////////////////////////////////////////////////////

    template<typename R, typename N>
    class IndirectVassalPrefixFiltered : public IndirectVassal<R, N> {

      public:
        typedef typename IndirectVassal<R, N>::iterator iterator;
        typedef typename IndirectVassal<R, N>::index_t index_t;
        typedef R row_t;

      ////////////////////////////////////////////////////////////////////////////////
      // constructors and destructors
      ////////////////////////////////////////////////////////////////////////////////

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor
        /// Constructs a new Vassal from a given indirect table
        ////////////////////////////////////////////////////////////////////////////////

        template<typename S, typename P>
        IndirectVassalPrefixFiltered (S& input, typename S::row_t const * mark, P const&) {
          iterator beginPtr = input.begin();
          iterator endPtr   = input.end();
          index_t indexMark;

          indexMark.setRow(mark);

          pair<index_t*, index_t*> result =
          std::equal_range(beginPtr, endPtr, indexMark, P::isLess);

          this->dataBegin = result.first;
          this->dataEnd   = result.second;
        }
    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief HashFilteredCondition
    ////////////////////////////////////////////////////////////////////////////////

    struct PrefixFilteredCondition {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief IN operator
        ////////////////////////////////////////////////////////////////////////////////

        template<typename N, typename M>
        inline static bool IN (N left, M right[]) {
          size_t length = sizeof(right) / sizeof(int);

          for (size_t i = 0;  i < length;  i++) {
            if (left == right[i]) {
              return true;
            }
          }

          return false;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief EQ operator
        ////////////////////////////////////////////////////////////////////////////////

        inline static bool EQ (char const* left, char const* right) {
          return ::strcmp(left, right) == 0;
        }

    };

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief hashFilteredStep
    ////////////////////////////////////////////////////////////////////////////////

    template<typename S, typename R, typename Q>
    inline bool prefixFilteredStep (const R& r, const Q* param) {
      return S::prefixFilter(r, param);
    }


  }
}

#endif

