////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// A parser for parsing strings using the spirit library
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_HPDF_FILTER_PARSER_H
#define TRIAGENS_HPDF_FILTER_PARSER_H 1

#include <Hpdf/Common.h>

#include <boost/version.hpp>

#if BOOST_VERSION >= 103800
#include <boost/spirit/include/classic.hpp>
#include <boost/spirit/include/classic_ast.hpp>
#else
#include <boost/spirit/core.hpp>
#include <boost/spirit/tree/ast.hpp>
#endif

#include <Hpdf/HpdfBinaryTreeNode.h>

namespace triagens {
  namespace hpdf {
#if BOOST_VERSION >= 103800
    using namespace boost::spirit::classic;
#else
    using namespace boost::spirit;
#endif

    class AttributeDescriptor;
    class FilterParserGrammar;
    class HpdfBinaryTreeNode;
    class HpdfFilterTree;

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief FilterParser
    ///
    /// This object is created for dynamic filtering.  It
    /// perfroms three functions:
    ///
    /// (i) it parses the ast tree created by the boost parser and creates its own
    /// internal tree, and
    ///
    /// (ii) evaluates it (internal) node,
    ///
    /// (iii) finally it creates the matrix used for the (hopefully) fast evaluation
    /// of the original logical expression.
    ////////////////////////////////////////////////////////////////////////////////

    class FilterParser {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ///
        /// A HpdfFilterTree object (already created) is sent to this constructor as
        /// well as list of table attributes and the actual string to parse.
        ////////////////////////////////////////////////////////////////////////////////

        FilterParser (const string& str, HpdfFilterTree*, AttributeDescriptor*);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructor
        ////////////////////////////////////////////////////////////////////////////////

        ~FilterParser ();

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief Parse
        ///
        /// Performs the actually spirit parsing. In addition if the parse was
        /// successful, then tells the HpdfFilterTree object of this.
        ////////////////////////////////////////////////////////////////////////////////

        void parse ();


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief evaluateTree
        ///
        /// This method is called from the function HpdfFilterTreeDescriptor.
        /// It creates our own internal tree structure
        ////////////////////////////////////////////////////////////////////////////////

        void evaluateTree ();

      private:
        void createBinaryTree (tree_match<char const*>::tree_iterator const& i,
                               HpdfBinaryTreeNode* treeNode,
                               int treeHeight,
                               int& maxTreeHeight,
                               int& maxTreeWidth);

        void fillInEvaluationNodeArray (const int maxTreeHeight, const int maxTreeWidth);

        void fillArray (HpdfBinaryTreeNode*** evalNodeArray,
                        HpdfBinaryTreeNode* treeNode,
                        int treeHeight,
                        int const maxTreeWidth);

        LogicalType attributeIdentifier (tree_match<char const*>::tree_iterator const& itNode);
        LogicalType constantIdentifier (tree_match<char const*>::tree_iterator const& itNode);
        LogicalType logicalOperator (tree_match<char const*>::tree_iterator const& itNode);

        void printOutTree (tree_match<char const*>::tree_iterator const& i, const string& indent);

      private:
        string stringToParse;
        FilterParserGrammar* filterParserGrammar;
        tree_parse_info<> treeInfo;
        HpdfFilterTree* filterTree;
        AttributeDescriptor* attributeDescription;
    };
  }
}

#endif


