////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// A parser for parsing strings using the spirit library
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "FilterParser.h"

#include <Basics/Logger.h>
#include <Basics/StringUtils.h>
#include <Hpdf/AttributeDescriptor.h>
#include <Hpdf/HpdfBinaryTreeNode.h>
#include <Hpdf/HpdfFilterTree.h>

using namespace std;
using namespace triagens::basics;

// -----------------------------------------------------------------------------
// helper functions
// -----------------------------------------------------------------------------

namespace {
  bool atolSafe (const string& strValue, long int& longValue) {
    string value = StringUtils::trim(strValue);

      if (value == "0") {
        longValue = 0;
        return true;
      }

      longValue = ::atol(strValue.c_str());

      if (longValue != 0) {
        return true;
      }

      return false;
  }



  bool atodSafe (const string& strValue, double& doubleValue) {
    string value = StringUtils::trim(strValue);

    if (value == "0" || value == "0.0") {
      doubleValue = 0.0;
      return true;
    }

    doubleValue = ::atof(strValue.c_str());

    // check that the number is not out range
    if (doubleValue != 0.0) {
      if (doubleValue == HUGE_VAL) {
        return false;
      }

      return true;
    }

    return false;
  }
}

namespace triagens {
  namespace hpdf {

    // -----------------------------------------------------------------------------
    // FilterParserGrammar
    // -----------------------------------------------------------------------------

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief FilterParserGrammar
    ///
    /// The grammer rules used to parse a string with logical conditions
    ////////////////////////////////////////////////////////////////////////////////

    class FilterParserGrammar : public grammar<FilterParserGrammar> {
      public:

        // used to identify the logical operator tokens (matches LogicalType)
        static const int andKeywordID          = 10;
        static const int equalKeywordID        = 20;
        static const int greaterKeywordID      = 30;
        static const int greaterEqualKeywordID = 40;
        static const int lessKeywordID         = 50;
        static const int lessEqualKeywordID    = 60;
        static const int notEqualKeywordID     = 70;
        static const int notKeywordID          = 80;
        static const int orKeywordID           = 90;
        static const int logicalOperatorID     = 100;

        // used to identify the constant or literal tokens
        static const int integerConstantID     = 1010;  // all integers are long integers
        static const int doubleConstantID      = 1020;  // all floating point numbers are double
        static const int stringConstantID      = 1040;
        static const int dateConstantID        = 1050;  // all dates are represented as long integers
        static const int dateTimeConstantID    = 1060;  // all date time fields are represented as double
        static const int booleanConstantID     = 1060;
        static const int constantID            = 1100;

        static const int identifierID          = 2010;

        static const int andConditionID        = 3010;
        static const int orConditionID         = 3020;
        static const int conditionID           = 3100;

        static const int expressionID          = 4000;

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief parser definition
        ////////////////////////////////////////////////////////////////////////////////

        template <typename ScannerT>
        class definition {
          public:

            ////////////////////////////////////////////////////////////////////////////////
            /// @brief constructs a new definition
            ////////////////////////////////////////////////////////////////////////////////

            definition (FilterParserGrammar const&) { // self

              // -----------------------------------------------------------------------------
              // TODO : add the grammer rules for data/time
              // -----------------------------------------------------------------------------

              // -----------------------------------------------------------------------------
              // define the grammer rules
              // -----------------------------------------------------------------------------

              // rules for the logical operator rules
              andKeyword          = as_lower_d["and"];
              equalKeyword        = str_p("==") | str_p("=");
              greaterKeyword      = str_p(">");
              greaterEqualKeyword = str_p(">=");
              lessKeyword         = str_p("<");
              lessEqualKeyword    = str_p("<=");
              notEqualKeyword     = str_p("!=") | str_p("<>");
              notKeyword          = as_lower_d["not"];
              orKeyword           = as_lower_d["or"];

              logicalOperator     = equalKeyword
                                  | notEqualKeyword
                                  | greaterEqualKeyword
                                  | greaterKeyword
                                  | lessEqualKeyword
                                  | lessKeyword;

              doubleConstant      = real_p;
              integerConstant     = int_p;
              booleanConstant     = as_lower_d["true"] | as_lower_d["false"] |  as_lower_d["yes"] |  as_lower_d["no"];
              stringConstant      = lexeme_d[ inner_node_d[ ch_p('"')  >> leaf_node_d[ *(anychar_p - ch_p('"')) ]  >> ch_p('"') ] ]
                                  | lexeme_d[ inner_node_d[ ch_p('\'') >> leaf_node_d[ *(anychar_p - ch_p('\'')) ] >> ch_p('\'') ] ];

              constant            = integerConstant
                                  | doubleConstant
                                  | booleanConstant
                                  | stringConstant;

              identifier          = lexeme_d[ token_node_d[ ( range_p('A','Z') | range_p('a','z') )
                                    >> *( range_p('A','Z') | range_p('a','z') | range_p('0','9') | ch_p('_') ) ] ]
                                  - (andKeyword | notKeyword | orKeyword );

              condition           = constant   >> root_node_d[ logicalOperator ]      >> constant
                                  | constant   >> root_node_d[ logicalOperator ]      >> identifier
                                  | identifier >> root_node_d[ logicalOperator ]      >> constant
                                  | identifier >> root_node_d[ logicalOperator ]      >> identifier
                                  | inner_node_d[ch_p('(')  >> expression             >> ch_p(')')]
                                  | root_node_d[ notKeyword ]                         >> condition;

              andCondition        = condition % root_node_d[ andKeyword ];

              orCondition         = andCondition % root_node_d[orKeyword ];

              expression          = orCondition;
            }

            rule<ScannerT, parser_context<>, parser_tag<andKeywordID> >          andKeyword;
            rule<ScannerT, parser_context<>, parser_tag<equalKeywordID> >        equalKeyword;
            rule<ScannerT, parser_context<>, parser_tag<greaterKeywordID> >      greaterKeyword;
            rule<ScannerT, parser_context<>, parser_tag<greaterEqualKeywordID> > greaterEqualKeyword;
            rule<ScannerT, parser_context<>, parser_tag<lessKeywordID> >         lessKeyword;
            rule<ScannerT, parser_context<>, parser_tag<lessEqualKeywordID> >    lessEqualKeyword;
            rule<ScannerT, parser_context<>, parser_tag<notEqualKeywordID> >     notEqualKeyword;
            rule<ScannerT, parser_context<>, parser_tag<notKeywordID> >          notKeyword;
            rule<ScannerT, parser_context<>, parser_tag<orKeywordID> >           orKeyword;
            rule<ScannerT, parser_context<>, parser_tag<logicalOperatorID> >     logicalOperator;

            rule<ScannerT, parser_context<>, parser_tag<booleanConstantID> >  booleanConstant;
            rule<ScannerT, parser_context<>, parser_tag<integerConstantID> >  integerConstant;
            rule<ScannerT, parser_context<>, parser_tag<doubleConstantID> >   doubleConstant;
            rule<ScannerT, parser_context<>, parser_tag<stringConstantID> >   stringConstant;
            rule<ScannerT, parser_context<>, parser_tag<dateConstantID> >     dateConstant;
            rule<ScannerT, parser_context<>, parser_tag<dateTimeConstantID> > dateTimeConstant;
            rule<ScannerT, parser_context<>, parser_tag<constantID> >         constant;

            rule<ScannerT, parser_context<>, parser_tag<identifierID> >   identifier;

            rule<ScannerT, parser_context<>, parser_tag<andConditionID> > andCondition;
            rule<ScannerT, parser_context<>, parser_tag<orConditionID> >  orCondition;
            rule<ScannerT, parser_context<>, parser_tag<conditionID> >    condition;
            rule<ScannerT, parser_context<>, parser_tag<expressionID> >   expression;

            rule<ScannerT, parser_context<>, parser_tag<expressionID> > const&
            start() const { return expression; }
        };
    };

    // -----------------------------------------------------------------------------
    // FilterParser constructors and destructors
    // -----------------------------------------------------------------------------

    FilterParser::FilterParser (const string& str, HpdfFilterTree* fTree, AttributeDescriptor* desc)
      : stringToParse(str), filterTree(fTree), attributeDescription(desc)  {
      filterParserGrammar = new FilterParserGrammar;
    }



    FilterParser::~FilterParser () {
      delete filterParserGrammar;
    }

    // -----------------------------------------------------------------------------
    // FilterParser public methods
    // -----------------------------------------------------------------------------

    void FilterParser::parse () {
      filterTree->validFilterTree = true;
      treeInfo = ast_parse(stringToParse.c_str(), *filterParserGrammar >> !end_p, space_p);
      filterTree->validFilterTree = (treeInfo.full && filterTree->validFilterTree);
    }



    void FilterParser::evaluateTree () {

      // if the string entered by the user is invalid, just return - every row is automatically false
      if (! filterTree->validFilterTree) {
        LOGGER_DEBUG << "TREE INVALID";
        return;
      }

      int maxTreeWidth  = 0;
      int maxTreeHeight = 0;

      // create the root node object with the defaults
      HpdfBinaryTreeNode* rootNode = new HpdfBinaryTreeNode(0);

      filterTree->rootNode = rootNode;
      createBinaryTree(treeInfo.trees.begin(), filterTree->rootNode, 0, maxTreeHeight, maxTreeWidth);

      // oh dear something happened when we were trying to evaluate each node
      if (! filterTree->validFilterTree) {
        LOGGER_DEBUG << "TREE INVALID";
        return;
      }

      // fill in the evaluation ARRAY
      fillInEvaluationNodeArray(maxTreeHeight, maxTreeWidth);

      filterTree->maxNodeArrayHeight = maxTreeHeight;
      filterTree->maxNodeArrayWidth = maxTreeWidth;
    }

    // -----------------------------------------------------------------------------
    // FilterParser private methods
    // -----------------------------------------------------------------------------

    void FilterParser::fillInEvaluationNodeArray (int const maxTreeHeight, int const maxTreeWidth) {

      // create the two-dimensional array of pointers
      // make the height of the array slightly bigger - useful for traversing the array

      HpdfBinaryTreeNode*** evalNodeArray = new HpdfBinaryTreeNode**[maxTreeHeight + 1];

      // assign our array to our HpdfFilterTree Object
      filterTree->evaluationNodeArray = evalNodeArray;

      // size the array according to the parse tree (width + 1) and height
      // this extra column will be usual as we traverse the matrix

      for (int i = 0;  i <= maxTreeHeight;  ++i) {
        HpdfBinaryTreeNode** evalNodeArrayX = new HpdfBinaryTreeNode*[maxTreeWidth + 1];
        evalNodeArray[i] = evalNodeArrayX;
      }

      // make the two-dimensional array full of zeros -- will be useful as we traverse the matrix.
      for (int i = 0; i <= maxTreeHeight; ++i) {
        for (int j = 0; j <= maxTreeWidth; ++j) {
          evalNodeArray[i][j] = 0;
        }
      }

      // fill in the array according to the node tree previously created from the parse tree
      fillArray(evalNodeArray, filterTree->rootNode, -1, maxTreeWidth);

      // re-zero the last column again and the last row - just in case we have re-used
      // this space for temporary storage

      for (int i = 0; i <= maxTreeHeight; ++i) {
        evalNodeArray[i][maxTreeWidth] = 0;
      }

      for (int j = 0; j <= maxTreeWidth; ++j) {
        evalNodeArray[maxTreeHeight][j] = 0;
      }
    }



    void FilterParser::fillArray (HpdfBinaryTreeNode*** evalNodeArray,
                                  HpdfBinaryTreeNode* treeNode,
                                  int treeHeight,
                                  int const maxTreeWidth) {

      HpdfBinaryTreeNode* tNode;

      // increase the height of the tree by 1
      ++treeHeight;

      // find the first free column position available for this row
      // we can use this last extra column for storage of the free column
      // since there should be plenty of room to store an integer

      void* lastCol = evalNodeArray[treeHeight][maxTreeWidth];
      evalNodeArray[treeHeight][(intptr_t)lastCol] = treeNode;

      lastCol = (void*)((intptr_t)lastCol + 1);
      evalNodeArray[treeHeight][maxTreeWidth] = (HpdfBinaryTreeNode*)lastCol;

      tNode = treeNode->leftChildNode;

      // not a leaf
      if (tNode != 0) {
        fillArray(evalNodeArray, tNode, treeHeight, maxTreeWidth);
      }

      tNode = treeNode->rightChildNode;

      // not a leaf
      if (tNode != 0) {
        fillArray(evalNodeArray, tNode, treeHeight, maxTreeWidth);
      }

      // we must be at a leaf within the tree - leaf can be identifier or constant
    }



    LogicalType FilterParser::attributeIdentifier (tree_match<char const*>::tree_iterator const& itNode) {
      switch (itNode->value.id().to_long()) {
        case FilterParserGrammar::identifierID:
          return identifierID;

        default:
          return unknownID;
      }
    }



    LogicalType FilterParser::constantIdentifier (tree_match<char const*>::tree_iterator const& itNode) {
      switch (itNode->value.id().to_long()) {
        case FilterParserGrammar::integerConstantID:
          return integerConstantID;

        case FilterParserGrammar::doubleConstantID:
          return doubleConstantID;

        case FilterParserGrammar::stringConstantID:
          return stringConstantID;

        default:
          return unknownID;
      }
    }



    LogicalType FilterParser::logicalOperator (tree_match<char const*>::tree_iterator const& itNode) {
      switch (itNode->value.id().to_long()) {
        case FilterParserGrammar::andKeywordID:
          return andKeywordID;

        case FilterParserGrammar::equalKeywordID:
          return equalKeywordID;

        case FilterParserGrammar::greaterKeywordID:
          return greaterKeywordID;

        case FilterParserGrammar::greaterEqualKeywordID:
          return greaterEqualKeywordID;

        case FilterParserGrammar::lessKeywordID:
          return lessKeywordID;

        case FilterParserGrammar::lessEqualKeywordID:
          return lessEqualKeywordID;

        case FilterParserGrammar::notEqualKeywordID:
          return notEqualKeywordID;

        case FilterParserGrammar::notKeywordID:
          return notKeywordID;

        case FilterParserGrammar::orKeywordID:
          return orKeywordID;

        default:
          return unknownID;
      }
    }



    void FilterParser::printOutTree (tree_match<char const*>::tree_iterator const& i, const string& indent) {
      cout << indent << string(i->value.begin(), i->value.end()) << endl;

      for (tree_match<char const*>::tree_iterator j = i->children.begin();  j != i->children.end();  j++) {
        printOutTree(j, indent + "..");
      }
    }



    void FilterParser::createBinaryTree (tree_match<char const*>::tree_iterator const& i,
                                         HpdfBinaryTreeNode* treeNode,
                                         int treeHeight,
                                         int& maxTreeHeight,
                                         int& maxTreeWidth) {
      LogicalType result;

      // no point in going on if the filterTree is already invalid
      if (! filterTree->validFilterTree) {
        return;
      }

      // increase the height of the tree by 1 and if it exceeds the maxTreeHeight save it
      ++treeHeight;
      maxTreeHeight = max(treeHeight, maxTreeHeight);

      // determine if the symbol at this node is a logical operator
      result = logicalOperator(i);

      if (result != 0) {
        treeNode->currentNode.logicalOperator = result;

        // this node must have exactly two descentants EXCEPT when the logical operator is NOT
        for (tree_match<char const*>::tree_iterator j = i->children.begin();  j != i->children.end();  j++) {
          HpdfBinaryTreeNode* tNode = new HpdfBinaryTreeNode(0);
          tNode->parentNode = treeNode;

          if (j == i->children.begin()) {
            treeNode->leftChildNode = tNode;
          }
          else {
            treeNode->rightChildNode = tNode;
          }

          createBinaryTree(j, tNode, treeHeight, maxTreeHeight, maxTreeWidth);
        }

        return;
      }

      //////////////////////////////////////////////////////////////////////////////
      /// TODO: remember to adjust everything below later for arithmetic
      ///       operators +, * , - , /
      /////////////////////////////////////////////////////////////////////////////

      // We must be at a leaf within the tree - leaf can be identifier or constant
      // increase the width of the tree by 1 since we are at a leaf
      ++maxTreeWidth;
      treeNode->leftChildNode = 0;
      treeNode->rightChildNode = 0;

      // determine if the leaf is an attribute of the table
      result = attributeIdentifier(i);

      if (result != 0) {

        // since it is an attribute of the table, fill in node describing the attribute
        // find which attribute it is
        bool foundAttribName = false;

        for (AttributeDescriptor* attribPtr = attributeDescription;   attribPtr->name != 0;   ++attribPtr) {
          string parseName  = string(i->value.begin(), i->value.end());
          string attribName = string(attribPtr->name);

          if (parseName != attribName) {
            continue;
          }

          foundAttribName = true;
          treeNode->currentNode.dataType   = TypeAttribute(attribPtr->type);
          treeNode->currentNode.dataOffset = attribPtr->offset;
          treeNode->currentNode.dataSize   = TypeAttributeSize((TypeAttribute) attribPtr->type);

          break;
        }

        ///////////////////////////////////////////////////////////////////////////
        /// TODO : Somehow tell the user that the string entered for parsing does
        ///        not match any attributes in the table.
        //////////////////////////////////////////////////////////////////////////

        // tree is invalid -- user did not enter a valid field name the name of
        // of the invalid attribute is given below in the variable parseName.

        if (! foundAttribName) {
          string parseName  = string(i->value.begin(), i->value.end());
          filterTree->validFilterTree = false;
        }

        return;
      }

      // finally deal with the constants entered by the user
      result = constantIdentifier(i);

      if (result != 0) {

        // data offset = 0, since these are constant values entered by user
        treeNode->currentNode.dataOffset = 0;
        string parseStr  = string(i->value.begin(), i->value.end());

        // There are some special constants which the user can enter, one of them
        // is a boolean constant which must be hanbdled differently
        if (result == FilterParserGrammar::booleanConstantID) {
          treeNode->currentNode.dataType = AT_BOOLEAN;
          bool* tempData = new bool(false);

          if ((parseStr == "yes") || (parseStr == "true")) {
            (*tempData) = true;
          }

          treeNode->currentNode.dataSize  = sizeof(bool);
          treeNode->currentNode.dataConstants = (char*)tempData;
        }

        // is the constant entered a long integer constant?
        else if (result == FilterParserGrammar::integerConstantID) {
          treeNode->currentNode.dataType = AT_INTEGER;
          long int* tempData = new long int;

          if (! atolSafe(parseStr, *tempData) ) {
            delete tempData;
            LOGGER_DEBUG << "TREE INVALID";
            filterTree->validFilterTree = false;
            return;
          }

          treeNode->currentNode.dataSize  = sizeof(long int);
          treeNode->currentNode.dataConstants = (char*)tempData;
        }

        // is the constant entered a floating point constant?
        else if (result == FilterParserGrammar::doubleConstantID) {
          treeNode->currentNode.dataType = AT_DOUBLE;
          double* tempData = new double;

          if (! atodSafe(parseStr,*tempData) ) {
            delete tempData;
            LOGGER_DEBUG << "TREE INVALID";
            filterTree->validFilterTree = false;
            return;
          }

          treeNode->currentNode.dataSize  = sizeof(double);
          treeNode->currentNode.dataConstants = (char*)tempData;
        }

        // is the constant entered a string constant?
        else if (result == FilterParserGrammar::stringConstantID) {
          treeNode->currentNode.dataType = AT_STRING;

          treeNode->currentNode.dataSize  = parseStr.length();
          treeNode->currentNode.dataConstants = StringUtils::duplicate(parseStr);
        }

        // oh dear
        else {
          LOGGER_DEBUG << "TREE INVALID";
          filterTree->validFilterTree = false;
        }
      }
    }
  }
}


