////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Dr. Oreste Costa-Panaia
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "DynamicCondition.h"

#include <Basics/Exceptions.h>
#include <Hpdf/HpdfBinaryTreeNode.h>
#include <Hpdf/HpdfFilterTree.h>

using namespace triagens::hpdf;

// -----------------------------------------------------------------------------
// helper functions
// -----------------------------------------------------------------------------

namespace {
  struct EQUAL {
    template<typename L, typename R>
    bool operator() (L const& left, R const& right) {
      return left == right;
    }
  };

  struct NOT_EQUAL {
    template<typename L, typename R>
    bool operator() (L const& left, R const& right) {
      return left != right;
    }
  };

  struct GREATER {
    template<typename L, typename R>
    bool operator() (L const& left, R const& right) {
      return left > right;
    }
  };

  struct GREATER_EQUAL {
    template<typename L, typename R>
    bool operator() (L const& left, R const& right) {
      return left >= right;
    }
  };

  struct LESS {
    template<typename L, typename R>
    bool operator() (L const& left, R const& right) {
      return left < right;
    }
  };

  struct LESS_EQUAL {
    template<typename L, typename R>
    bool operator() (L const& left, R const& right) {
      return left <= right;
    }
  };



  template<typename C>
  bool compareObjects (TypeAttribute leftType,
                       char const* leftData,
                       TypeAttribute rightType,
                       char const* rightData,
                       C compare) {
    switch (leftType) {

      // -----------------------------------------------------------------------------
      // left DOUBLE data type == something
      // -----------------------------------------------------------------------------

      case AT_DOUBLE:
        switch (rightType) {
          case AT_DOUBLE:
            return compare(*reinterpret_cast<double const*>(leftData), *reinterpret_cast<double const*>(rightData));

          case AT_FLOAT:
            return compare(*reinterpret_cast<double const*>(leftData), *reinterpret_cast<float const*>(rightData));

          case AT_INTEGER:
            return compare(*reinterpret_cast<double const*>(leftData), *reinterpret_cast<int const*>(rightData));

          default:
            THROW_INTERNAL_ERROR("invalid double comparision");
        }

        break;

      // -----------------------------------------------------------------------------
      // left FLOAT data type == something
      // -----------------------------------------------------------------------------

      case AT_FLOAT:
        switch (rightType) {
          case AT_DOUBLE:
            return compare(*reinterpret_cast<float const*>(leftData), *reinterpret_cast<double const*>(rightData));

          case AT_FLOAT:
            return compare(*reinterpret_cast<float const*>(leftData), *reinterpret_cast<float const*>(rightData));

          case AT_INTEGER:
            return compare(*reinterpret_cast<float const*>(leftData), *reinterpret_cast<int const*>(rightData));

          default:
            THROW_INTERNAL_ERROR("invalid float comparision");
        }

        break;

      // -----------------------------------------------------------------------------
      // left INTEGER data type == something
      // -----------------------------------------------------------------------------

      case AT_INTEGER:
        switch (rightType) {
          case AT_DOUBLE:
            return compare(*reinterpret_cast<int const*>(leftData), *reinterpret_cast<double const*>(rightData));

          case AT_FLOAT:
            return compare(*reinterpret_cast<int const*>(leftData), *reinterpret_cast<float const*>(rightData));

          case AT_INTEGER:
            return compare(*reinterpret_cast<int const*>(leftData), *reinterpret_cast<int const*>(rightData));

          default:
            THROW_INTERNAL_ERROR("invalid integer comparision");
        }

        break;

      // -----------------------------------------------------------------------------
      // comparing a left string data type
      // -----------------------------------------------------------------------------

      case AT_STRING:
        switch (rightType) {
          case AT_STRING: {
            int r = 0;

            if (leftData == 0) {
              if (rightData == 0) {
                r = 0;
              }
              else {
                r = 1;
              }
            }
            else if (rightData == 0) {
              r = -1;
            }
            else {
              r = strcmp(leftData, rightData);
            }

            return compare(r, 0);
          }

          default:
            THROW_INTERNAL_ERROR("invalid string comparision");
        }

        break;

      // -----------------------------------------------------------------------------
      // invalid data type for now (need to add the types date, datetime etc)
      // -----------------------------------------------------------------------------

      default:
        THROW_INTERNAL_ERROR("Invalid data type used for comparison");
    }
  }
}

namespace triagens {
  namespace hpdf {

    // -----------------------------------------------------------------------------
    // static public methods
    // -----------------------------------------------------------------------------

    bool DynamicCondition::dynamicFilter (HpdfFilterTree& filterTree, void const* dataRow) {

      // if the syntax entered by the user was invalid - simply return false
      if (! filterTree.validFilterTree) {
        return false;
      }

      // collapse the matrix upto but not including the first row
      for (int i = filterTree.maxNodeArrayHeight - 1; i > 0; --i) {
        for (int j = 0; j < filterTree.maxNodeArrayWidth; ++j) {
          HpdfBinaryTreeNode* tNode = filterTree.evaluationNodeArray[i][j];

          // -----------------------------------------------------------------------------
          // then we are at the end of the row
          // -----------------------------------------------------------------------------

          if (tNode == 0) {
            break;
          }

          HpdfBinaryTreeNodeObject& leftObject  = tNode->currentNode;
          HpdfBinaryTreeNodeObject& rightObject  = tNode->parentNode->rightChildNode->currentNode;
          HpdfBinaryTreeNodeObject& parentObject = tNode->parentNode->currentNode;

          // -----------------------------------------------------------------------------
          // this is a logical operator
          // -----------------------------------------------------------------------------

          if (leftObject.dataType == 0) {
            switch (parentObject.logicalOperator) {
              case andKeywordID:  {
                parentObject.logicalValue = (leftObject.logicalValue && tNode->parentNode->rightChildNode->currentNode.logicalValue);
                ++j; // since we do not need to visit the right operator again.
                break;
              }

              case notKeywordID: {
                parentObject.logicalValue = !(leftObject.logicalValue);
                break;
              }

              case orKeywordID:  {
                parentObject.logicalValue = (leftObject.logicalValue || tNode->parentNode->rightChildNode->currentNode.logicalValue);
                ++j; // since we do not need to visit the right operator again.
                break;
              }

              default: {
                THROW_INTERNAL_ERROR("Undefined logical operator");
              }
            }
          }

          // -----------------------------------------------------------------------------
          // this is an attribute or constant value
          // -----------------------------------------------------------------------------

          else {

            // left data
            char const* leftData  = leftObject.dataConstants;

            // the left attribute represents actual data from a field within the row
            if (leftData == 0) {
              if (leftObject.dataType == AT_STRING) {
                leftData = * (char const**) (reinterpret_cast<char const*>(dataRow) + leftObject.dataOffset);
              }
              else {
                leftData = reinterpret_cast<char const*>(dataRow) + leftObject.dataOffset;
              }
            }

            // an attribute of the type boolean is special, there is only one thing to do -- store its truth value.
            if (leftObject.dataType == AT_BOOLEAN) {
              leftObject.logicalValue = * reinterpret_cast<bool const*>(leftData);
              continue;
            }

            // increment counter so that we skip the right child node of the parent
            // but only when the data type requires a comparison of two nodes
            ++j;

            // right data
            char const* rightData = 0;

            rightData = rightObject.dataConstants;

            // the right attribute represents actual data from a field within the row
            if (rightData == 0) {
              if (rightObject.dataType == AT_STRING) {
                rightData = * (char const**) (reinterpret_cast<char const*>(dataRow) + rightObject.dataOffset);
              }
              else {
                rightData = reinterpret_cast<char const*>(dataRow) + rightObject.dataOffset;
              }
            }

            bool c = false;

            switch (parentObject.logicalOperator) {
              case equalKeywordID:
                c = compareObjects(leftObject.dataType, leftData, rightObject.dataType, rightData, EQUAL());
                break;

              case greaterKeywordID:
                c = compareObjects(leftObject.dataType, leftData, rightObject.dataType, rightData, GREATER());
                break;

              case greaterEqualKeywordID:
                c = compareObjects(leftObject.dataType, leftData, rightObject.dataType, rightData, GREATER_EQUAL());
                break;

              case lessKeywordID:
                c = compareObjects(leftObject.dataType, leftData, rightObject.dataType, rightData, LESS());
                break;

              case lessEqualKeywordID:
                c = compareObjects(leftObject.dataType, leftData, rightObject.dataType, rightData, LESS_EQUAL());
                break;

              case notEqualKeywordID:
                c = compareObjects(leftObject.dataType, leftData, rightObject.dataType, rightData, NOT_EQUAL());
                break;

              default:
                THROW_INTERNAL_ERROR("should not happen");
            }

            parentObject.logicalValue = c;
          }
        }
      }

      return (filterTree.rootNode->currentNode.logicalValue);
    }
  }
}
