////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2008-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "Common.h"

#include <Basics/Exceptions.h>
#include <Basics/StringUtils.h>

using namespace std;
using namespace triagens::basics;

namespace triagens {
  namespace hpdf {

    // -----------------------------------------------------------------------------
    // TypeAttribute
    // -----------------------------------------------------------------------------

    string NameTypeAttribute (TypeAttribute type) {
      switch (type) {
        case AT_UNKNOWN:                return "AT_UNKNOWN";
        case AT_BIG_INTEGER:            return "AT_BIG_INTEGER";
        case AT_BLOB:                   return "AT_BLOB";
        case AT_BOOLEAN:                return "AT_BOOLEAN";
        case AT_DATE:                   return "AT_DATE";
        case AT_DATETIME:               return "AT_DATETIME";
        case AT_DOUBLE:                 return "AT_DOUBLE";
        case AT_FLOAT:                  return "AT_FLOAT";
        case AT_INTEGER:                return "AT_INTEGER";
        case AT_SMALL_INTEGER:          return "AT_SMALL_INTEGER";
        case AT_STRING:                 return "AT_STRING";
        case AT_STRING_POINTER:         return "AT_STRING_POINTER";
        case AT_UNSIGNED_BIG_INTEGER:   return "AT_UNSIGNED_BIG_INTEGER";
        case AT_UNSIGNED_INTEGER:       return "AT_UNSIGNED_INTEGER";
        case AT_UNSIGNED_SMALL_INTEGER: return "AT_UNSIGNED_SMALL_INTEGER";
        case AT_POINTER:                return "AT_POINTER";
      }

      THROW_INTERNAL_ERROR("unknown TypeAttribute");
    }



    int32_t TypeAttributeSize (TypeAttribute type) {
      switch (type) {
        case AT_UNKNOWN:                return 0;
        case AT_BIG_INTEGER:            return sizeof(int64_t);
        case AT_BLOB:                   return sizeof(blob_t);
        case AT_BOOLEAN:                return sizeof(bool);
        case AT_DATE:                   return sizeof(date_t);
        case AT_DATETIME:               return sizeof(datetime_t);
        case AT_DOUBLE:                 return sizeof(double);
        case AT_FLOAT:                  return sizeof(float);
        case AT_INTEGER:                return sizeof(int32_t);
        case AT_SMALL_INTEGER:          return sizeof(int16_t);
        case AT_STRING:                 return -1; // can only be determined at run-time
        case AT_STRING_POINTER:         return sizeof(char*);
        case AT_UNSIGNED_BIG_INTEGER:   return sizeof(uint64_t);
        case AT_UNSIGNED_INTEGER:       return sizeof(uint32_t);
        case AT_UNSIGNED_SMALL_INTEGER: return sizeof(uint16_t);
        case AT_POINTER:                return sizeof(void*);

      };

      return 0;
    }



    uint32_t TypeAttributeValue (TypeAttribute type) {
      return static_cast<uint32_t>(type);
    }

    // -----------------------------------------------------------------------------
    // LanguageAttribute
    // -----------------------------------------------------------------------------

    string NameLangAttribute (LangAttribute type) {
      switch (type) {
        case LT_UNKNOWN: return "unknown";
        case LT_NONE:    return "none";
        case LT_DEDE:    return "dede";
        case LT_ENAU:    return "enau";
        case LT_ITIT:    return "itit";
      }

      THROW_INTERNAL_ERROR("unkown language");
    }



    LangAttribute LangAttributeByName (string const& type) {
      if (type == "unknown") { return LT_UNKNOWN; }
      if (type == "none"   ) { return LT_NONE;    }
      if (type == "dede"   ) { return LT_DEDE;    }
      if (type == "enau"   ) { return LT_ENAU;    }
      if (type == "itit"   ) { return LT_ITIT;    }

      THROW_INTERNAL_ERROR("unkown language");
    }

    // -----------------------------------------------------------------------------
    // EncodingAttribute
    // -----------------------------------------------------------------------------

    string NameEncodeAttribute (EncodeAttribute type) {
      switch (type) {
        case ET_UNKNOWN: return "unknown";
        case ET_NONE:    return "none";
        case ET_LATIN1:  return "latin1";
        case ET_UTF8:    return "utf8";
      }

      THROW_INTERNAL_ERROR("unkown encoding");
    }



    EncodeAttribute EncodeAttributeByName (string const& type) {
      if (type == "unknown")   { return ET_UNKNOWN; }
      if (type == "none"   )   { return ET_NONE;    }
      if (type == "ascii"  )   { return ET_ASCII;   }
      if (type == "latin1" )   { return ET_LATIN1;  }
      if (type == "iso88591" ) { return ET_LATIN1;  }
      if (type == "utf8" )     { return ET_UTF8;    }

      THROW_INTERNAL_ERROR("unknown encoding");
    }

    // -----------------------------------------------------------------------------
    // IndexAttribute
    // -----------------------------------------------------------------------------

    string NameIndexType (IndexType type) {
      switch (type) {
        case IX_UNKNOWN:      return "IX_UNKNOWN";
        case IX_NO_INDEX:     return "IX_NO_INDEX";
        case IX_HASH_INDEX:   return "IX_HASH_INDEX";
        case IX_PQUE_INDEX:   return "IX_PQUE_INDEX";
        case IX_PREFIX_INDEX: return "IX_PREFIX_INDEX";
      }

      THROW_INTERNAL_ERROR("unknown index type");
    }

    // -----------------------------------------------------------------------------
    // Priority Queue
    // -----------------------------------------------------------------------------

    string NamePriorityQueueType (PriorityQueueType type) {
      switch (type) {
        case PQ_UNKNOWN:          return "PQ_UNKNOWN";
        case PQ_NONE:             return "PQ_NONE";
        case PQ_STANDARD_LIBRARY: return "PQ_STANDARD_LIBRARY";
      }

      THROW_INTERNAL_ERROR("unknown prirotiy queue type");
    }

    // -----------------------------------------------------------------------------
    // row deletion attribute
    // -----------------------------------------------------------------------------

    string NameRowDeletionType (RowDeletionType type) {
      switch (type) {
      case RD_UNKNOWN:  return "RD_UNKNOWN";
        case RD_STANDARD: return "RD_STANDARD";
        case RD_UNSTABLE: return "RD_UNSTABLE";
      }

      THROW_INTERNAL_ERROR("unknown Row Deletion type");
    }



    RowDeletionType RowDeletionByName (string const& type) {
      const string& lowerType = StringUtils::trim(StringUtils::tolower(type),"\"\'\t\n\r");
      if ( (lowerType == "unknown")  || (lowerType == "rd_unknown") ) { return RD_UNKNOWN; }
      if ( (lowerType == "standard") || (lowerType == "rd_standard") ) { return RD_STANDARD; }
      if ( (lowerType == "unstable") || (lowerType == "rd_unstable") ) { return RD_UNSTABLE; }

      THROW_INTERNAL_ERROR("unkown Row Deletion");
    }
  }
}
