////////////////////////////////////////////////////////////////////////////////
/// @brief program options description
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_BASICS_PROGRAM_OPTIONS_DESCRIPTION_H
#define TRIAGENS_BASICS_PROGRAM_OPTIONS_DESCRIPTION_H 1

#include <Basics/Common.h>

namespace triagens {
  namespace basics {

    ////////////////////////////////////////////////////////////////////////////////
    /// @brief program options description
    ////////////////////////////////////////////////////////////////////////////////

    class ProgramOptionsDescription {
      friend struct ProgramOptionsData;
      friend class ProgramOptions;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription ();

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructor
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription (string const& name);

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a new section
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (ProgramOptionsDescription&);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a new hidden section
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (ProgramOptionsDescription&, bool hidden);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a new flag
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a string argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, string* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a string vector argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, vector<string>* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds an int32_t argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, int32_t* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds an int32_t vector argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, vector<int32_t>* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds an int64_t argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, int64_t* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds an int64_t vector argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, vector<int64_t>* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds an uint32_t argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, uint32_t* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds an uint32_t vector argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, vector<uint32_t>* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds an uint64_t argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, uint64_t* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds an uint64_t vector argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, vector<uint64_t>* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a double argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, double* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a double vector argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, vector<double>* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a boolean argument
        ////////////////////////////////////////////////////////////////////////////////

        ProgramOptionsDescription& operator() (string const& name, bool* value, string const& text);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds positional arguments
        ////////////////////////////////////////////////////////////////////////////////

        void arguments (vector<string>*);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns the usage message
        ////////////////////////////////////////////////////////////////////////////////

        string usage ();

      private:
        enum option_type_e {
          OPTION_TYPE_FLAG,
          OPTION_TYPE_STRING,
          OPTION_TYPE_VECTOR_STRING,
          OPTION_TYPE_INT32,
          OPTION_TYPE_VECTOR_INT64,
          OPTION_TYPE_INT64,
          OPTION_TYPE_VECTOR_INT32,
          OPTION_TYPE_UINT32,
          OPTION_TYPE_VECTOR_UINT32,
          OPTION_TYPE_UINT64,
          OPTION_TYPE_VECTOR_UINT64,
          OPTION_TYPE_DOUBLE,
          OPTION_TYPE_VECTOR_DOUBLE,
          OPTION_TYPE_BOOL
        };

      private:
        string check (string const& name);
        string check (string const& name, void* value);

      private:
        string name;

        vector<ProgramOptionsDescription> subDescriptions;
        vector<ProgramOptionsDescription> hiddenSubDescriptions;

        vector<string> optionNames;

        map< string, option_type_e > optionTypes;
        map< string, string > long2short;
        map< string, string > helpTexts;

        map< string, string* > stringOptions;
        map< string, vector<string>* > vectorStringOptions;

        map< string, int32_t* > int32Options;
        map< string, vector<int32_t>* > vectorInt32Options;

        map< string, int64_t* > int64Options;
        map< string, vector<int64_t>* > vectorInt64Options;

        map< string, uint32_t* > uint32Options;
        map< string, vector<uint32_t>* > vectorUint32Options;

        map< string, uint64_t* > uint64Options;
        map< string, vector<uint64_t>* > vectorUint64Options;

        map< string, double* > doubleOptions;
        map< string, vector<double>* > vectorDoubleOptions;

        map< string, bool* > boolOptions;

        vector<string>* positionals;
    };
  }
}

#endif
