////////////////////////////////////////////////////////////////////////////////
/// @brief program options description
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2010, triagens GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "ProgramOptionsDescription.h"

#include <Basics/Exceptions.h>
#include <Basics/StringUtils.h>

namespace triagens {
  namespace basics {

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    ProgramOptionsDescription::ProgramOptionsDescription ()
      : positionals(0) {
    }



    ProgramOptionsDescription::ProgramOptionsDescription (string const& name)
      : name(name), positionals(0) {
    }

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    ProgramOptionsDescription& ProgramOptionsDescription::operator() (ProgramOptionsDescription& sub) {
      subDescriptions.push_back(sub);
      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (ProgramOptionsDescription& sub, bool hidden) {
      if (hidden) {
        hiddenSubDescriptions.push_back(sub);
      }
      else {
        subDescriptions.push_back(sub);
      }

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, string const& text) {
      string name = check(full);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_FLAG;
      helpTexts[name] = text;

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, string* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_STRING;
      stringOptions[name] = value;

      if (value->empty()) {
        helpTexts[name] = text;
      }
      else {
        helpTexts[name] = text + " (default: \"" + *value + "\")";
      }

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, vector<string>* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_VECTOR_STRING;
      vectorStringOptions[name] = value;
      helpTexts[name] = text;

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, int32_t* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_INT32;
      int32Options[name] = value;
      helpTexts[name] = text + " (default: " + StringUtils::itoa(*value) + ")";

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, vector<int32_t>* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_VECTOR_INT32;
      vectorInt32Options[name] = value;
      helpTexts[name] = text;

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, int64_t* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_INT64;
      int64Options[name] = value;
      helpTexts[name] = text + " (default: " + StringUtils::itoa(*value) + ")";

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, vector<int64_t>* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_VECTOR_INT64;
      vectorInt64Options[name] = value;
      helpTexts[name] = text;

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, uint32_t* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_UINT32;
      uint32Options[name] = value;
      helpTexts[name] = text + " (default: " + StringUtils::itoa(*value) + ")";

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, vector<uint32_t>* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_VECTOR_UINT32;
      vectorUint32Options[name] = value;
      helpTexts[name] = text;

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, uint64_t* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_UINT64;
      uint64Options[name] = value;
      helpTexts[name] = text + " (default: " + StringUtils::itoa(*value) + ")";

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, vector<uint64_t>* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_VECTOR_UINT64;
      vectorUint64Options[name] = value;
      helpTexts[name] = text;

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, double* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_DOUBLE;
      doubleOptions[name] = value;
      helpTexts[name] = text + " (default: " + StringUtils::ftoa(*value) + ")";

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, vector<double>* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_VECTOR_DOUBLE;
      vectorDoubleOptions[name] = value;
      helpTexts[name] = text;

      return *this;
    }



    ProgramOptionsDescription& ProgramOptionsDescription::operator() (string const& full, bool* value, string const& text) {
      string name = check(full, value);

      optionNames.push_back(name);
      optionTypes[name] = OPTION_TYPE_BOOL;
      boolOptions[name] = value;
      helpTexts[name] = text + " (default: " + (*value ? "true" : "false") + ")";

      return *this;
    }



    void ProgramOptionsDescription::arguments (vector<string>* value) {
      if (value == 0) {
        THROW_INTERNAL_ERROR("value is 0");
      }

      if (positionals != 0) {
        THROW_INTERNAL_ERROR("positional arguments are already defined");
      }

      positionals = value;
    }



    string ProgramOptionsDescription::usage () {
      string desc;

      // produce a head line
      if (! name.empty() && ! (subDescriptions.empty() && optionNames.empty())) {
        desc = name + ":\n";
      }

      // collect the maximal width of the option names
      size_t oWidth = 0;

      map<string, string> names;

      for (vector<string>::iterator i = optionNames.begin();  i != optionNames.end();  ++i) {
        string const& option = *i;

        string name;

        switch (optionTypes[option]) {
          case OPTION_TYPE_FLAG:
            name = option;
            break;

          case OPTION_TYPE_STRING:
            name = option + " <string>";
            break;

          case OPTION_TYPE_VECTOR_STRING:
            name = option + " <string>";
            break;

          case OPTION_TYPE_INT32:
            name = option + " <int32>";
            break;

          case OPTION_TYPE_VECTOR_INT64:
            name = option + " <int64>";
            break;

          case OPTION_TYPE_INT64:
            name = option + " <int64>";
            break;

          case OPTION_TYPE_VECTOR_INT32:
            name = option + " <int32>";
            break;

          case OPTION_TYPE_UINT32:
            name = option + " <uint32>";
            break;

          case OPTION_TYPE_VECTOR_UINT32:
            name = option + " <uint32>";
            break;

          case OPTION_TYPE_UINT64:
            name = option + " <uint64>";
            break;

          case OPTION_TYPE_VECTOR_UINT64:
            name = option + " <uint64>";
            break;

          case OPTION_TYPE_DOUBLE:
            name = option + " <double>";
            break;

          case OPTION_TYPE_VECTOR_DOUBLE:
            name = option + " <double>";
            break;

          case OPTION_TYPE_BOOL:
            name = option + " <bool>";
            break;
        }

        names[option] = name;

        if (name.size() > oWidth) {
          oWidth = name.size();
        }
      }

      // construct the parameters
      size_t tWidth = 80;
      size_t sWidth = 8;
      size_t dWidth = (oWidth + sWidth < tWidth) ? (tWidth - oWidth - sWidth) : (tWidth / 2);

      for (vector<string>::iterator i = optionNames.begin();  i != optionNames.end();  ++i) {
        string const& option = *i;
        string const& name = names[option];
        string const& text = helpTexts[option];

        if (text.size() <= dWidth) {
          desc += "  --" + StringUtils::rFill(name, oWidth) + "    " + helpTexts[option] + "\n";
        }
        else {
          vector<string> wrap = StringUtils::wrap(text, dWidth, " ,");

          string sep = "  --" + StringUtils::rFill(name, oWidth) + "    ";

          for (vector<string>::iterator j = wrap.begin();  j != wrap.end();  ++j) {
            desc += sep + *j + "\n";
            sep = string(oWidth + sWidth, ' ');
          }
        }
      }

      // check for sub-descriptions
      string sep = optionNames.empty() ? "" : "\n";

      for (vector<ProgramOptionsDescription>::iterator i = subDescriptions.begin();  i != subDescriptions.end();  ++i) {
        desc += sep + i->usage();
        sep = "\n";
      }

      return desc;
    }

    // -----------------------------------------------------------------------------
    // private methods
    // -----------------------------------------------------------------------------

    string ProgramOptionsDescription::check (string const& name) {
      vector<string> s = StringUtils::split(name, ',');

      if (name.empty()) {
        THROW_INTERNAL_ERROR("option name is empty");
      }

      if (s.size() > 2) {
        THROW_INTERNAL_ERROR("option '" + name + "' should be <long-option>,<short-option> or <long-option>");
      }

      string longOption = s[0];

      if (optionTypes.find(longOption) != optionTypes.end()) {
        THROW_INTERNAL_ERROR("option '" + longOption + "' is already defined");
      }

      if (s.size() > 1) {
        long2short[longOption] = s[1];
      }

      return longOption;
    }



    string ProgramOptionsDescription::check (string const& name, void* value) {
      if (value == 0) {
        THROW_INTERNAL_ERROR("value is 0");
      }

      return check(name);
    }
  }
}


