////////////////////////////////////////////////////////////////////////////////
/// @brief program options
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "ProgramOptions.h"

#include <fstream>

#include <boost/program_options.hpp>

#include <Basics/Exceptions.h>

using namespace std;

namespace bpo = boost::program_options;

namespace {
  string const& lookup (map<string, string> const& m, string const& key) {
    static string empty = "";

    map<string, string>::const_iterator i = m.find(key);

    if (i == m.end()) {
      return empty;
    }
    else {
      return i->second;
    }
  }


  template<typename T>
  T* lookup (map<string, T*> const& m, string const& key) {
    typename map<string, T*>::const_iterator i = m.find(key);

    if (i == m.end()) {
      return 0;
    }
    else {
      return i->second;
    }
  }


  template<typename T>
  T find (map<string, T> const& m, string const& key) {
    typename map<string, T>::const_iterator i = m.find(key);

    if (i == m.end()) {
      THROW_INTERNAL_ERROR("cannot find option '" + key + "'");
    }

    return i->second;
  }
}

namespace triagens {
  namespace basics {

    // -----------------------------------------------------------------------------
    // implementation structure
    // -----------------------------------------------------------------------------

    struct ProgramOptionsData {
      bpo::options_description setupDescription (ProgramOptionsDescription const& description) {
        return setupSubDescription(description);
      }

      bpo::options_description setupSubDescription (ProgramOptionsDescription const& description) {
        bpo::options_description desc(description.name);

        for (vector<string>::const_iterator i = description.optionNames.begin();  i != description.optionNames.end();  ++i) {
          string const& name = *i;
          string const& help = lookup(description.helpTexts, name);
          string option = name;

          map<string, string>::const_iterator j = description.long2short.find(name);

          if (j != description.long2short.end()) {
            option += "," + j->second;
          }

          options.push_back(option);

          string const& boption = options[options.size() - 1];

          ProgramOptionsDescription::option_type_e type = find(description.optionTypes, name);

          switch (type) {
            case ProgramOptionsDescription::OPTION_TYPE_FLAG:
              desc.add_options()(boption.c_str(), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_STRING:
              desc.add_options()(boption.c_str(), bpo::value<string>(find(description.stringOptions, name)), help.c_str());
              stringValues[name] = find(description.stringOptions, name);
              break;

            case ProgramOptionsDescription::OPTION_TYPE_VECTOR_STRING:
              desc.add_options()(boption.c_str(), bpo::value< vector<string> >(find(description.vectorStringOptions, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_INT32:
              desc.add_options()(boption.c_str(), bpo::value<int32_t>(find(description.int32Options, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_VECTOR_INT32:
              desc.add_options()(boption.c_str(), bpo::value< vector<int32_t> >(find(description.vectorInt32Options, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_INT64:
              desc.add_options()(boption.c_str(), bpo::value<int64_t>(find(description.int64Options, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_VECTOR_INT64:
              desc.add_options()(boption.c_str(), bpo::value< vector<int64_t> >(find(description.vectorInt64Options, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_UINT32:
              desc.add_options()(boption.c_str(), bpo::value<uint32_t>(find(description.uint32Options, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_VECTOR_UINT32:
              desc.add_options()(boption.c_str(), bpo::value< vector<uint32_t> >(find(description.vectorUint32Options, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_UINT64:
              desc.add_options()(boption.c_str(), bpo::value<uint64_t>(find(description.uint64Options, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_VECTOR_UINT64:
              desc.add_options()(boption.c_str(), bpo::value< vector<uint64_t> >(find(description.vectorUint64Options, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_DOUBLE:
              desc.add_options()(boption.c_str(), bpo::value<double>(find(description.doubleOptions, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_VECTOR_DOUBLE:
              desc.add_options()(boption.c_str(), bpo::value< vector<double> >(find(description.vectorDoubleOptions, name)), help.c_str());
              break;

            case ProgramOptionsDescription::OPTION_TYPE_BOOL:
              desc.add_options()(boption.c_str(), bpo::value<bool>(find(description.boolOptions, name)), help.c_str());
              break;

            default:
              THROW_INTERNAL_ERROR("unknown option type");
          }
        }

        for (vector<ProgramOptionsDescription>::const_iterator i = description.subDescriptions.begin();
             i != description.subDescriptions.end();
             ++i) {
          desc.add(setupSubDescription(*i));
        }

        for (vector<ProgramOptionsDescription>::const_iterator i = description.hiddenSubDescriptions.begin();
             i != description.hiddenSubDescriptions.end();
             ++i) {
          desc.add(setupSubDescription(*i));
        }

        return desc;
      }

      map<string, string*> stringValues;

      boost::program_options::variables_map vm;
      vector<string> options;
    };

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    ProgramOptions::ProgramOptions () {
      data = new ProgramOptionsData;
    }



    ProgramOptions::~ProgramOptions () {
      delete data;
    }

    // -----------------------------------------------------------------------------
    // public methods
    // -----------------------------------------------------------------------------

    bool ProgramOptions::parse (ProgramOptionsDescription const& description, int argc, char** argv) {
      boost::program_options::options_description desc = data->setupDescription(description);

      if (description.positionals != 0) {
        bpo::positional_options_description pdesc;
        pdesc.add("arguments", -1);

        bpo::options_description hidden("Positional Arguments");
        desc.add(hidden);
      }

      try {
        bpo::store(boost::program_options::command_line_parser(argc, argv).options(desc).run(), data->vm);
        boost::program_options::notify(data->vm);
        return true;
      }
      catch (const boost::program_options::error& e) {
        errorMessage = e.what();
        return false;
      }
    }



    bool ProgramOptions::parse (ProgramOptionsDescription const& description, string const& filename) {
      boost::program_options::options_description desc = data->setupDescription(description);

      try {
        ifstream ifs(filename.c_str());

        bpo::store(parse_config_file(ifs, desc), data->vm);
        boost::program_options::notify(data->vm);
        return true;
      }
      catch (const boost::program_options::error& e) {
        errorMessage = e.what();
        return false;
      }
    }



    bool ProgramOptions::has (string const& key) {
      return data->vm.count(key.c_str()) > 0;
    }



    string ProgramOptions::stringValue (string const& key) {
      string* value = lookup(data->stringValues, key);

      return value == 0 ? "" : *value;
    }
  }
}
