////////////////////////////////////////////////////////////////////////////////
/// @brief logger
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2007-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_BASICS_LOGGER_H
#define TRIAGENS_BASICS_LOGGER_H 1

#include <Basics/Common.h>

#include <Basics/LoggerInfo.h>
#include <Basics/LoggerTiming.h>
#include <Basics/LoggerStream.h>
#include <Basics/Timing.h>

////////////////////////////////////////////////////////////////////////////////
/// @defgroup Logging Logging
////////////////////////////////////////////////////////////////////////////////

// -----------------------------------------------------------------------------
// logging enabled
// -----------------------------------------------------------------------------

#ifdef ENABLE_LOGGER

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs fatal errors
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_FATAL                                                              \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isFatal()) \
    triagens::basics::Logger::singleton                                           \
    << triagens::basics::LoggerData::LEVEL_FATAL                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                               \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs fatal errors
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_FATAL_I(a)                                                         \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isFatal()) \
    triagens::basics::Logger::singleton                                           \
    << (a)                                                                        \
    << triagens::basics::LoggerData::LEVEL_FATAL                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                               \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs errors
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_ERROR                                                              \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isError()) \
    triagens::basics::Logger::singleton                                           \
    << triagens::basics::LoggerData::LEVEL_ERROR                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                               \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs errors
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_ERROR_I(a)                                                         \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isError()) \
    triagens::basics::Logger::singleton                                           \
    << (a)                                                                        \
    << triagens::basics::LoggerData::LEVEL_ERROR                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                               \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs warnings
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_WARNING                                                              \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isWarning()) \
    triagens::basics::Logger::singleton                                             \
    << triagens::basics::LoggerData::LEVEL_WARNING                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                                 \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs warnings
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_WARNING_I(a)                                                         \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isWarning()) \
    triagens::basics::Logger::singleton                                             \
    << (a)                                                                          \
    << triagens::basics::LoggerData::LEVEL_WARNING                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                                 \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs info messages
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_INFO                                                              \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isInfo()) \
    triagens::basics::Logger::singleton                                          \
    << triagens::basics::LoggerData::LEVEL_INFO                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                              \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs info messages
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_INFO_I(a)                                                         \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isInfo()) \
    triagens::basics::Logger::singleton                                          \
    << (a)                                                                       \
    << triagens::basics::LoggerData::LEVEL_INFO                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                              \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs debug messages
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_DEBUG                                                              \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isDebug()) \
    triagens::basics::Logger::singleton                                           \
    << triagens::basics::LoggerData::LEVEL_DEBUG                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                               \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs debug messages
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_DEBUG_I(a)                                                         \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isDebug()) \
    triagens::basics::Logger::singleton                                           \
    << (a)                                                                        \
    << triagens::basics::LoggerData::LEVEL_DEBUG                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                               \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs trace messages
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_TRACE                                                              \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isTrace()) \
    triagens::basics::Logger::singleton                                           \
    << triagens::basics::LoggerData::LEVEL_TRACE                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                               \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs trace messages
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_TRACE_I(a)                                                         \
  if (triagens::basics::Logger::isHuman() && triagens::basics::Logger::isTrace()) \
    triagens::basics::Logger::singleton                                           \
    << (a)                                                                        \
    << triagens::basics::LoggerData::LEVEL_TRACE                                  \
    << triagens::basics::LoggerData::SEVERITY_HUMAN                               \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs non-human messsages
///
/// @param[in]
///    severity         severity level to log
////////////////////////////////////////////////////////////////////////////////

#define VOC_LOGGER(severity)                                                   \
  if (triagens::basics::Logger::is ## severity ())                             \
    triagens::basics::Logger::singleton                                        \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief logs non-human messsages
///
/// @param[in]
///    severity         severity level to log
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define VOC_LOGGER_I(severity, a)                                              \
  if (triagens::basics::Logger::is ## severity ())                             \
    triagens::basics::Logger::singleton                                        \
    << (a)                                                                     \
    << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief request-in start
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_REQUEST_IN_START                                                \
  VOC_LOGGER(Technical)                                                        \
    << triagens::basics::LoggerData::CATEGORY_REQUEST_IN_START

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief request-in start
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_REQUEST_IN_START_I(a)                                           \
  VOC_LOGGER_I(Technical, a)                                                   \
    << triagens::basics::LoggerData::CATEGORY_REQUEST_IN_START

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief request-in end
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_REQUEST_IN_END                                                  \
  VOC_LOGGER(Technical)                                                        \
    << triagens::basics::LoggerData::CATEGORY_REQUEST_IN_END

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief request-in end
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_REQUEST_IN_END_I(a)                                             \
  VOC_LOGGER_I(Technical, a)                                                   \
    << triagens::basics::LoggerData::CATEGORY_REQUEST_IN_END

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief request-out start
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_REQUEST_OUT_START                                               \
  VOC_LOGGER(Technical)                                                        \
    << triagens::basics::LoggerData::CATEGORY_REQUEST_OUT_START

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief request-out start
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_REQUEST_OUT_START_I(a)                                          \
  VOC_LOGGER_I(Technical, a)                                                   \
    << triagens::basics::LoggerData::CATEGORY_REQUEST_OUT_START

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief request-out end
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_REQUEST_OUT_END                                                 \
  VOC_LOGGER(Technical)                                                        \
    << triagens::basics::LoggerData::CATEGORY_REQUEST_OUT_END

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief request-out end
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_REQUEST_OUT_END_I(a)                                            \
  VOC_LOGGER_I(Technical, a)                                                   \
    << triagens::basics::LoggerData::CATEGORY_REQUEST_OUT_END

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief module in start
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_MODULE_IN_START                                                 \
  VOC_LOGGER(Technical)                                                        \
    << triagens::basics::LoggerData::CATEGORY_MODULE_IN_START

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief module in start
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_MODULE_IN_START_I(a)                                            \
  VOC_LOGGER_I(Technical, a)                                                   \
    << triagens::basics::LoggerData::CATEGORY_MODULE_IN_START

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief module in end
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_MODULE_IN_END                                                   \
  VOC_LOGGER(Technical)                                                        \
    << triagens::basics::LoggerData::CATEGORY_MODULE_IN_END

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief module in end
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_MODULE_IN_END_I(a)                                              \
  VOC_LOGGER_I(Technical, a)                                                   \
    << triagens::basics::LoggerData::CATEGORY_MODULE_IN_END

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief function in start
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_FUNCTION_IN_START                                               \
  VOC_LOGGER(Technical)                                                        \
    << triagens::basics::LoggerData::CATEGORY_FUNCTION_IN_START

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief function in start
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_FUNCTION_IN_START_I(a)                                          \
  VOC_LOGGER_I(Technical, a)                                                   \
    << triagens::basics::LoggerData::CATEGORY_FUNCTION_IN_START

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief function in end
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_FUNCTION_IN_END                                                 \
  VOC_LOGGER(Technical)                                                        \
    << triagens::basics::LoggerData::CATEGORY_FUNCTION_IN_END

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief function in end
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_FUNCTION_IN_END_I(a)                                            \
  VOC_LOGGER_I(Technical, a)                                                   \
    << triagens::basics::LoggerData::CATEGORY_FUNCTION_IN_END

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief function step
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_STEP                                                            \
  VOC_LOGGER(Development)                                                      \
  << triagens::basics::LoggerData::CATEGORY_STEP                               \
  << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief function step
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_STEP_I(a)                                                       \
  VOC_LOGGER_I(Development, a)                                                 \
  << triagens::basics::LoggerData::CATEGORY_STEP                               \
  << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief function loop
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_LOOP                                                            \
  VOC_LOGGER(Development)                                                      \
  << triagens::basics::LoggerData::CATEGORY_LOOP                               \
  << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief function loop
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_LOOP_I(a)                                                       \
  VOC_LOGGER_I(Development, a)                                                 \
  << triagens::basics::LoggerData::CATEGORY_LOOP                               \
  << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief heartbeat
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_HEARTBEAT                                                       \
  VOC_LOGGER(Technical)                                                        \
  << triagens::basics::LoggerData::CATEGORY_HEARTBEAT                          \
  << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief heartbeat
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_HEARTBEAT_I(a)                                                  \
  VOC_LOGGER_I(Technical, a)                                                   \
  << triagens::basics::LoggerData::CATEGORY_HEARTBEAT                          \
  << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief heartpulse
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_HEARTPULSE                                                      \
  VOC_LOGGER(Development)                                                      \
  << triagens::basics::LoggerData::CATEGORY_HEARTPULSE                         \
  << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

////////////////////////////////////////////////////////////////////////////////
/// @ingroup Logging
/// @brief heartpulse
///
/// @param[in]
///    a                info
////////////////////////////////////////////////////////////////////////////////

#define LOGGER_HEARTPULSE_I(a)                                                 \
  VOC_LOGGER_I(Development, a)                                                 \
  << triagens::basics::LoggerData::CATEGORY_HEARTPULSE                         \
  << triagens::basics::LoggerData::Position(__FUNCTION__, __FILE__, __LINE__)

#else

// -----------------------------------------------------------------------------
// logging disabled
// -----------------------------------------------------------------------------

#define LOGGER_FATAL                    if (false) triagens::basics::Logger::singleton
#define LOGGER_FATAL_I(a)               if (false) triagens::basics::Logger::singleton
#define LOGGER_ERROR                    if (false) triagens::basics::Logger::singleton
#define LOGGER_ERROR_I(a)               if (false) triagens::basics::Logger::singleton
#define LOGGER_WARNING                  if (false) triagens::basics::Logger::singleton
#define LOGGER_WARNING_I(a)             if (false) triagens::basics::Logger::singleton
#define LOGGER_INFO                     if (false) triagens::basics::Logger::singleton
#define LOGGER_INFO_I(a)                if (false) triagens::basics::Logger::singleton
#define LOGGER_DEBUG                    if (false) triagens::basics::Logger::singleton
#define LOGGER_DEBUG_I(a)               if (false) triagens::basics::Logger::singleton
#define LOGGER_TRACE                    if (false) triagens::basics::Logger::singleton
#define LOGGER_TRACE_I(a)               if (false) triagens::basics::Logger::singleton

#define VOC_LOGGER(a)                   if (false) triagens::basics::Logger::singleton
#define VOC_LOGGER_I(a,b)               if (false) triagens::basics::Logger::singleton

#define LOGGER_REQUEST_IN_START         if (false) triagens::basics::Logger::singleton
#define LOGGER_REQUEST_IN_START_I(a)    if (false) triagens::basics::Logger::singleton
#define LOGGER_REQUEST_IN_END           if (false) triagens::basics::Logger::singleton
#define LOGGER_REQUEST_IN_END_I(a)      if (false) triagens::basics::Logger::singleton
#define LOGGER_REQUEST_OUT_START        if (false) triagens::basics::Logger::singleton
#define LOGGER_REQUEST_OUT_START_I(a)   if (false) triagens::basics::Logger::singleton
#define LOGGER_REQUEST_OUT_END          if (false) triagens::basics::Logger::singleton
#define LOGGER_REQUEST_OUT_END_I(a)     if (false) triagens::basics::Logger::singleton
#define LOGGER_MODULE_IN_START          if (false) triagens::basics::Logger::singleton
#define LOGGER_MODULE_IN_START_I(a)     if (false) triagens::basics::Logger::singleton
#define LOGGER_MODULE_IN_END            if (false) triagens::basics::Logger::singleton
#define LOGGER_MODULE_IN_END_I(a)       if (false) triagens::basics::Logger::singleton
#define LOGGER_FUNCTION_IN_START        if (false) triagens::basics::Logger::singleton
#define LOGGER_FUNCTION_IN_START_I(a)   if (false) triagens::basics::Logger::singleton
#define LOGGER_FUNCTION_IN_END          if (false) triagens::basics::Logger::singleton
#define LOGGER_FUNCTION_IN_END_I(a)     if (false) triagens::basics::Logger::singleton
#define LOGGER_STEP                     if (false) triagens::basics::Logger::singleton
#define LOGGER_STEP_I(a)                if (false) triagens::basics::Logger::singleton
#define LOGGER_LOOP                     if (false) triagens::basics::Logger::singleton
#define LOGGER_LOOP_I(a)                if (false) triagens::basics::Logger::singleton
#define LOGGER_HEARTBEAT                if (false) triagens::basics::Logger::singleton
#define LOGGER_HEARTBEAT_I(a)           if (false) triagens::basics::Logger::singleton
#define LOGGER_HEARTPULSE               if (false) triagens::basics::Logger::singleton
#define LOGGER_HEARTPULSE_I(a)          if (false) triagens::basics::Logger::singleton

#endif

namespace triagens {
  namespace basics {

    ////////////////////////////////////////////////////////////////////////////////
    /// @ingroup Logging
    /// @brief logger
    ///
    /// This class provides various static members which can be used as logging
    /// streams. Output to the logging stream is appended by using the operator <<,
    /// as soon as a line is completed endl should be used to flush the stream.
    /// Each line of output is prefixed by some informational data.
    ////////////////////////////////////////////////////////////////////////////////

    class Logger : boost::noncopyable {
     friend class LoggerStream;
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief buffer type
        ////////////////////////////////////////////////////////////////////////////////

        struct buffer_t {
          uint64_t lid;
          LoggerData::level_e level;
          time_t timestamp;
          string text;
        };

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief global logger
        ////////////////////////////////////////////////////////////////////////////////

        static Logger singleton;

      public:


        ////////////////////////////////////////////////////////////////////////////////
        /// @brief last log entries
        ///
        /// Returns a log entries of a given level (or less) and starting from position
        /// (including this position).
        ////////////////////////////////////////////////////////////////////////////////

        static vector<buffer_t> buffer (LoggerData::level_e, uint64_t pos);

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief initialise (called from global init)
        ////////////////////////////////////////////////////////////////////////////////

        static void initialise ();

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for exceptions
        ///
        /// @return
        ///    true if exception logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isException () {
          return exception;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for technical information
        ///
        /// @return
        ///    true if technical logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isTechnical () {
          return technical;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for functional information
        ///
        /// @return
        ///    true if functional logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isFunctional () {
          return functional;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for development information
        ///
        /// @return
        ///    true if development logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isDevelopment () {
          return development;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for human readable logging
        ///
        /// @return
        ///    true if human logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isHuman () {
          return human;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for fatal logging
        ///
        /// @return
        ///    true if fatal logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isFatal () {
          return fatal;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for error logging
        ///
        /// @return
        ///    true if error logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isError () {
          return error;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for warning logging
        ///
        /// @return
        ///    true if warning logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isWarning () {
          return warning;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for info logging
        ///
        /// @return
        ///    true if info logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isInfo () {
          return info;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for debug logging
        ///
        /// @return
        ///    true if debug logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isDebug () {
          return debug;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief checks for trace logging
        ///
        /// @return
        ///    true if trace logging is active
        ////////////////////////////////////////////////////////////////////////////////

        static bool isTrace () {
          return trace;
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns true if logging to standard out or error
        ///
        /// @return
        ///    true if logging to stdout or stderr
        ////////////////////////////////////////////////////////////////////////////////

        static bool isStandardStream ();

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief changes the application name
        ///
        /// @param[in]
        ///    name     the application name
        ////////////////////////////////////////////////////////////////////////////////

        static void setApplicationName (string const& name);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief changes the host name
        ///
        /// @param[in]
        ///    name     the host name
        ////////////////////////////////////////////////////////////////////////////////

        static void setHostName (string const& name);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief changes the facility
        ///
        /// @param[in]
        ///    name     the facility name
        ////////////////////////////////////////////////////////////////////////////////

        static void setFacility (string const& name);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief changes the log level
        ///
        /// @param[in]
        ///    level    the log level for human logging
        ////////////////////////////////////////////////////////////////////////////////

        static void setLogLevel (string const& level);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns the log level
        ////////////////////////////////////////////////////////////////////////////////

        static string logLevel ();

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief changes the severity
        ///
        /// @param[in]
        ///    severities       the log severity or severities separated by comma
        ////////////////////////////////////////////////////////////////////////////////

        static void setLogSeverity (string const& severities);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief changes the log format
        ///
        /// The machine readable log format is defined as a string with "%x" being
        /// the various log pieces.
        ///
        /// - "%A"  application name
        /// - "%C"  category
        /// - "%E"  extras
        /// - "%F"  facility
        /// - "%H"  host name
        /// - "%K"  task
        /// - "%M"  message identifier
        /// - "%P"  peg
        /// - "%S"  severity
        /// - "%T"  timestamp
        /// - "%U"  measure unit
        /// - "%V"  measure value
        /// - "%Z"  timestamp in GMT (zulu)
        /// - "%f"  source code module
        /// - "%l"  source code line
        /// - "%m"  source code method
        /// - "%p"  pid
        /// - "%s"  pthread identifier
        /// - "%t"  tid
        /// - "%u"  user identifier
        /// - "%x"  text
        ///
        /// @param[in]
        ///    format   the log format
        ////////////////////////////////////////////////////////////////////////////////

        static void setLogFormat (string const& format);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief assigns a prefix for log messages
        ///
        /// @param[in]
        ///    prefix           the new prefix
        ////////////////////////////////////////////////////////////////////////////////

        static void definePrefix (string const& prefix);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief use thread identifiers
        ///
        /// @param[in]
        ///    show             true means show thread identifier
        ////////////////////////////////////////////////////////////////////////////////

        static void useThreadIdentifier (bool show);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief changes the log file
        ///
        /// @param[in]
        ///    file            The log-file. "-" means stdout, "+" means stderr,
        ///                     "" means no logging to file at all
        ////////////////////////////////////////////////////////////////////////////////

        static void assignLogFile (string const& file);

#ifdef ENABLE_SYSLOG

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief set syslog
        ///
        /// @param[in]
        ///    name             syslog name
        ///
        /// @param[in]
        ///    facility         syslog facility
        ////////////////////////////////////////////////////////////////////////////////

        static void assignSyslog (string const& name, string const& facility);

#endif

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructs a new logger
        ////////////////////////////////////////////////////////////////////////////////

        Logger ();

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief destructs a logger
        ////////////////////////////////////////////////////////////////////////////////

        ~Logger ();

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief delegates work to the underlying logger stream
        ///
        /// @param[in]
        ///    value            the value to output
        ///
        /// @return
        ///    the current logger stream
        ////////////////////////////////////////////////////////////////////////////////

        template<typename T>
        LoggerStream operator<< (T const& value) {
          LoggerStream stream;

          stream << value;

          return stream;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief delegates work to the underlying logger stream
        ///
        /// @param[in]
        ///    value            the info block to use
        ///
        /// @return
        ///    the current logger stream
        ////////////////////////////////////////////////////////////////////////////////

        LoggerStream operator<< (LoggerInfo& value) {
          LoggerStream stream(value.info);
          return stream;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief delegates work to the underlying logger stream
        ///
        /// @param[in]
        ///    value            the info block to use
        ///
        /// @return
        ///    the current logger stream
        ////////////////////////////////////////////////////////////////////////////////

        LoggerStream operator<< (LoggerTiming& value) {
          value.measure();
          LoggerStream stream(value.info);
          return stream;
        }

      private:
        static void output (string const& msg, LoggerData::Info const& info);

      private:
        static bool exception;
        static bool technical;
        static bool functional;
        static bool development;
        static bool human;

        static bool fatal;
        static bool error;
        static bool warning;
        static bool info;
        static bool debug;
        static bool trace;
    };
  }
}

#endif
